/* $ZEL: sis1100_read_dma_linux.c,v 1.8 2006/02/14 19:51:25 wuestner Exp $ */

/*
 * Copyright (c) 2001-2004
 * 	Matthias Drochner, Peter Wuestner.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "sis1100_sc.h"

int
_sis1100_read_dma(
    struct sis1100_softc* sc,
    struct sis1100_fdata* fd,
    u_int32_t addr,           /* VME or SDRAM address */
    int32_t am,               /* address modifier, not used if <0 */
    int size,                 /* datasize must be 4 or 2 */
    int space,                /* remote space (1,2: VME; 6: SDRAM) */
    int fifo_mode,
    size_t count,             /* words to be transferred */
                              /* count==0 is illegal */
    size_t* count_read,       /* words transferred */
    u_int8_t* data,           /* destination (user virtual address) */
    int* prot_error           /* protocol error (or 0) */
    /*int* eot*/              /* end of transport indicator */
    )
{
    int res, i, sgcount, aborted=0;
    u_int32_t head, dmamode;
    sigset_t oldset;
#ifdef USE_SGL
    int nr_pages;
#else
    struct kiobuf* iobuf=sc->iobuf;
    int err, offs;
#endif
    /*
    printk(KERN_ERR "r addr=%d size=%d fifo=%d count=%lld data=%p\n",
                addr, size, fifo_mode, (unsigned long long)count, data);
    */
    count*=size;
    if (count>DMA_MAX)
        count=DMA_MAX;
    /*
    printk(KERN_ERR "DMA_MAX=%ld count=%lld SGL_SIZE=%d\n",
            DMA_MAX, (unsigned long long)count, SGL_SIZE);
    */
    {
        u_int32_t val;
        val=plxreadreg(sc, DMACSR0_DMACSR1);
        if (!(val&0x10)) {
            printk(KERN_CRIT "sis1100_read_dma: DMACSR0=%04x\n", val);
            printk(KERN_CRIT "sis1100_read_dma: old DMA still active.\n");
            return EIO;
        }
    }

#ifdef USE_SGL
    nr_pages=sgl_map_user_pages(sc->sglist, SGL_SIZE, data, count, READ);
    /*printk(KERN_ERR "R nr_pages=%d\n", nr_pages);*/
    if (nr_pages<0) {
        printk(KERN_INFO "sis1100[%d] sgl_map_user_pages failed\n", sc->unit);
        return -nr_pages;
    }
    sgcount=pci_map_sg(sc->pcidev, sc->sglist, nr_pages,
        PCI_DMA_FROMDEVICE/*|0xf000*/);
    /*printk(KERN_ERR "R sgcount=%d\n", sgcount);*/
    if (!sgcount) {
        printk(KERN_ERR "sis1100[%d] read_dma: pci_map_sg failed\n", sc->unit);
        sgl_unmap_user_pages(sc->sglist, nr_pages, 0);
        return EIO;
    }
#else
    err=map_user_kiobuf(READ, iobuf, (unsigned long)data, count);
    if (err) {
        printk(KERN_INFO "sis1100[%d] map_user_kiobuf failed\n", sc->unit);
        return err;
    }
    offs=iobuf->offset;
    for (i=0; i<iobuf->nr_pages-1; i++) {
        sc->sglist[i].address=0;
        sc->sglist[i].page=iobuf->maplist[i];
        sc->sglist[i].offset=offs;
        sc->sglist[i].length=PAGE_SIZE-offs;
        sc->sglist[i].dma_length=0;
        offs=0;
    }
    sc->sglist[i].address=0;
    sc->sglist[i].page=iobuf->maplist[i];
    sc->sglist[i].offset=offs;
    sc->sglist[i].length=iobuf->length-i*PAGE_SIZE+iobuf->offset-offs;
    sc->sglist[i].dma_length=0;
    sgcount=pci_map_sg(sc->pcidev, sc->sglist, iobuf->nr_pages,
            PCI_DMA_FROMDEVICE);
    if (!sgcount) {
        printk(KERN_ERR "sis1100[%d] read_dma: pci_map_sg failed\n",
            sc->unit);
        unmap_kiobuf(iobuf);
        return EIO;
    }
#endif

    dmamode=0x43|(1<<7)|(1<<8)|(1<<10)|(1<<11)|(1<<12)|(1<<14)|(1<<17);
    if (sgcount>1) { /* use scatter/gather mode */
        struct plx9054_dmadesc* desclist=
            (struct plx9054_dmadesc*)sc->descbuf.cpu_addr;
        struct scatterlist* sgl=sc->sglist;
        dma_addr_t next_handle=sc->descbuf.dma_handle;
        dmamode|=1<<9;
        for (i=sgcount; i; i--) {
            next_handle+=sizeof(struct plx9054_dmadesc);
            desclist->pcistart=cpu_to_le32(sg_dma_address(sgl));
            desclist->localstart=cpu_to_le32(0);
            desclist->size=cpu_to_le32(sg_dma_len(sgl));
            desclist->next=cpu_to_le32(next_handle|9);
            desclist++; sgl++;
        }
        desclist[-1].next|=2;
        plxwritereg(sc, DMADPR0, sc->descbuf.dma_handle|1);
    } else { /* only one page --> use block mode */
        plxwritereg(sc, DMAPADR0, cpu_to_le32(sg_dma_address(sc->sglist)));
        plxwritereg(sc, DMALADR0, 0);
        plxwritereg(sc, DMASIZ0, cpu_to_le32(sg_dma_len(sc->sglist)));
        plxwritereg(sc, DMADPR0, cpu_to_le32(8));
    }
/*XXX cpu_to_le32 oder implicit in plxwritereg???*/
/* prepare PLX */
    plxwritereg(sc, DMACSR0_DMACSR1, 1<<3); /* clear irq */
    plxwritereg(sc, DMAMODE0, dmamode);

/* prepare add on logic */
    /* BT, EOT, start with t_adl */
    head=0x0080A002|((0x00f00000<<size)&0x0f000000)|(space&0x3f)<<16;
    if (am>=0) {
        head|=0x800;
        sis1100writereg(sc, t_am, am);
    }
    if (fifo_mode) head|=0x4000;
    sis1100writereg(sc, t_hdr, head);
    wmb();
    if (sc->remote_hw==sis1100_hw_vme)
        sis1100writereg(sc, t_dal, count*size/4);
    else
        sis1100writereg(sc, t_dal, count);
    sis1100writereg(sc, d0_bc, 0);
    sis1100writereg(sc, d0_bc_buf, 0);

    sis1100writereg(sc, p_balance, 0);
    sis1100writereg(sc, sr, 0x200); /* clear EOT */

/* block signals */
    spin_lock_irq(&current->SIGMASK_LOCK);
    oldset = current->blocked;
    sigfillset(&current->blocked);
    sigdelset(&current->blocked, SIGKILL);
    /* dangerous, should be removed later */
    /*if (!sigismember(&oldset, SIGINT)) sigdelset(&current->blocked, SIGINT);*/
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    recalc_sigpending(current);
#else
    recalc_sigpending();
#endif
    spin_unlock_irq(&current->SIGMASK_LOCK);

/* enable dma */
    plxwritereg(sc, DMACSR0_DMACSR1, 3);

/* enable irq */
    sc->got_irqs=0;
    sis1100_enable_irq(sc, plxirq_dma0, irq_synch_chg|irq_prot_l_err);

/* start transfer */
    mb();
    sis1100writereg(sc, t_adl, addr);
    wmb();

/* wait for dma */
    res=wait_event_interruptible(
	sc->local_wait,
	(sc->got_irqs & (got_dma0|got_sync|got_l_err))
	);
    sis1100_disable_irq(sc, plxirq_dma0, irq_prot_l_err);
/*
    if (sc->got_irqs&got_dma0) pINFO(sc, "got_dma0");
    if (sc->got_irqs&got_sync) pINFO(sc, "got_sync");
    if (sc->got_irqs&got_l_err) pINFO(sc, "got_l_err");
*/
    if (sc->got_irqs&(got_dma0|got_l_err)) { /* transfer complete or error */
        *count_read=sis1100readreg(sc, d0_bc)/size;
    } else /*(res||(sc->got_irqs&(got_sync)))*/ { /* fatal */
        aborted=0x300;
        if (res) {
            printk(KERN_WARNING "sis1100[%d] read_dma: interrupted\n", sc->unit);
            aborted|=1;
        }
        if (sc->got_irqs&got_sync) {
            printk(KERN_WARNING "sis1100[%d] read_dma: synchronisation lost\n",
                    sc->unit);
            aborted|=2;
        }
        if (aborted==0x300) {
            printk(KERN_CRIT "sis1100[%d] read_dma: got_irqs=0x%x\n",
                    sc->unit, sc->got_irqs);
        }
    }
    if (!(sc->got_irqs&got_dma0)) {
        u_int32_t val;
        val=plxreadreg(sc, DMACSR0_DMACSR1);
        if (!(val&0x10)) { /* DMA not stopped yet; abort it */
            sis1100writereg(sc, sr, sr_abort_dma);
            do {
                val=plxreadreg(sc, DMACSR0_DMACSR1);
            } while (!(val&0x10));
        }
    }

    plxwritereg(sc, DMACSR0_DMACSR1, 0);

    spin_lock_irq(&current->SIGMASK_LOCK);
    current->blocked = oldset;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    recalc_sigpending(current);
#else
    recalc_sigpending();
#endif
    spin_unlock_irq(&current->SIGMASK_LOCK);

#ifdef USE_SGL
    pci_unmap_sg(sc->pcidev, sc->sglist, nr_pages, PCI_DMA_FROMDEVICE);
    sgl_unmap_user_pages(sc->sglist, nr_pages, 1);
#else
    pci_unmap_sg(sc->pcidev, sc->sglist, iobuf->nr_pages, PCI_DMA_FROMDEVICE);
    unmap_kiobuf(iobuf);
#endif

    *prot_error=sis1100readreg(sc, prot_error);
    /* *eot=!!(sis1100readreg(sc, sr)&0x200); */

    /*if (aborted) sis1100_dump_glink_status(sc, "after abort", 1);*/
    if (aborted) sis1100_flush_fifo(sc, "after abort", 0);
    if (aborted) *prot_error=aborted;
    if ((*prot_error!=0) && ((*prot_error&0x200)!=0x200)) {
        /*pERROR(sc, "read_dma_linux: prot_error=0x%x", *prot_error);*/
        res=EIO;
    }

    /*if (res!=0)
        pERROR(sc, "sis1100_read_dma: res=%d", res);*/
    return res;
}
