
/********************************************************************\

  ProcessIO.cc

  Main class processing user input
 
  Sebastian Commichau, Sabrina Stark-Schneebeli, Oliver Grimm

\********************************************************************/

#include "ProcessIO.h"

static const char* state_str[]    = {"active", "stopped", "n.a."};

// Branch table for command evaluation
static const struct CL_Struct { const char *Name;    
								void (ProcessIO::*CommandPointer)();
								unsigned int MinNumParameter;
								const char *Parameters;
								const char *Help;
  } CommandList[] = 
   {{"board", &ProcessIO::cmd_board, 1, "<i>|<i j>|<all>" ,"Address board i, boards i-j or all boards or list boards"},
	{"hv", &ProcessIO::cmd_hv, 2, "<id>|<ch>|<all> <v>", "Change bias of pixel or (all) chan. of active boards"},
	{"status", &ProcessIO::cmd_status, 0, "[dac]", "Show status information (DAC values if requested)"},
	{"config", &ProcessIO::cmd_config, 0, "", "Print configuration"},
	{"load", &ProcessIO::cmd_load, 1, "<file>", "Load and set bias settings from file"},
	{"save", &ProcessIO::cmd_save, 1, "<file>", "Save current bias settings to file"},
	{"exit", &ProcessIO::cmd_exit, 0, "", "Exit program"},
	{"rate", &ProcessIO::cmd_rate, 1, "<rate>", "Set status refresh rate in Hz"},
	{"timeout", &ProcessIO::cmd_timeout, 1, "<time>", "Set timeout to return from read in seconds"},
	{"reset", &ProcessIO::cmd_reset, 0, "", "Reset active bias board"},
	{"start", &ProcessIO::cmd_start, 0, "", "Start bias status monitor"},
	{"stop", &ProcessIO::cmd_stop, 0, "", "Stop bias status monitor"},
	{"uptime", &ProcessIO::cmd_uptime, 0, "", "Get program uptime"},
	{"help", &ProcessIO::cmd_help, 0, "", "Print help"}};


using namespace std;


// Constructor
ProcessIO::ProcessIO(): EvidenceServer(SERVER_NAME) {

  // Get program start time
  time (&StartTime);

  // Initialize status variables
  state       = active;
  ConsoleText = NULL;

  NumHVBoards = 0;
  FirstBoard  = 0;
  LastBoard   = -1;

  // Initialize mutex for thread synchronisation
  if (pthread_mutex_init(&Mutex, NULL) != 0) {
    Message(FATAL, "pthread_mutex_init() failed");
  }

  // DIM console service used in PrintMessage()
  ConsoleOut = new DimService(SERVER_NAME"/ConsoleOut", (char *) "");

  // Get configuration data
  vector<string> Boards = Tokenize(GetConfig("Boards"), " \t");
  fTimeOut = atof(GetConfig("TimeOut").c_str());
  fStatusRefreshRate = atof(GetConfig("StatusRefreshRate").c_str());
  DACMin = atoi(GetConfig("DACMin").c_str());
  DACMax = atoi(GetConfig("DACMax").c_str());
  fHVCalibOffset = atof(GetConfig("HVCalibOffset").c_str());
  fHVCalibSlope = atof(GetConfig("HVCalibSlope").c_str());
  fHVMaxDiff = atoi(GetConfig("HVMaxDiff").c_str());

  if (fStatusRefreshRate < MIN_RATE || fStatusRefreshRate > MAX_RATE)  fStatusRefreshRate = 1;

  // Open HV devices
  fHVBoard = new HVBoard* [Boards.size()]; 
  for (unsigned int i=0; i<Boards.size(); i++) {
    fHVBoard[NumHVBoards] = new HVBoard(NumHVBoards, Boards[i], this);
    if(fHVBoard[NumHVBoards]->fDescriptor >= 0) {
       PrintMessage("Synchronized and reset board %d (%s)\n", NumHVBoards, fHVBoard[NumHVBoards]->BoardName);
       NumHVBoards++;
    }
    else {
      Message(WARN, "Failed to synchronize board %d (%s)", NumHVBoards, fHVBoard[NumHVBoards]->BoardName);
      delete fHVBoard[NumHVBoards];
    }
  }
  LastBoard = NumHVBoards-1;

  // Create instances
  calib  = new HVCalib(this);
  pm 	 = new PixelMap(GetConfig("PixMapTable"));
  
  // Install DIM command (after all initialized)
  Command = new DimCommand((char *) SERVER_NAME"/Command", (char *) "C", this);
}


// Destructor
ProcessIO::~ProcessIO() {
  
  delete Command;

  for (int i=0; i<NumHVBoards; i++) delete fHVBoard[i];
  delete[] fHVBoard;
    
  delete pm;
  delete calib;
  delete ConsoleOut;	
  free(ConsoleText);
  
  // Destroy mutex
  if (pthread_mutex_destroy(&Mutex) != 0) Message(ERROR, "pthread_mutex_destroy() failed");
}


// Process user input
void ProcessIO::CommandControl(char *Command) {

  // Ignore empty commands
  if (strlen(Command)==0) return;

  // Shell command
  if(Command[0]=='.') {
    system(&(Command[1]));
    return;
  }

  // Parse command into tokens
  Parameter.clear();
  char *Start; 
  while(true) {
    while (isspace(*Command)) Command++; // Ignore initial white spaces
    if(*Command=='\0') break;
    if (*Command == '\"') {
	  Start = ++Command;
      while(*Command!='\"' && *Command!='\0') Command++;
    }
    else {
	  Start = Command;
      while(!isspace(*Command) && *Command!='\0') Command++;
    }
    if(*Command != '\0') *Command++ = '\0';
	Parameter.push_back(Start);
  }

  // Search for command in command list
  for(unsigned int CmdNumber=0; CmdNumber<sizeof(CommandList)/sizeof(CL_Struct); CmdNumber++) {
    if (Match(Parameter[0], CommandList[CmdNumber].Name)) {
      if(Parameter.size()-1 < CommandList[CmdNumber].MinNumParameter) {
		PrintMessage("Usage: %s %s\n", CommandList[CmdNumber].Name, CommandList[CmdNumber].Parameters);
		return;
	  }

	  // Jump to command function
	  (this->*CommandList[CmdNumber].CommandPointer)();
	  return;  
    }
  }
  PrintMessage("Unknown command '%s'\n", Parameter[0].c_str());
}


// Adress board
void ProcessIO::cmd_board() {
    
  // Print list of boards
  if (Parameter.size() == 1) {
    for (int i=0; i<NumHVBoards; i++) {
      PrintMessage("Board %d: %s\n", fHVBoard[i]->GetBoardNumber(), fHVBoard[i]->BoardName);
    }
    return;
  }

  //Select board(s)
  if (Match(Parameter[1].c_str(), "all")) {
    FirstBoard = 0;
    LastBoard = NumHVBoards-1;
  } 
  else if (Parameter.size()==2 && atoi(Parameter[1].c_str())>=0 && atoi(Parameter[1].c_str())<NumHVBoards) {
    FirstBoard = atoi(Parameter[1].c_str());
    LastBoard = FirstBoard;
  }
  else if (Parameter.size()==3 && atoi(Parameter[1].c_str())>=0 && atoi(Parameter[1].c_str())<NumHVBoards && 
         atoi(Parameter[2].c_str())>0 && atoi(Parameter[2].c_str())<NumHVBoards) {
    FirstBoard = atoi(Parameter[1].c_str());
    LastBoard = atoi(Parameter[2].c_str());
  }
  else PrintMessage("Cannot address board(s), out of range.\n");
} 


// Print configuration
void ProcessIO::cmd_config() {

  PrintMessage(	" %d USB devices:\n", NumHVBoards);  

  for (int i=0; i<NumHVBoards; i++) PrintMessage(" Board %d: %s\n", i, fHVBoard[i]->BoardName);

  PrintMessage(	" StatusRefreshRate: %.2f Hz\n"
    	    	" DACMin value:      %d\n"
    	    	" DACMax value:      %d\n"
    	    	" HVCalibOffset :    %f\n"
    	    	" HVCalibSlope :     %f\n"
    	    	" HVMaxDiff :        %u\n",
		fStatusRefreshRate, DACMin,
		DACMax, fHVCalibOffset, fHVCalibSlope, fHVMaxDiff);
}

// Print help
void ProcessIO::cmd_help() {

  char Buffer[MAX_COM_SIZE];
  for(unsigned int i=0; i<sizeof(CommandList)/sizeof(CL_Struct); i++) {
    snprintf(Buffer, sizeof(Buffer), "%s %s", CommandList[i].Name, CommandList[i].Parameters);
    PrintMessage("%-28s%s\n", Buffer, CommandList[i].Help);
  }
  
  PrintMessage(".<command>                  Execute shell command\n\n"
  	"Items in <> are mandatory, in [] optional, | indicates mutual exclusive or.\n");
} 

// Set new bias voltage
void ProcessIO::cmd_hv() {

  int Int, SingleChannel; 
  unsigned int DACValue, Errors=0;
  double Double;
  bool SetDac;

  // Interpretation as DAC value requested?
  if (Parameter.size() == 4 && Match(Parameter[3], "dac")) SetDac = true;
  else SetDac = false;

  // Interprete first number (might be channel number)
  if (!ConvertToInt(Parameter[1], &SingleChannel)) SingleChannel = -1;

  // Loop over all boards
  for (int i=FirstBoard; i<=LastBoard; i++) {
	// Loop over all channels given as command parameter
 	for (unsigned int n=1; n<Parameter.size()-1; n+=2) {
	  // Loop over all channels
	  for (int j=0; j<NUM_CHAINS; j++) for (int k=0; k<NUM_CHANNELS; k++) {

		// Current channel must be considered?
		if (!Match(Parameter[1], "all") && !(i == (int) pm->Pixel_to_HVboard(Parameter[n]) &&
			j == (int) pm->Pixel_to_HVchain(Parameter[n]) && k == (int) pm->Pixel_to_HVchannel(Parameter[n])) &&
			!(j == SingleChannel/NUM_CHANNELS && k == SingleChannel%NUM_CHANNELS)) continue;

		// Voltage change (number starts with + oder -) ignored if current DAC value is zero
		if (isdigit(Parameter[n+1][0])==0 && fHVBoard[i]->HV[j][k] == 0) continue;

		// Set new voltage/DAC value
		if (!SetDac){
		  // Convert voltage value and check format 
		  if (!ConvertToDouble(Parameter[n+1], &Double)) {
			PrintMessage("Error: Wrong number format for voltage setting\n");
			goto LeaveLoop;
		  }
		  // Adjust voltage and DAV value
		  if (isdigit(Parameter[n+1][0]) == 0) fHVBoard[i]->HVV[j][k] += Double;
		  else fHVBoard[i]->HVV[j][k] = Double;
		  DACValue = calib->HVToDAC(fHVBoard[i]->HVV[j][k], i, j, k);
		}
		else {
		  // Convert DAC value and check format 
		  if (!ConvertToInt(Parameter[n+1], &Int)) {
			PrintMessage("Error: Wrong number format for DAC voltage setting\n");
			goto LeaveLoop;
		  } 
		  // Adjust DAC value
		  if (isdigit(Parameter[n+1][0]) == 0) DACValue = fHVBoard[i]->HV[j][k] + Int;
		  else DACValue = Int;
		}

		// Set new voltage (if DAC value, update calibrated value)
		if (!RampVoltage(DACValue, i, j, k)) Errors++;
		if (SetDac) fHVBoard[i]->HVV[j][k] = calib->DACToHV(fHVBoard[i]->HV[j][k], i, j, k);
	  } // Channels and chains	
	} // Loop over command argument

	// Update DIM service for this boar
	LeaveLoop:
	fHVBoard[i]->BiasVolt->updateService();
  } // Boards
  
  if (Errors > 0) PrintMessage("Errors on %d channel(s) occurred\n", Errors);
}

// Load bias settings from file
void ProcessIO::cmd_load() {

  char Buffer[MAX_COM_SIZE];
  int NBoards = 0, Errors = 0, Chain, Channel;
  unsigned int DACValue;
  FILE *File;

  if ((File=fopen(Parameter[1].c_str(), "r")) == NULL) {
    PrintMessage("Error: Could not open file '%s' (%s)\n", Parameter[1].c_str(), strerror(errno));
    return;
  }

  while (fgets(Buffer, sizeof(Buffer), File) != NULL) {
    for (int Board=0; Board<NumHVBoards; Board++) {
	  if (Match(fHVBoard[Board]->BoardName, Buffer)) {
		PrintMessage("Found bias settings for board %d (%s)\n\r",fHVBoard[Board]->GetBoardNumber(), fHVBoard[Board]->BoardName);

		Chain = 0;  Channel = 0;
		while (fscanf(File, "%u", &DACValue)==1 && Chain<NUM_CHAINS) {
    		  if (!RampVoltage(DACValue, Board, Chain, Channel)) {
	    	Errors++;
	    	PrintMessage("Error: Could not ramp chain %d, channel %d\n", Chain, Channel);
    		  }
		  else {
	    	PrintMessage("Ramped chain %d, channel %d to %u (%.2f V)                         \r",
	    	   Chain,Channel,DACValue,calib->DACToHV(DACValue,Board,Chain,Channel));
    		  }
		  fHVBoard[Board]->HVV[Chain][Channel] = calib->DACToHV(fHVBoard[Board]->HV[Chain][Channel], Board, Chain, Channel);

		  if(++Channel == NUM_CHANNELS) {
	    	Chain++;
	    	Channel = 0;
		  }
		}

    	// Update DIM service
		fHVBoard[Board]->BiasVolt->updateService();

		if (ferror(File) != 0) {
		  PrintMessage("Error reading DAC value from file, terminating. (%s)\n",strerror(errno));
    		  return;
		}
		else PrintMessage("\nFinished updating board\n");
    	NBoards++;
	  }
    } // Loop over boards
  } // while()
    	    
  if (NBoards != NumHVBoards) {
    PrintMessage("Warning: Could not load bias settings for all connected HV boards\n");
  }
  else if (Errors == 0) PrintMessage("Success: Read bias settings for all connected HV boards\n");
  if (Errors != 0) PrintMessage("Warning: Errors on %d channel(s) occurred\n", Errors);

  if (fclose(File) != 0) PrintMessage("Error: Could not close file '%s'\n",Parameter[1].c_str());
}
	   
// Set status refresh rate
void ProcessIO::cmd_rate() {

  double Rate;

  if (!ConvertToDouble(Parameter[1], &Rate)) {
     PrintMessage("Error: Wrong number format\n");
     return;   
  }

  // Check limits
  if (Rate<MIN_RATE || Rate>MAX_RATE) {
    PrintMessage("Refresh rate out of range (min: %.2f Hz, max: %.2f Hz)\n", MIN_RATE, MAX_RATE);
    return;
  }

  fStatusRefreshRate = Rate;
  PrintMessage("Refresh rate set to %.2f Hz\n", fStatusRefreshRate);
}
  
// Reset
void ProcessIO::cmd_reset() {

  for (int i=FirstBoard; i<=LastBoard; i++) ResetBoard(i);
}

// Save bias settings of all boards
void ProcessIO::cmd_save() {

  FILE *File;
  time_t time_now_secs;
  struct tm *Time;

  time(&time_now_secs);
  Time = localtime(&time_now_secs);

  if ((File = fopen(Parameter[1].c_str(), "w")) == NULL) {
    PrintMessage("Error: Could not open file '%s' (%s)\n", Parameter[1].c_str(), strerror(errno));
    return;
  }

  fprintf(File,"********** Bias settings, %04d %02d %02d, %02d:%02d:%02d **********\n\n",
	  1900 + Time->tm_year, 1 + Time->tm_mon,
	  Time->tm_mday, Time->tm_hour, Time->tm_min, Time->tm_sec);

  for (int i=0; i<NumHVBoards; i++) {
    fprintf(File, "%s\n\n", fHVBoard[i]->BoardName);

    for (int j=0; j<NUM_CHAINS; j++) {
  for (int k=0; k<NUM_CHANNELS; k++) fprintf(File,"%5d ",fHVBoard[i]->HV[j][k]);
  fprintf(File, "\n");
    }
    fprintf(File, "\n");
  }

  if (fclose(File) != 0) {
    PrintMessage("Error: Could not close file '%s' (%s)\n", Parameter[1].c_str(), strerror(errno));  
  }
}

// Start monitoring
void ProcessIO::cmd_start() {

  state = active;
  pthread_kill(HVMonitor, SIGUSR1);
  Message(INFO, "Status monitoring activated");
}

// Print status
void ProcessIO::cmd_status() {

  PrintMessage("\n Status monitor: %s\n", state_str[state]);
  PrintMessage(" Status refresh rate [Hz]: %.2f\n", fStatusRefreshRate);
  PrintMessage(" Total number of boards: %d\n", NumHVBoards);
  PrintMessage(" Active boards: %d\n\n", LastBoard - FirstBoard + 1);

  for (int i=FirstBoard; i<=LastBoard; i++) {
    PrintMessage(" BOARD %d (%s)   Wrap counter: %s (%d)  Manual reset: %s\n    Time-out: %.2f s  Error count: %d\n\n",
		fHVBoard[i]->GetBoardNumber(), fHVBoard[i]->BoardName,
		fHVBoard[i]->WrapOK ? "ok":"error",	fHVBoard[i]->LastWrapCount, 
		fHVBoard[i]->ResetButton ? "yes" : "no", fHVBoard[i]->fTimeOut, fHVBoard[i]->ErrorCount);

    for (int j=0; j<NUM_CHAINS; j++) {
	  PrintMessage("  CHAIN %d     Over-current: %s\n", j, fHVBoard[i]->Overcurrent[j] ? "yes" : "no");
	  for (int l=0; l<NUM_CHANNELS; l++) {
		if (l%8 == 0) PrintMessage("\r%3.1d:  ", j*NUM_CHANNELS+l);
		if (Parameter.size() == 2) PrintMessage("%5d ",fHVBoard[i]->HV[j][l]);
    	else PrintMessage("%#5.2f ",fHVBoard[i]->HVV[j][l]);
		if (l%8 == 7) PrintMessage("\n");
      }
    }
  }
} 

// Stop monitoring
void ProcessIO::cmd_stop() {

  state = stopped;
  pthread_kill(HVMonitor, SIGUSR1);
  Message(INFO, "Status monitor stopped");
}

// Set timeout to return from read
void ProcessIO::cmd_timeout() {

  double Timeout;

  if (!ConvertToDouble(Parameter[1], &Timeout)) {
     PrintMessage("Error: Wrong number format\n");
     return;   
  }

  for (int i=0; i<NumHVBoards; i++) fHVBoard[i]->SetTimeOut(Timeout);
  if (NumHVBoards > 0) PrintMessage("Timeout set to %.2f s for all boards\n", Timeout);
}
    
// Print uptime
void ProcessIO::cmd_uptime() {

  time_t ActualT;
  time (&ActualT);
  PrintMessage("%d:%02d:%02d\n", (int) difftime(ActualT, StartTime)/3600, ((int) difftime(ActualT, StartTime)/60)%60, (int) difftime(ActualT, StartTime)%60);
} 

// Exit program
void ProcessIO::cmd_exit() {

  ExitRequest = true;
  pthread_kill(HVMonitor, SIGUSR1); 
}
  

// Print message to screen and to DIM text service
void ProcessIO::PrintMessage(const char *Format, ...) {

  static char Error[] = "vasprintf() failed in PrintMessage()";
  char *Text;

  // Evaluate arguments    
  va_list ArgumentPointer;
  va_start(ArgumentPointer, Format);
  if (vasprintf(&Text, Format, ArgumentPointer) == -1) Text = Error;
  va_end(ArgumentPointer);
 
  // Print to console
  if(strlen(Text)>0 && Text[strlen(Text)-1]=='\n') printf("\r%s%s", Text, Prompt); // New prompt
  else printf("%s", Text);
  fflush(stdout);

  // Send to DIM text service
  ConsoleOut->updateService(Text); 

  // Free old text
  if (ConsoleText != Error) free(ConsoleText);
  ConsoleText = Text; 
}


// Ramp to new voltage with maximum step size given in fHVMaxDiff
// No ramping when decreasing voltage
bool ProcessIO::RampVoltage(unsigned int Target, int Board, int Chain, int Channel) {

  while (fHVBoard[Board]->HV[Chain][Channel] != (int) Target) {	  
    int Diff = Target - fHVBoard[Board]->HV[Chain][Channel];
    if (Diff > (int) fHVMaxDiff) Diff = fHVMaxDiff;

    if (fHVBoard[Board]->SetHV(Chain, Channel, fHVBoard[Board]->HV[Chain][Channel]+Diff) != 1) {
      Message(ERROR, "Could not set bias of board %d, chain %d, channel %d. Skipping channel\n",fHVBoard[Board]->GetBoardNumber(),Chain,Channel);
      return false;
    }
  }

  return true;
}


// Check board status (ignore board if it has more than 10 read/write errors)
void ProcessIO::Monitor() {

  static bool Warned = false;
  int Ret;

  // Lock because command execution runs in different thread
  if ((Ret = pthread_mutex_lock(&Mutex)) != 0) {
	Message(FATAL, "pthread_mutex_lock() failed in commandHandler() (%s)", strerror(Ret));
  }

  for (int i=0; i<NumHVBoards; i++) {
    if (fHVBoard[i]->ErrorCount > 10) {
      if (!Warned) {
        Warned = true;
        Message(WARN, "Warning: Board %d has many read/write errors, status monitor disabled", i);
      }
      continue;
    }

    if (fHVBoard[i]->GetStatus() != 1) {
      Message(ERROR, "Error: Monitor could not read status of board %d", fHVBoard[i]->GetBoardNumber());
    }
    
    if (fHVBoard[i]->ResetButton) {
      Message(INFO, "Manual reset of board %d", fHVBoard[i]->GetBoardNumber());
      ResetBoard(i);
    }
    
    if (!fHVBoard[i]->WrapOK) {
      Message(ERROR, "Error: Wrap counter mismatch board %d",fHVBoard[i]->GetBoardNumber());
    }

    for (int j=0; j<NUM_CHAINS; j++) {
      if (fHVBoard[i]->Overcurrent[j]) {
		Message(WARN, "Warning: Overcurrent in chain %d of board %d",j,fHVBoard[i]->GetBoardNumber());
		ResetBoard(i);
      }
    }
  }
  
  // Unlock
  if ((Ret = pthread_mutex_unlock(&Mutex)) != 0) {
	Message(FATAL, "pthread_mutex_lock() failed in commandHandler() (%s)", strerror(Ret));
  }

}


// Send reset to board and clear voltage arrays
void ProcessIO::ResetBoard(int i) {
    
  if (fHVBoard[i]->Reset() == 1) {
    PrintMessage("Reset of board %d\n", fHVBoard[i]->GetBoardNumber());
    PrintBoardStatus(i);
  }
  else PrintMessage("Error: Could not reset board %d\n",fHVBoard[i]->GetBoardNumber());
}


// Print current board status
void ProcessIO::PrintBoardStatus(int i) {

  PrintMessage("Status board %d (%s): MR %s OC0 %s OC1 %s OC2 %s OC3 %s WC %s (%d)\n",
    	  fHVBoard[i]->GetBoardNumber(), fHVBoard[i]->BoardName,
	  fHVBoard[i]->ResetButton ? "yes" : "no",
	  fHVBoard[i]->Overcurrent[0] ? "yes" : "no",
	  fHVBoard[i]->Overcurrent[1] ? "yes" : "no",
	  fHVBoard[i]->Overcurrent[2] ? "yes" : "no",
	  fHVBoard[i]->Overcurrent[3] ? "yes" : "no",
	  fHVBoard[i]->WrapOK ? "ok":"error", fHVBoard[i]->LastWrapCount);
}


// Command handling (mutex needed because of monitor thread)
void ProcessIO::commandHandler() {

  int Ret;

  if ((Ret = pthread_mutex_lock(&Mutex)) != 0) {
	Message(FATAL, "pthread_mutex_lock() failed in commandHandler() (%s)", strerror(Ret));
  }

  if ((getCommand()==Command) && (*(Command->getString()+Command->getSize()-1)=='\0')) {
	SendToLog("Command '%s' from %s (ID %d)", Command->getString(), getClientName(), getClientId());
	CommandControl(Command->getString());
  }

  if ((Ret = pthread_mutex_unlock(&Mutex)) != 0) {
	Message(FATAL, "pthread_mutex_unlock() failed in commandHandler() (%s)", strerror(Ret));
  }
}

// Check if two strings match (min 1 character must match)
bool Match(string str, const char *cmd) {
  return strncasecmp(str.c_str(),cmd,strlen(str.c_str())==0 ? 1:strlen(str.c_str())) ? false:true;
}


// Convert string to double
// Returns false if conversion did not stop on whitespace or EOL character
bool ConvertToDouble(string String, double *Result) {

  char *EndPointer;
  
  *Result = strtod(String.c_str(), &EndPointer);
  if(!isspace(*EndPointer) && *EndPointer!='\0') return false;
  return true;
}


// Convert string to int
// Returns false if conversion did not stop on whitespace or EOL character
bool ConvertToInt(string String, int *Result) {

  char *EndPointer;
  
  *Result = (int) strtol(String.c_str(), &EndPointer, 0);
  if(!isspace(*EndPointer) && *EndPointer!='\0') return false;
  return true;
}
