
/********************************************************************\

  Name:         HV.cc

  Created by:   Sebastian Commichau, November 2008
                commichau@phys.ethz.ch

  Contents:     Main class for HV supply
                
\********************************************************************/

#include "HV.h"
#include "ProcessIO.h" // Must be not in HV.h to avoid problem with declaring class ProcessIO

using namespace std;

// Constructor
HVBoard::HVBoard(int DeviceNumber, string Name, class ProcessIO *PIO) {
   
  struct termios tio;
 
  m = PIO;
  SetTimeOut(m->fTimeOut);
  BoardNumber = DeviceNumber;
  BoardName = new char [Name.size()+1];
  strcpy(BoardName, Name.c_str());
  
  // Create DIM services
  stringstream ID;
  ID << setfill('0') << setw(2) << BoardNumber;

  NameService = new DimService ((SERVER_NAME"/NAME/ID"+ID.str()).c_str(), BoardName);
  BiasVolt = new DimService ((char *) (SERVER_NAME"/VOLT/ID"+ID.str()).c_str(), (char *) "D", HVV, NUM_CHAINS*NUM_CHANNELS*sizeof(double));

  // Initialise board variables
  for (int i=0; i<NUM_CHAINS; i++) Overcurrent[i] = false;
  ResetButton = false;
  WrapOK = true;
  LastWrapCount = -1;
  ErrorCount = 0;

  ClearVoltageArrays();

  // Open device
  if ((fDescriptor = open(("/dev/"+Name).c_str(), O_RDWR|O_NOCTTY|O_NDELAY)) == -1) {
    if(errno != 2) m->PrintMessage("Error: Could not open device %d/%s (%s)\n", DeviceNumber, Name.c_str(), strerror(errno));
    return;
  }

  // Get current serial port settings
  if (tcgetattr(fDescriptor, &tio) == -1) {
    m->PrintMessage("Error: tcgetattr() failed (%d/%s)\n", errno, strerror(errno));
    return;   
  }

  // Set baudrate and raw mode
  if (cfsetspeed(&tio, BAUDRATE) == -1) m->PrintMessage("Error: Could not set baud rate (%s)\n", strerror(errno));
  cfmakeraw(&tio);
  if (tcsetattr(fDescriptor, TCSANOW, &tio ) == -1) m->PrintMessage("Error: tcsetattr() failed (%s)\n", strerror(errno));

  //  Synchronize HV board (if fails, closes device and sets fDescriptor to -2) 
  unsigned char wbuf = REG_STATUS;
  int trial = 0, ret;
  
  while(++trial<=3) {
    if((ret = Communicate(&wbuf, 1)) == 1) {
      Reset();
      return;
    }
    if (ret==0) break;
    wbuf = 0;
  }
  close(fDescriptor);
  fDescriptor = -2;
  return;
}

// Destructor (Resetting board)
HVBoard::~HVBoard() {
  if(fDescriptor >= 0) {
    Reset();
    close(fDescriptor);
  }

  delete NameService;
  delete BiasVolt;
  delete[] BoardName;
}


// Communicate: Write and read from HV Board until fTimeOut has been reached 
//
// Returns: 0 error, 1 success, -1 fTimeOut exceeded
int HVBoard::Communicate(unsigned char* wbuf, int Bytes) {

  unsigned char rbuf;
  int ret;
  fd_set SelectDescriptor;
  
  // === Write data ===
  if ((ret=write(fDescriptor, wbuf, Bytes)) < Bytes) {
    if (ret == -1) m->Message(m->ERROR, "Could not write data to HV board (%s)", strerror(errno));
    else m->Message(m->ERROR, "Could write only %d of %d bytes to HV board", ret, Bytes);
    ErrorCount++;
    return 0;
  }

  // === Try to read until time-out ===
  FD_ZERO(&SelectDescriptor);   FD_SET(fDescriptor, &SelectDescriptor);
  struct timeval WaitTime = {(long) fTimeOut, (long) ((fTimeOut-(long) fTimeOut)*1e6)};
  if (select(fDescriptor+1, &SelectDescriptor, NULL, NULL, &WaitTime)==-1) {
    m->Message(m->ERROR, "Error with select() (%s)", strerror(errno));
    return 0;
  }
  // Time-out expired?
  if (!FD_ISSET(fDescriptor, &SelectDescriptor)) return -1;

  // Read error?    
  if ((ret = read(fDescriptor, &rbuf, 1)) == -1) {
    m->Message(m->ERROR, "Read error (%s)", strerror(errno));
    ErrorCount++;
    return 0;
  }

  // === Update status information ===
  if (LastWrapCount != -1) {
    if ((LastWrapCount+1)%8 == (rbuf & 0X07)) WrapOK = true;
    else WrapOK = false;
  }
  LastWrapCount = rbuf & 0X07;  
  for (int i=0; i<NUM_CHAINS; i++) Overcurrent[i] = (rbuf & (0X08 << i));
  ResetButton = (bool) (rbuf & 0X80);

  return 1;
}


/* Reset HV board - uses TRead() and has same return values */
int HVBoard::Reset() {
  
  unsigned char wbuf[] = {REG_RESET,0,0};
  int ret;
  
  if((ret = Communicate(wbuf, 3)) == 1) {
    ClearVoltageArrays();
    ErrorCount = 0;
  }  
  return ret;
}


/* Read status register - uses TRead() and has same return values */
int HVBoard::GetStatus() {

  unsigned char wbuf[] = {REG_STATUS,0,0};
  
  return Communicate(wbuf, 3);
}


/* Set high voltage - uses TRead() and has same return values */
int HVBoard::SetHV(int chain, unsigned int channel, int hv) {

  unsigned char wbuf[] = {0,0,0};
  int ret;
  
  if (hv<0 || hv > 0x3FFF) {
    m->PrintMessage("Error: Voltage out of range 0-0x3FFF\n"); 
    return 0;
  }
  
  switch (chain) {
    case 0: wbuf[0] = REG_HV0; break;
    case 1: wbuf[0] = REG_HV1; break;
    case 2: wbuf[0] = REG_HV2; break;
    case 3: wbuf[0] = REG_HV3; break;

    default : m->PrintMessage(" Error: Chain %d does not exist\n",chain); return 0;
  }
 
  // Assemble bytes
  wbuf[0] |= (unsigned char)((channel >> 2) & 0X00000007);  // Add address [A4-A3]
  wbuf[1] |= (unsigned char)((hv >> 8) & 0X000000FF);       // Add [D13-D8]
  wbuf[1] |= (unsigned char)((channel << 6)  & 0X000000C0); // Add [A1-A0]
  wbuf[2] |= (unsigned char)(hv & 0X000000FF);              // Add [D7-D0]
   
  if ((ret = Communicate(wbuf, 3)) == 1) HV[chain][channel] = hv;
  return ret;
}


// Set all voltages of board to zero
void HVBoard::ClearVoltageArrays() {

  for (int j=0; j<NUM_CHAINS; j++) {
    for (int k=0; k<NUM_CHANNELS; k++){
      HV[j][k] = 0;
      HVV[j][k] = 0.0;      
    }
  }
  // Update DIM services
  BiasVolt->updateService();
}


// Set time-out to wait for read
void HVBoard::SetTimeOut(double t) {

  if (t >= MIN_TIMEOUT && t <= MAX_TIMEOUT) fTimeOut  = t;
  else fTimeOut  = 1.0;
}
