#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/stat.h>

#define PORT 8899

char* title[]={"w0", "r0", "w1", "r1"};

struct speed {
    struct speed *prev, *next;
    int size, num;
    float* f;
};

struct speed* speed;

static int
xrecv(int s, int n, int* v)
{
    int res, rest=n*4;
    char* p=(char*)v;
    while (rest) {
        res=recv(s, p, rest, 0);
        if (res<0) {
            if (errno!=EINTR) {
                perror("recv");
                return -1;
            } else
                res=0;
        } else if (res==0) {
            fprintf(stderr, "no more data\n");
            return -1;
        }
        rest-=res;
        p+=res;
    }
    return 0;
}

static int
add_entry(struct speed* e)
{
    struct speed *prev, *next;

    next=speed; prev=0;
    
    while (next && (next->size<=e->size)) {prev=next; next=next->next;}
    fprintf(stderr, "size=%d; ", e->size);
    if (prev)
        fprintf(stderr, "prev.size=%d; ", prev->size);
    else
        fprintf(stderr, "prev=0; ");
    if (next)
        fprintf(stderr, "next.size=%d\n", next->size);
    else
        fprintf(stderr, "next=0\n");

    if (next) {
        e->next=next;
        next->prev=e;
    } else
        e->next=0;
    if (prev) {
        e->prev=prev;
        prev->next=e;
    } else {
        e->prev=0;
        speed=e;
    }

    return 0;
}

static void
dump_speed(void)
{
    struct speed *e=speed;
    fprintf(stderr, "---------------\n");
    while (e) {
        int i;
        fprintf(stderr, "%6d", e->size);
        for (i=0; i<e->num; i++)
                fprintf(stderr, " %12.2f\n", e->f[i]);
        fprintf(stderr, "\n");
        e=e->next;
    }
}

static int
setup_plot(char* name)
{
    struct stat buf;
    static time_t last_mtime=0;

    if (stat(name, &buf)<0) {
        fprintf(stderr, "cannot stat \"%s\": %s\n", name, strerror(errno));
        return -1;
    }
    
    if (buf.st_mtime!=last_mtime) {
        FILE* f;
        char s[1024];

        f=fopen(name, "r");
        if (!f) {
            fprintf(stderr, "cannot open \"%s\": %s\n", name, strerror(errno));
            return -1;
        }
        while (fgets(s, 1024, f)) {
            printf("%s\n", s);
        }
        fclose(f);
        last_mtime=buf.st_mtime;
    }
    return 0;
}

static void
plot_it(void)
{
    struct speed *e;
    int i;

    if (!speed) return;

    setup_plot("gnuplot.ini");

    printf("plot ");
    for (i=0; i<speed->num; i++)
        printf("\"-\" us 1:2 title '%s' w l%s",
                title[i], (i<speed->num-1)?", ":"\n");

    for (i=0; i<speed->num; i++) {
        e=speed;
        while (e) {
            printf("%6d %12.2f\n", e->size, e->f[i]);
            e=e->next;
        }
        printf("e\n\n");
    }
    fflush(stderr);
}

static void
init_plot(void)
{
    printf("set term x11\n");
    printf("set title \"Throughput\"\n");
    printf("set xlabel \"Size/words\"\n");
    printf("set ylabel \"Byte/s\"\n");
    printf("plot \"-\" with points\n");
    printf("0 0\n1 1\ne\n\n");
}


int main(int argc, char* argv)
{
    int s, ns;
    struct sockaddr_in addr;
    struct sockaddr caddr;
    struct in_addr in_addr;
    int tmp, res;

    res=setvbuf(stdout, 0, _IONBF, 0);
    if (res<0) {perror("setvbuf"); return 1;}

    speed=0;

    init_plot();

    bzero(&addr, sizeof(struct sockaddr_in));
    bzero(&caddr, sizeof(struct sockaddr));
    addr.sin_family=AF_INET;
    addr.sin_port=htons(PORT);
    addr.sin_addr.s_addr=INADDR_ANY;
    s=socket(addr.sin_family, SOCK_STREAM, 0);
    if (s<0)
        {perror("socket"); return 1;}
    if (bind(s, (struct sockaddr*)&addr, sizeof(struct sockaddr_in))<0)
        {perror("bind"); return 1;}
    if (listen(s, 1)<0) {perror("listen"); return 1;}
    tmp=sizeof(struct sockaddr_in);
    ns=accept(s, (struct sockaddr*)&addr, &tmp);
    if (ns<0) {perror("accept"); return 1;}
    in_addr.s_addr=addr.sin_addr.s_addr;
    fprintf(stderr, "%s accepted\n", inet_ntoa(in_addr));

    do {
        int n, i;
        struct speed *e;

        if (xrecv(ns, 1, &n)<0) return 1;

        e=malloc(sizeof(struct speed)+n*sizeof(float));
        if (!e) {
            perror("malloc");
            return -1;
        }
        e->f=malloc(n*sizeof(float));
        if (!e->f) {
            perror("malloc");
            return -1;
        }
        e->num=n;

        if (xrecv(ns, 1, &e->size)<0) return 1;
        for (i=0; i<n; i++) {
            if (xrecv(ns, 1, (int*)(e->f+i))<0) return 1;
        }
        add_entry(e);
        plot_it();
    } while (1);

    return 0;
}
