#include "MHEvent.h"

#include <TStyle.h>
#include <TCanvas.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MMcEvt.hxx"

#include "MHCamera.h"
#include "MParList.h"
#include "MTaskList.h"
#include "MPedestalCam.h"
#include "MPedestalPix.h"
#include "MParList.h"
#include "MCerPhotEvt.h"
#include "MRawEvtHeader.h"
#include "MRawRunHeader.h"

MHEvent::~MHEvent()
{
    if (fHist)
        delete fHist;
}

Bool_t MHEvent::SetupFill(const MParList *plist)
{
    MTaskList *tlist = (MTaskList*)plist->FindObject("MTaskList");

    fImgCleanStd = (MImgCleanStd*)tlist->FindObject("MImgCleanStd");
    fMcEvt       = (MMcEvt*)plist->FindObject("MMcEvt");

    fRawEvtData = (MRawEvtData*)plist->FindObject("MRawEvtData");
    if (!fRawEvtData)
        *fLog << warn << "MRawEvtData not found..." << endl;

    fRawRunHeader = (MRawRunHeader*)plist->FindObject("MRawRunHeader");
    if (!fRawRunHeader)
        *fLog << warn << dbginf << "MRawRunHeader not found..." << endl;

    fRawEvtHeader = (MRawEvtHeader*)plist->FindObject("MRawEvtHeader");
    if (!fRawEvtHeader)
        *fLog << warn << dbginf << "MRawEvtHeader not found..." << endl;

    MGeomCam *cam = (MGeomCam*)plist->FindObject("MGeomCam");
    if (!cam)
    {
        *fLog << err << GetDescriptor() << ": No MGeomCam found... aborting." << endl;
        return kFALSE;
    }

    if (fHist)
        delete (fHist);

    fHist = new MHCamera(*cam);
    fHist->AddNotify(fClone);

    switch (fType)
    {
    case kEvtSignal:
        fHist->SetName("Signal");
        fHist->SetYTitle("S [au]");
        break;
    case kEvtPedestal:
        fHist->SetName("Pedestal");
        fHist->SetYTitle("P [au]");
        break;
    case kEvtPedestalRMS:
        fHist->SetName("Pedestal RMS");
        fHist->SetYTitle("\\sigma_{P} [au]");
        break;
    case kEvtRelativeSignal:
        fHist->SetName("Signal/PedRMS");
        fHist->SetYTitle("S/P_{rms}");
        break;
    case kEvtCleaningLevels:
        fHist->SetName("CleanLevels");
        fHist->SetYTitle("L");
        if (!fImgCleanStd)
            return kFALSE;
        break;
    }

    return kTRUE;
}

Bool_t MHEvent::Fill(const MParContainer *par, const Stat_t weight=1)
{
    if (fHist->IsFreezed())
        return kTRUE;

    if (!par)
        return kFALSE;

    if (fRawEvtData)
        fRawEvtData->Copy(fClone);

    switch (fType)
    {
    case kEvtSignal:
    case kEvtPedestal:
        fHist->SetCamContent(*(MCamEvent*)par, 0);
        break;
    case kEvtPedestalRMS:
        fHist->SetCamContent(*(MCamEvent*)par, 0);
        break;
    case kEvtRelativeSignal:
    case kEvtCleaningLevels:
        fHist->SetCamContent(*(MCamEvent*)par, 0);
        break;
    }

    TString s;
    if (fRawEvtHeader)
    {
        s += "Event #";
        s += fRawEvtHeader->GetDAQEvtNumber();
    }
    if (fRawEvtHeader && fRawRunHeader)
        s += " of ";

    if (fRawRunHeader)
    {
        s += "Run #";
        s += fRawRunHeader->GetNumEvents();
    }

    if (fMcEvt)
    {
        TString txt;// = "#splitline{";

        switch (fMcEvt->GetPartId())
        {
        case kGAMMA:
            txt += "Gamma: ";
            break;
        case kPROTON:
            txt += "Proton: ";
            break;
        case kHELIUM:
            txt += "Helium: ";
            break;
        default:
            s += "Particle Id#";
            s += fMcEvt->GetPartId();
            s += ": ";
        }

        s.Insert(0, txt);

        //s += "}{";
        s += "  E=";
        if (fMcEvt->GetEnergy()>1000)
            txt += Form("%.1fTeV", fMcEvt->GetEnergy()/1000);
        else
            txt += Form("%dGeV", fMcEvt->GetEnergy()+.5);
        s += " r=";
        s += (int)(fMcEvt->GetImpact()/100+.5);
        s += "m ZA=";
        s += (int)(fMcEvt->GetTheta()*180/TMath::Pi()+.5);
        s += " ";
        s += fMcEvt->GetPhotElfromShower();
        s += "PhEl";
        //s += "}";
    }

    gPad=NULL;
    fHist->SetTitle(s);

    return kTRUE;
}

void MHEvent::Draw(Option_t *)
{
    if (!fHist)
    {
        *fLog << warn << "MHEvent::Draw - fHist==NULL not initialized." << endl;
        return;
    }

    TVirtualPad *pad = gPad ? gPad : MakeDefCanvas(this);
    pad->SetBorderMode(0);

    pad->Divide(1,1);

    pad->cd(1);
    gPad->SetBorderMode(0);
    gPad->Divide(1,1);
    gPad->cd(1);
    gPad->SetBorderMode(0);
    fHist->Draw();
}
