/********************************************************************\

  Read 8 Channels from FAD-Board
  Write Commands to Socket 0, Commands must be (n * 16) Bit long
  
  Usage: SocketClient [IP-Address]
  
  kw, 05.10
  
  based on:
  Generic example for remote control via a socket interface

  Oliver Grimm, March 2009

\********************************************************************/
#ifndef PATH_MAX
#define PATH_MAX 1000
#endif
#include "simple_daq.h"

int SocketDescriptor[8];

int main(int argc, char *argv[])
{
	char Buffer[MAX_COM_SIZE];
	
	// default server
	char ServerName[IP_ADDR_LENGTH] = IP_ADDR_DEFAULT; // default IP-Address

	int i, read_return;

	int DAQPort = FIRST_DAQPORT;
	struct sockaddr_in SocketAddress[8];
	struct in_addr Serveripv4addr;
	fd_set ReadFileDescriptor;

	FILE* fhandle[8];
	char fname[PATH_MAX];

	int max_fd = 0;

	// Get IP-Address from command line
	if (argc > 1)
	{
		strncpy (ServerName, argv[1], IP_ADDR_LENGTH);
	}
	
	// Convert IP-Addr to binary
	if (inet_pton (AF_INET, ServerName, &Serveripv4addr) <= 0)
	{
		printf ("Error: network address not valid\n");
		exit_program (EXIT_FAILURE);
	}

	// Open sockets
	for (i = 0; i < 8; i++)
	{
		if ((SocketDescriptor[i] = socket (PF_INET, SOCK_STREAM, 0)) == -1)
		{
			printf ("Error: Could not open socket Nr.: %d\n", i);
			exit_program (EXIT_FAILURE);
		}
		else
		{
			if (SocketDescriptor[i] > max_fd)
			{
				max_fd = SocketDescriptor[i];
			}
		}
	}
	
	// Open files for output
	for (i = 0; i < 8; i++)
	{
		sprintf (fname, "socket-%d.dat", i);
		if ((fhandle[i] = fopen (fname, "w")) == NULL)
		{
			printf ("Error: Could not open file %s\n", fname);
			exit_program (EXIT_FAILURE);
		}
	}

	// Connect to server
	printf ("Trying to connect to %s...\n", ServerName);
	for (i = 0; i < 8; i++)
	{
		SocketAddress[i].sin_family = PF_INET;
		SocketAddress[i].sin_port = htons ((unsigned short) DAQPort + i);
		SocketAddress[i].sin_addr = Serveripv4addr;
		
		if (connect (SocketDescriptor[i], (struct sockaddr *) &SocketAddress[i], sizeof (SocketAddress[i])) == -1)
		{
			printf ("Error: Could not connect to server %s (port %d)\n", ServerName, DAQPort + i);
			exit_program (EXIT_FAILURE);
		}
		else
		{
			printf ("Connected to %s:%d\n", ServerName, DAQPort + i);
		}
	}

	signal (SIGPIPE, SIG_IGN); // Do not kill process if writing to closed socket
	signal (SIGINT, int_handler); // Cleanup after SIGINT (CTRL-C)

	// Main loop
	while (true)
	{
		fflush (stdout);

		FD_ZERO (&ReadFileDescriptor);
		
		FD_SET(STDIN_FILENO, &ReadFileDescriptor);
		
		for (i = 0; i < 8; i++)
		{
			FD_SET (SocketDescriptor[i], &ReadFileDescriptor);
		}

		// Wait for data from sockets
		if (select (((int) max_fd) + 1, &ReadFileDescriptor, NULL, NULL, NULL) == -1)
		{
			perror ("Error with select()\n");
			break;
		}
		
		memset (Buffer, 0, sizeof (Buffer));

		// Data from STDIN
		if (FD_ISSET (STDIN_FILENO, &ReadFileDescriptor))
		{
			fgets (Buffer, MAX_COM_SIZE, stdin);
			// Send command to socket 0
			cmd_send (Buffer, SocketDescriptor[0]);			
		}
		// Data from sockets
		else
		{
			// Check all sockets
			for (i = 0; i < 8; i++)
			{
				if (FD_ISSET (SocketDescriptor[i], &ReadFileDescriptor))
				{ 
					// Only for testing
					memset (Buffer, 0xAA, sizeof (Buffer));
					if ((read_return = read (SocketDescriptor[i], Buffer, MAX_COM_SIZE)) == 0)
					{
						printf ("Error: Server not existing anymore, exiting...\n");
						exit_program (EXIT_FAILURE);
					}
	
					if (read_return > 0)
					{
						printf ("Socket [%d]: Read %d Bytes\n", i, read_return);
					
						fwrite (Buffer, 1, (size_t) read_return, fhandle[i]);
						// Important!!!
						fflush (fhandle[i]);
					}
				}
			}
		}
	} // while (TRUE)

	exit (EXIT_SUCCESS);
}

// close sockets and exit
void exit_program (int exit_status)
{
	int i;
	
	printf ("\nClosing Sockets...");
	for (i = 0; i < 8; i++)
	{
		close (SocketDescriptor[i]);
	}
	printf (" done\n");
	
	exit (exit_status);
}

// SIGINT
void int_handler (int sig)
{
	exit_program (EXIT_SUCCESS);
}
