/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 11/2003 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
//  MReportCamera
//
//////////////////////////////////////////////////////////////////////////////
#include "MReportCamera.h"

#include "MLogManip.h"

#include "MAstro.h"
#include "MParList.h"

#include "MCameraCalibration.h"
#include "MCameraCooling.h"
#include "MCameraHV.h"
#include "MCameraLV.h"
#include "MCameraAUX.h"
#include "MCameraLids.h"

ClassImp(MReportCamera);

using namespace std;

MReportCamera::MReportCamera() : MReport("CAMERA-REPORT")
{
    fName = "MReportCamera";
}

Bool_t MReportCamera::SetupReading(MParList &plist)
{
    fCooling = (MCameraCooling*)plist.FindCreateObj("MCameraCooling");
    if (!fCooling)
        return kFALSE;

    fLids = (MCameraLids*)plist.FindCreateObj("MCameraLids");
    if (!fLids)
        return kFALSE;

    fAUX = (MCameraAUX*)plist.FindCreateObj("MCameraAUX");
    if (!fAUX)
        return kFALSE;

    fHV = (MCameraHV*)plist.FindCreateObj("MCameraHV");
    if (!fHV)
        return kFALSE;

    fLV = (MCameraLV*)plist.FindCreateObj("MCameraLV");
    if (!fLV)
        return kFALSE;

    fCalibration = (MCameraCalibration*)plist.FindCreateObj("MCameraCalibration");
    if (!fCalibration)
        return kFALSE;

    return MReport::SetupReading(plist);
}

Bool_t MReportCamera::CheckTag(TString &str, const char *tag) const
{
    if (!str.BeginsWith(tag))
    {
        *fLog << err << "ERROR - '" << tag << "' tag not found." << endl;
        return kFALSE;
    }
    str.Remove(0, strlen(tag)); // Remove DC currents
    return kTRUE;
}

Bool_t MReportCamera::InterpreteDC(TString &str)
{
    if (!CheckTag(str, "DC "))
        return kFALSE;

    str.Remove(0, 577*4); // Remove DC currents
    str=str.Strip(TString::kLeading);
    return kTRUE;
}

Bool_t MReportCamera::InterpreteHV(TString &str)
{
    if (!CheckTag(str, "HV "))
        return kFALSE;

    for (int i=0; i<577; i++)
    {
        const Char_t hex[4] = { str[i*3], str[i*3+1], str[i*3+2], 0 };
        const Int_t n=sscanf(hex, "%3hx", &fHV->fHV[i]);
        if (n==1)
            continue;

        *fLog << err << "ERROR - Reading hexadecimal HV information." << endl;
        return kFALSE;
    }
    str.Remove(0, 577*3); // Remove DC currents
    str=str.Strip(TString::kLeading);
    return kTRUE;
}

Bool_t MReportCamera::InterpreteCOOL(TString &str)
{
    if (!CheckTag(str, "COOL "))
        return kFALSE;

    Int_t len;

    Int_t wall, opt, center, water;
    Short_t hwall, hcenter, hip, lop, pump, ref, valv, res, fans;
    const Int_t n=sscanf(str.Data(), "%d %d %d %d %hu %hu %hu %hu %hu %hu %hu %hu %hu %n",
                         &wall, &opt, &center, &water, &hwall, &hcenter,
                         &hip,  &lop, &pump, &ref, &valv, &res, &fans, &len);
    if (n!=13)
    {
        *fLog << err << "ERROR - Reading information of 'COOL' section." << endl;
        return kFALSE;
    }

    fCooling->fTempWall            = 0.1*wall;
    fCooling->fTempOptLink         = 0.1*opt;
    fCooling->fTempCenter          = 0.1*center;
    fCooling->fTempWater           = 0.1*water;
    fCooling->fHumWall             = (Byte_t)hwall;
    fCooling->fHumCenter           = (Byte_t)hcenter;
    fCooling->fStatusPressureHi    = (Bool_t)hip;
    fCooling->fStatusPressureLo    = (Bool_t)lop;
    fCooling->fStatusPump          = (Bool_t)pump;
    fCooling->fStatusRefrigrerator = (Bool_t)ref;
    fCooling->fStatusValve         = (Bool_t)valv;
    fCooling->fStatusResistor      = (Bool_t)res;
    fCooling->fStatusFans          = (Bool_t)fans;

    str.Remove(0, len);
    str=str.Strip(TString::kLeading);
    return kTRUE;
}

Bool_t MReportCamera::InterpreteLID(TString &str)
{
    if (!CheckTag(str, "LID "))
        return kFALSE;

    Int_t len;
    Short_t limao, limac, limbo, limbc;
    Short_t slimao, slimac, slimbo, slimbc;
    Short_t slida, slidb, mlida, mlidb;
    const Int_t n=sscanf(str.Data(), "%hu %hu %hu %hu %hu %hu %hu %hu %hu %hu %hu %hu %n",
                         &limao, &limac, &limbo, &limbc,
                         &slimao, &slimac, &slimbo, &slimbc,
                         &slida, &slidb, &mlida, &mlidb,
                         &len);
    if (n!=12)
    {
        *fLog << err << "ERROR - Reading information of 'LID' section." << endl;
        return kFALSE;
    }

    fLids->fLidA.fLimitOpen       = (Bool_t)limao;
    fLids->fLidA.fLimitClose      = (Bool_t)limac;
    fLids->fLidA.fSafetyLimitOpen = (Bool_t)slimao;
    fLids->fLidA.fSafetyLimitClose= (Bool_t)slimac;
    fLids->fLidA.fStatusLid       = (Byte_t)slida;
    fLids->fLidA.fStatusMotor     = (Byte_t)mlida;

    fLids->fLidB.fLimitOpen       = (Bool_t)limbo;
    fLids->fLidB.fLimitClose      = (Bool_t)limbc;
    fLids->fLidB.fSafetyLimitOpen = (Bool_t)slimbo;
    fLids->fLidB.fSafetyLimitClose= (Bool_t)slimbc;
    fLids->fLidB.fStatusLid       = (Byte_t)slidb;
    fLids->fLidB.fStatusMotor     = (Byte_t)mlidb;

    str.Remove(0, len);
    str=str.Strip(TString::kLeading);
    return kTRUE;
}

Bool_t MReportCamera::InterpreteHVPS(TString &str)
{
    if (!CheckTag(str, "HVPS "))
        return kFALSE;

    Int_t len;
    Short_t c1, c2;
    const Int_t n=sscanf(str.Data(), "%hd %hd %hd %hd %n",
                         &fHV->fVoltageA, &fHV->fVoltageB, &c1, &c2, &len);
    if (n!=4)
    {
        *fLog << err << "ERROR - Reading information of 'HVPS' section." << endl;
        return kFALSE;
    }

    fHV->fCurrentA = (Byte_t)c1;
    fHV->fCurrentB = (Byte_t)c2;

    str.Remove(0, len);
    str=str.Strip(TString::kLeading);
    return kTRUE;
}

Bool_t MReportCamera::InterpreteLV(TString &str)
{
    if (!CheckTag(str, "LV "))
        return kFALSE;

    Int_t len;
    Short_t vap5, vap12, van12, vbp5, vbp12, vbn12;
    Short_t valp12, vblp12, cap5, cap12, can12, cbp5, cbp12;
    Short_t cbn12, calp12, cblp12, lvps, temp, hum;
    const Int_t n=sscanf(str.Data(), "%hd %hd %hd %hd %hd %hd %hd %hd %hd %hd %hd %hd %hd %hd %hd %hd %hd %hd %hd %n",
                         &vap5, &vap12, &van12, &vbp5, &vbp12, &vbn12,
                         &valp12, &vblp12, &cap5, &cap12, &can12, &cbp5, &cbp12,
                         &cbn12, &calp12, &cblp12, &lvps, &temp, &hum, &len);
    if (n!=19)
    {
        *fLog << err << "ERROR - Reading information of 'LV' section." << endl;
        return kFALSE;
    }

    fLV->fRequestPowerSupply = (Bool_t)lvps;
    fLV->fTemp = 0.1*temp;
    fLV->fHumidity = (Byte_t)hum;

    fLV->fPowerSupplyA.fVoltagePos5V         = 0.01*vap5;
    fLV->fPowerSupplyA.fVoltagePos12V        = 0.01*vap12;
    fLV->fPowerSupplyA.fVoltageNeg12V        = 0.01*van12;
    fLV->fPowerSupplyA.fVoltageOptLinkPos12V = 0.01*valp12;
    fLV->fPowerSupplyA.fCurrentPos5V         = 0.001*cap5;
    fLV->fPowerSupplyA.fCurrentPos12V        = 0.001*cap12;
    fLV->fPowerSupplyA.fCurrentNeg12V        = 0.001*can12;
    fLV->fPowerSupplyA.fCurrentOptLinkPos12V = 0.001*calp12;

    fLV->fPowerSupplyB.fVoltagePos5V         = 0.01*vbp5;
    fLV->fPowerSupplyB.fVoltagePos12V        = 0.01*vbp12;
    fLV->fPowerSupplyB.fVoltageNeg12V        = 0.01*vbn12;
    fLV->fPowerSupplyB.fVoltageOptLinkPos12V = 0.01*vblp12;
    fLV->fPowerSupplyB.fCurrentPos5V         = 0.001*cbp5;
    fLV->fPowerSupplyB.fCurrentPos12V        = 0.001*cbp12;
    fLV->fPowerSupplyB.fCurrentNeg12V        = 0.001*cbn12;
    fLV->fPowerSupplyB.fCurrentOptLinkPos12V = 0.001*cblp12;

    str.Remove(0, len);
    str=str.Strip(TString::kLeading);
    return kTRUE;
}

Bool_t MReportCamera::InterpreteAUX(TString &str)
{
    if (!CheckTag(str, "AUX "))
        return kFALSE;

    Int_t len;
    Short_t led, fan;
    const Int_t n=sscanf(str.Data(), "%hd %hd %n", &led, &fan, &len);
    if (n!=2)
    {
        *fLog << err << "ERROR - Reading information of 'AUX' section." << endl;
        return kFALSE;
    }

    fAUX->fRequestCaosLEDs=(Bool_t)led;
    fAUX->fRequestFansFADC=(Bool_t)fan;

    str.Remove(0, len);
    str=str.Strip(TString::kLeading);
    return kTRUE;
}

Bool_t MReportCamera::InterpreteCAL(TString &str)
{
    if (!CheckTag(str, "CAL "))
        return kFALSE;

    Int_t len;
    Short_t hv, lv, cont, pin;

    const Int_t n=sscanf(str.Data(), "%hd %hd %hd %hd %n", &hv, &lv, &cont, &pin, &len);
    if (n!=4)
    {
        *fLog << err << "ERROR - Reading information of 'CAL' section." << endl;
        return kFALSE;
    }

    fCalibration->fRequestHiVoltage = (Bool_t)hv;
    fCalibration->fRequestLoVoltage = (Bool_t)lv;
    fCalibration->fRequestContLight = (Bool_t)cont;
    fCalibration->fRequestPinDiode  = (Bool_t)pin;

    str.Remove(0, len);
    str=str.Strip(TString::kBoth);
    return kTRUE;
}

Bool_t MReportCamera::InterpreteCamera(TString &str)
{
    //
    // I have tried to do it with pure pointer arithmentics, but most of the time is spent
    // to do the sscanf. So we gain less than 5% not using TString like it is done here.
    Int_t len;
    Short_t cal, stat, hvps, lid, lv, cool, hv, dc, led, fan, can, io, clv;
    Int_t n=sscanf(str.Data(), " %hd %hd %hd %hd %hd %hd %hd %hd %hd %hd %hd %hd %hd %n",
                   &cal, &stat, &hvps, &lid, &lv, &cool, &hv,
                   &dc, &led, &fan, &can, &io, &clv, &len);
    if (n!=13)
    {
        *fLog << err << "ERROR - Cannot interprete status' of subsystems." << endl;
        return kFALSE;
    }
    str.Remove(0, len);
    str=str.Strip(TString::kLeading);

    fHV->fStatus                   = (Byte_t)hvps;
    fLids->fStatus                 = (Byte_t)lid;
    fLV->fStatus                   = (Byte_t)lv;
    fCooling->fStatus              = (Byte_t)cool;
    fHV->fStatusRamping            = (Byte_t)hv;
    fAUX->fStatusCaosLEDs          = (Bool_t)led;
    fAUX->fStatusFansFADC          = (Bool_t)fan;
    fCalibration->fStatus          = (Bool_t)cal;
    fCalibration->fStatusCANbus    = (Bool_t)can;
    fCalibration->fStatusIO        = (Bool_t)io;
    fCalibration->fStatusLoVoltage = (Bool_t)clv;
    fStatus                        = (Byte_t)stat;
    fStatusDC                      = (Byte_t)dc;

    return kTRUE;
}

// CAMERA-REPORT 06 2003 11 04 23 53 34 438 00 0000 00 00 00 00 00 000
// 0 1 4 4 5 5 4 5 4 9 0 2 3 DC 00000025001E000000070000000F00250025001E0016000000000007000700000007000000070025002D002D001E001E001E0000000000000000000F0007000700000000000F0007000F002D0034002D0034001E001E001E001E00000000000000000007000F000F000F00070000000000000007000F000F0007003400340034003400340016002D0016001E001E0007000F000000070201001E000F000F000F000F00070007000701DB00000007000F00070007000F002D002D0034002D00340025001E001E001E002D001E001E000700000007000000000000000F0016000F000F000F000F00000000000000000007000000070007000F0416000F020B002D0034002D002D002D0034002D002D001E00250025002D001E0025000700070000000F0007000F0000000F000F0007000F000F00160007000000070007000F0000000700070007000F00070007000F000700070025002D0025002D0025002500250034001E001E001E002500250025002D002500070016000F00070016000F000F0007000700070007000F000F00070007000F00000007000F000000000000000000000007000700070007000F0000000F000F002D002D0025002500250025002D0025002D001E001E00250025001E0025001E002D001E00070000000700070007000700070000000700070007000F000000070000000F000700070000000000070000000000070007000F00000007000700000007000700070007000F0007002D0025002D0034002D002500250025002D0025001E001E001E001E001E0025001E001E001E00250000000000000000000000000000000000000007000F0000000F000F00070000000F0007000F00000007000F0007000000000007000000070000000F0007000700070007000000070007000700000000002D0034002D003400340034002D0034003400340034002D001E0025001E0025001E001E001E002D001E001E0000000700000007000700000000000F000700000000000F000F000F0016000F0016000F0007000700070007000000070000000F000F00070000000F0007000F000700070007000F0007001600070000000700070007000F002D003400340034002D0034001E002D002D002500250025000F000F000000070007000F001E001E0016001E001E001E00160016000F0016001600160016000F000F000F000F0007002D00250034002D0034002D00340025002D001E002D001E002D0025000F000F000F0007000700000007001E001E001E0016001E001E001E000F000F0016000F000F000F000F0000000700070016000F000F000F002D00250025002D00340025002D002D001E002D002D002D0025002D0025002D0007025B00070007000F000000070007001E001E00160016001E0016001E001600160007000F0000000F000F0007000F00070000000F000700070000000700070025002D0025002D002500340025002D0025001E0016001E001E0025001E002D001E0025000000070007000000070000000700070000001E001E001E001E00160016001600160016022E0016000F0016000F000F000F000F0007000000070007000700070000000700070007 HV 00006906806B06B06C06906906906806806C06C06B06B06C06C06906906906906906806806806B06C06C06B06B06B06C06C06C06906906906906906906906806806806806B06C06C06C06A06B06A06B06C00106C06C06806906906906906906906906906806806806806806C06B06B06C06B06A06A06A06A06B06C06C06B06C06C06906906906906906906906906806906906806906806806806806C06C06C06B06B06C06A06B06A06A06A06B06B06C06C06B06C06C06906A06806900006906906906906906906906906806806806806806806806C06C06C06B06B06B06C06A06A06A06A06A06A06A06B06B06B06B06B06B06C05C06906906906906906806906906906906906906906906806806806806806806806806C06B06C06C06C06C06B06C06A06A06A06A06906A06A06A06B06B06B06B06B06B06B06B06906906906906906906906906906906906906906906906906906806806806806806806806806806B06C06B06B06B06B06B06B06B06A06A06906906A06A06A06A06906B06C06B06B06B08906B06B06B06906906906906906906906906906806806906906906906806906906906806806806806806806806806806806C06B06C06C06B06C06B06B06B06B06906906A06A06A06A06A06906906A06B06B06B06B06B06B06B06B06B06B06806906906906906906906906906906806906906906906906906806906906906806806806806806806806806806806806B06B06B06B06B06B06C06A06C06B06C06A06A06A06A06A06A06A06A06A06A06A06B06B09B06B06B06B06B06B06B06B06B06906906906906906906906906906906806906906906906A06906806806806806906806B06B06B06B06A06B06A06A06906A06B06A08F06B06C06B06B06B06906906906906906906906906906906906906906806806806806806806806B06B06B06B06B06B06B06906A06A06A06A06A06A06B06B06B06B06B06B06B06906906906906906906906906906906906906906906906806806806806806806806806B06B06B06B06B06B06B06B06906A06906906A06906A06A06B06B06B06B06B06B06B06B06906906906906906906A06906906906906906906906906906906806806806806806806806806806B06B06B06B06B06B06B06B06B06A06906906A06906906A06906A06B06B06B06B06B06B07306B06B069069069069069069069069069 COOL 219 228 250 110 038 032 0 0 1 1 0 0 0 LID 0 1 0 1 0 0 0 0 2 2 0 0 HVPS 0105 0113 000 000 LV 00556 01120 -1190 00528 01080 -1209 01072 01064 03825 07350 02880 04779 06650 02898 09750 09150 1 204 031 AUX 0 0 CAL 0 1 0 0 OVER
Bool_t MReportCamera::InterpreteBody(TString &str)
{
    if (!InterpreteCamera(str))
        return kFALSE;

    if (!InterpreteDC(str))
        return kFALSE;

    if (!InterpreteHV(str))
        return kFALSE;

    if (!InterpreteCOOL(str))
        return kFALSE;

    if (!InterpreteLID(str))
        return kFALSE;

    if (!InterpreteHVPS(str))
        return kFALSE;

    if (!InterpreteLV(str))
        return kFALSE;

    if (!InterpreteAUX(str))
        return kFALSE;

    if (!InterpreteCAL(str))
        return kFALSE;

    if (str!="OVER")
    {
        *fLog << err << "ERROR - 'OVER' tag not found." << endl;
        return kFALSE;
    }

    return kTRUE;
}
