/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug   11/2003 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// MCalibrationPix                                                            //
//                                                                         //
// This is the storage container to hold informations about the pedestal   //
// (offset) value of one Pixel (PMT).                                      //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
#include "MCalibrationPix.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MCalibrationPix);

using namespace std;

static const TString gsDefHTitle = "Calibration Histograms Pixel ";
// --------------------------------------------------------------------------
//
// Default Constructor. 
//
MCalibrationPix::MCalibrationPix(Int_t pix, const char *name, const char *title)
    : fPixId(pix)
{

  fName  = name  ? name  : "MCalibrationPixel";
  fTitle = title ? title : "Container of the MHCalibrationPixels and the fit results";

  fHist = new MHCalibrationPixel(fPixId,"MHCalibrationPixel",gsDefHTitle.Data()+fPixId);

  fQ   = fErrQ     = 0.;
  fPed = fPedRms   = 0.;
  fT   = fSigmaT  = 0.;
  fRQ  = fErrRQ = 0.;
  fSigmaQ = fErrSigmaQ = 0.;
  fQProb  = 0.;

}

MCalibrationPix::~MCalibrationPix() 
{
  delete fHist;
}

// ------------------------------------------------------------------------
//
// Invalidate values
//
void MCalibrationPix::Clear(Option_t *o)
{
  fHist->Reset();
}

Bool_t MCalibrationPix::FitQ() 
{

  if (fHist->IsFitted())
    return kTRUE;

  if (fPed && fPedRms)
    fHist->SetLowerFitRange(fPed + 3.5*fPedRms);
  else
    *fLog << warn << "Cannot set lower fit range to suppress cosmics: Pedestals not available" << endl;

  if(!fHist->FitQ())
    {
      *fLog << warn << "Could not fit charges of pixel " << fPixId << endl;
      fHist->PrintQFitResult();
      return kFALSE;
    }
  
  fQ         = fHist->GetQMean();
  fErrQ      = fHist->GetQMeanErr(); 
  fSigmaQ    = fHist->GetQSigma();
  fErrSigmaQ = fHist->GetQSigmaErr();
  fQProb     = fHist->GetQProb();

  if (fPed)
    fRQ      = fQ - fPed;
  if (fPedRms)
    fErrRQ   = TMath::Sqrt(fErrQ*fErrQ + fPedRms*fPedRms);
  

  return kTRUE;
  
}

void MCalibrationPix::SetPedestal(Float_t ped, Float_t pedrms)
{

  fPed    = ped;    
  fPedRms = pedrms;
  
  if ((!fRQ) && fQ) 
    fRQ = fQ - fPed;
  if ((!fErrRQ) && fErrQ)
    fErrRQ   = TMath::Sqrt(fErrQ*fErrQ + fPedRms*fPedRms);

}

Bool_t MCalibrationPix::FitT() 
{

  if(!fHist->FitT())
    {
      *fLog << warn << "Could not fit times of pixel " << fPixId << endl;
      fHist->PrintTFitResult();
      return kFALSE;
    }

  fT       = fHist->GetTMean();
  fSigmaT  = fHist->GetTSigma();
  fTProb   = fHist->GetTProb();

  return kTRUE;
  
}

void MCalibrationPix::Test()
{
  *fLog << "TEST: pixid: " << fPixId << endl;  
}
