/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 11/2003 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
//  MReportDrive
//
// This is the class interpreting and storing the DRIVE-REPORT information.
//
// This is NOT the place to get the pointing position from. The class
// definition might change. But it is the place to calculate the
// correct pointing position from.
//
//////////////////////////////////////////////////////////////////////////////
#include "MReportDrive.h"

#include "MLogManip.h"

#include "MAstro.h"

ClassImp(MReportDrive);

using namespace std;

MReportDrive::MReportDrive() : MReport("DRIVE-REPORT")
{
    fName = "MReportDrive";
}

Bool_t MReportDrive::InterpreteBody(TString &str)
{
    MAstro::String2Angle(str, fRa);
    MAstro::String2Angle(str, fDec);
    MAstro::String2Angle(str, fHa);

    Int_t len;
    Int_t n=sscanf(str.Data(), "%lf %n", &fMjd, &len);
    if (n!=1)
        return kFALSE;

    str.Remove(0, len);

    MAstro::String2Angle(str, fNominalZd);
    MAstro::String2Angle(str, fNominalAz);
    MAstro::String2Angle(str, fCurrentZd);
    MAstro::String2Angle(str, fCurrentAz);

    n=sscanf(str.Data(), "%lf %lf %n", &fErrorZd, &fErrorAz, &len);
    if (n!=2)
        return kFALSE;

    str.Remove(0, len);

    str = str.Strip(TString::kBoth);

    return str.IsNull();
}

Double_t MReportDrive::GetAbsError() const
{
    const Double_t pzd = fNominalZd*TMath::DegToRad();
    const Double_t azd = fErrorZd  *TMath::DegToRad();
    const Double_t aaz = fErrorAz  *TMath::DegToRad();

    const double el = TMath::Pi()/2-pzd;

    const double dphi2 = aaz/2.;
    const double cos2  = cos(dphi2)*cos(dphi2);
    const double sin2  = sin(dphi2)*sin(dphi2);
    const double d     = cos(azd)*cos2 - cos(2*el)*sin2;

    //
    // Original:
    //   cos(Zd1)*cos(Zd2)+sin(Zd1)*sin(Zd2)*cos(dAz)
    //
    // Correct:
    //   const double d = cos(azd)*cos2 - cos(el1+el2)*sin2;
    //
    // Estimated:
    //   const double d = cos(azd)*cos2 - cos(2*el)*sin2;
    //

    return acos(d)*TMath::RadToDeg();
}
