#include "MDriveCom.h"

#include <iostream>

#include "coord.h"
#include "Slalib.h"
#include "MCosy.h"

using namespace std;

bool MDriveCom::ReadAngle(TString &str, Double_t &ret)
{
    Char_t  sgn;
    Int_t   d, len;
    UInt_t  m;
    Float_t s;

    // Skip whitespaces before %c and after %f
    int n=sscanf(str.Data(), " %c %d %d %f %n", &sgn, &d, &m, &s, &len);

    if (n!=4 || (sgn!='+' && sgn!='-'))
        return false;

    str.Remove(0, len);

    ret = Slalib::Dms2Deg(d, m, s, sgn);
    return true;
}

bool MDriveCom::ReadPosition(TString &str, Double_t &d1, Double_t &d2)
{
    if (!ReadAngle(str, d1))
        return false;

    if (!ReadAngle(str, d2))
        return false;

    return true;
}

bool MDriveCom::CommandRADEC(TString &str)
{
    Double_t ra, dec;
    if (!ReadPosition(str, ra, dec))
    {
        cout << "ERROR - Reading position from RADEC" << endl;
        return false;
    }
    if (!str.IsNull())
    {
        cout << "ERROR - Too many bytes in command RADEC" << endl;
        return false;
    }

    lout << "CC-COMMAND: Track " << ra << "h " << dec << "deg '" << str 
<< "'" << endl;

    ra *= 15; // h -> deg

    RaDec rd[2] = { RaDec(ra, dec), RaDec(ra, dec) };

    cout << "MDriveCom - TRACK... start." << endl;
    fQueue->PostMsg(WM_TRACK, &rd, sizeof(rd));
    cout << "MDriveCom - TRACK... done." << endl;
    return true;
}

bool MDriveCom::CommandZDAZ(TString &str)
{
    Double_t zd, az;
    if (!ReadPosition(str, zd, az))
    {
        cout << "ERROR - Reading position from ZDAZ" << endl;
        return false;
    }

    if (!str.IsNull())
    {
        cout << "ERROR - Too many bytes in command ZDAZ" << endl;
        return false;
    }

    lout << "CC-COMMAND: Move " << zd << "deg " << az << "deg" << endl;

    ZdAz za(zd, az);

    cout << "MDriveCom - POSITION... start." << endl;
    fQueue->PostMsg(WM_POSITION, &za, sizeof(za));
    cout << "MDriveCom - POSITION... done." << endl;
    return true;
}

bool MDriveCom::InterpreteCmd(TString cmd, TString str)
{
    if (cmd==(TString)"WAIT" && str.IsNull())
    {
        cout << "MDriveCom - WAIT... start." << endl;
        fQueue->PostMsg(WM_WAIT);
        cout << "MDriveCom - WAIT... done." << endl;
        return true;
    }

    if (cmd==(TString)"STOP!" && str.IsNull())
    {
        cout << "MDriveCom - STOP!... start." << endl;
        lout << "CC-COMMAND: STOP!" << endl;       
        fQueue->PostMsg(WM_STOP);
        cout << "MDriveCom - STOP!... done." << endl;
        return true;
    }

    if (cmd==(TString)"RADEC")
        return CommandRADEC(str);

    if (cmd==(TString)"ZDAZ")
        return CommandZDAZ(str);

    if (cmd==(TString)"PREPS")
    {
        cout << "Prepos: " << str << endl;
        return true;
    }

    if (cmd.IsNull() && str.IsNull())
    {
        cout << "Empty command (single '\\n') received." << endl;
        return false;
    }

    cout << "Unknown Command: '" << cmd << "':'" << str << "'" << endl;
    return false;
}

void MDriveCom::Print(TString &str, Double_t deg) const
{
    Char_t sgn;
    UShort_t d, m, s;

    Slalib::Deg2Dms(deg, sgn, d, m, s);

    str += Form("%c %03d %02d %03d ", sgn, d, m, s);
}

bool MDriveCom::SendReport(UInt_t stat, RaDec rd, ZdAz so, ZdAz is, ZdAz er)
{
    // so [rad]
    // is [deg]
    // er [rad]

    so *= kRad2Deg;
    er *= kRad2Deg;

    // Set status flag
    if (stat&kError)
        SetStatus(0);
    if (stat&kStopped)
        SetStatus(1);
    if (stat&kStopping || stat&kMoving)
        SetStatus(3);
    if (stat&kTracking)
        SetStatus(4);

    Timer t;
    t.Now();

    TString str;
    Print(str, rd.Ra());    // Ra
    Print(str, rd.Dec());   // Dec
    Print(str, 0);          // HA
    str += Form("%12.6f ", t.GetMjd()); // mjd
    Print(str, so.Zd());
    Print(str, so.Az());
    Print(str, is.Zd());
    Print(str, is.Az());
    str += Form("%08.3f ", er.Zd());
    str += Form("%08.3f", er.Az());

    return Send(str);
}
