/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 11/2003 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//                                                                          
//  MHEvent
//                                                                          
//  Display a single event in a canvas with as many informations as
//  possible, such as event number, run number, ...
//
//  WARNING: This class is not yet ready!
//
//  Input Containers:
//   MGeomCam
//   [MTaskList]
//   [MImgCleanStd]
//   [MRawEvtData]
//   [MRawRunHeader]
//   [MRawEvtHeader]
//
//  Output Containers:
//   -/-
//
//////////////////////////////////////////////////////////////////////////////
#include "MHEvent.h"

#include <TStyle.h>
#include <TCanvas.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MMcEvt.hxx"

#include "MHCamera.h"
#include "MParList.h"
#include "MTaskList.h"
#include "MPedestalCam.h"
#include "MPedestalPix.h"
#include "MParList.h"
#include "MCerPhotEvt.h"
#include "MRawEvtHeader.h"
#include "MRawRunHeader.h"
#include "MRawEvtData.h"
#include "MImgCleanStd.h"

ClassImp(MHEvent);

using namespace std;

// --------------------------------------------------------------------------
//
MHEvent::MHEvent(EventType_t type) : fHist(NULL), fType(type)
{
    fName = "MHEvent";
    fTitle = "Single Event display task";

    fClone = new MRawEvtData("MHEventData");
}

// --------------------------------------------------------------------------
//
MHEvent::~MHEvent()
{
    if (fHist)
        delete fHist;

    delete fClone;
}

Bool_t MHEvent::SetupFill(const MParList *plist)
{
    MTaskList *tlist = (MTaskList*)plist->FindObject("MTaskList");

    fImgCleanStd = tlist ? (MImgCleanStd*)tlist->FindObject("MImgCleanStd") : NULL;
    fMcEvt       = (MMcEvt*)plist->FindObject("MMcEvt");

    fRawEvtData = (MRawEvtData*)plist->FindObject("MRawEvtData");
    if (!fRawEvtData)
        *fLog << warn << "MRawEvtData not found..." << endl;

    fRawRunHeader = (MRawRunHeader*)plist->FindObject("MRawRunHeader");
    if (!fRawRunHeader)
        *fLog << warn << dbginf << "MRawRunHeader not found..." << endl;

    fRawEvtHeader = (MRawEvtHeader*)plist->FindObject("MRawEvtHeader");
    if (!fRawEvtHeader)
        *fLog << warn << dbginf << "MRawEvtHeader not found..." << endl;

    MGeomCam *cam = (MGeomCam*)plist->FindObject("MGeomCam");
    if (!cam)
    {
        *fLog << err << GetDescriptor() << ": No MGeomCam found... aborting." << endl;
        return kFALSE;
    }

    if (fHist)
        delete (fHist);

    fHist = new MHCamera(*cam);
    fHist->AddNotify(*fClone);

    switch (fType)
    {
    case kEvtSignal:
        fHist->SetName("Signal");
        fHist->SetYTitle("S [au]");
        break;
    case kEvtPedestal:
        fHist->SetName("Pedestal");
        fHist->SetYTitle("P [au]");
        break;
    case kEvtPedestalRMS:
        fHist->SetName("Pedestal RMS");
        fHist->SetYTitle("\\sigma_{P} [au]");
        break;
    case kEvtRelativeSignal:
        fHist->SetName("Signal/PedRMS");
        fHist->SetYTitle("S/P_{rms}");
        break;
    case kEvtCleaningLevels:
        if (!fImgCleanStd)
        {
            *fLog << err << "MImgCleanStd not found... aborting." << endl;
            return kFALSE;
        }
        fHist->SetName("CleanLevels");
        fHist->SetYTitle("L");
        break;
     case kEvtArrTime:
        fHist->SetName("Arrival Time");
        fHist->SetYTitle("t [slice unit]");
        break;
    }

    return kTRUE;
}

Bool_t MHEvent::Fill(const MParContainer *par, const Stat_t weight)
{
    if (fHist->IsFreezed())
        return kTRUE;

    if (!par)
        return kFALSE;

    if (fRawEvtData)
        fRawEvtData->Copy(*fClone);

    switch (fType)
    {
    case kEvtSignal:
    case kEvtArrTime:
    case kEvtPedestal:
        fHist->SetCamContent(*(MCamEvent*)par, 0);
        break;
    case kEvtPedestalRMS:
        fHist->SetCamContent(*(MCamEvent*)par, 0);
        break;
    case kEvtRelativeSignal:
        fHist->SetCamContent(*(MCamEvent*)par, 0);
        break;
    case kEvtCleaningLevels:
        {
            TArrayF lvl(2);
            lvl[0] = fImgCleanStd->GetCleanLvl2();
            lvl[1] = fImgCleanStd->GetCleanLvl1();
            fHist->SetCamContent(*(MCamEvent*)par, 0);
            fHist->SetLevels(lvl);
        }
        break;
    }

    TString s;
    if (fRawEvtHeader)
    {
        s += "Event #";
        s += fRawEvtHeader->GetDAQEvtNumber();
    }
    if (fRawEvtHeader && fRawRunHeader)
        s += " of ";

    if (fRawRunHeader)
    {
        s += "Run #";
        s += fRawRunHeader->GetRunNumber();
    }

    if (fMcEvt)
    {
        TString txt("#splitline{");

        switch (fMcEvt->GetPartId())
        {
        case kGAMMA:
            txt += "Gamma: ";
            break;
        case kPROTON:
            txt += "Proton: ";
            break;
        case kHELIUM:
            txt += "Helium: ";
            break;
        default:
            s += "Particle Id#";
            s += fMcEvt->GetPartId();
            s += ": ";
        }

        s.Insert(0, txt);

        s += "}{  E=";

        if (fMcEvt->GetEnergy()>1000)
            s += Form("%.1fTeV", fMcEvt->GetEnergy()/1000);
        else
            if (fMcEvt->GetEnergy()>10)
                s += Form("%dGeV", (Int_t)(fMcEvt->GetEnergy()+.5));
            else
                if (fMcEvt->GetEnergy()>1)
                    s += Form("%.1fGeV", fMcEvt->GetEnergy());
                else
                    s += Form("%dMeV", (Int_t)(fMcEvt->GetEnergy()*1000+.5));

        s += " r=";
        s += (int)(fMcEvt->GetImpact()/100+.5);
        s += "m Zd=";
        s += (int)(fMcEvt->GetTheta()*180/TMath::Pi()+.5);
        s += "\\circ ";
        if (fMcEvt->GetPhotElfromShower()>=10000)
            s += Form("%dk", (Int_t)(fMcEvt->GetPhotElfromShower()/1000.+.5));
        else
            if (fMcEvt->GetPhotElfromShower()>=1000)
                s += Form("%.1fk", fMcEvt->GetPhotElfromShower()/1000.);
            else
                s += fMcEvt->GetPhotElfromShower();
        s += "PhEl}";
    }

    gPad=NULL;
    fHist->SetTitle(s);

    return kTRUE;
}

void MHEvent::Draw(Option_t *)
{
    if (!fHist)
    {
        *fLog << warn << "MHEvent::Draw - fHist==NULL not initialized." << endl;
        return;
    }

    if (!gPad)
        MakeDefCanvas(this);
    fHist->Draw();
}
