/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Abelardo Moralejo, 12/2003 <mailto:moralejo@pd.infn.it>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
//  MMcCalibrationCalc
//
//  This task looks for the nformation about FADC pedestals in
//  MMcFadcHeader and translates it to the pedestal mean and rms (in adc counts),
//  and in the conversion factor between photons and ADC counts in
//  MCalibrationCam.
//  Then we create and fill also the MPedPhotCam object containing the pedestal
//  rms in units of photons.
//
//  Input Containers:
//   MMcFadcHeader
//   [MRawRunHeader]
//
//  Output Containers:
//   MCalibrationCam
//   MPedPhotCam
//
/////////////////////////////////////////////////////////////////////////////
#include "MMcCalibrationCalc.h"

#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MCalibrationPix.h"
#include "MCalibrationCam.h"
#include "MExtractedSignalCam.h"
#include "MExtractedSignalPix.h"
#include "MPedPhotCam.h"
#include "MPedPhotPix.h"

#include "MRawRunHeader.h"
#include "MMcFadcHeader.hxx"

ClassImp(MMcCalibrationCalc);

using namespace std;

MMcCalibrationCalc::MMcCalibrationCalc(const char *name, const char *title)
{
    fName  = name  ? name  : "MMcCalibrationCalc";
    fTitle = title ? title : "Write MC pedestals and conversion factors into MCalibration Container";

    fADC2PheInner = 1.;
    fADC2PheOuter = 1.;

}

// --------------------------------------------------------------------------
//
// Check for the run type. Return kTRUE if it is a MC run or if there
// is no MC run header (old camera files) kFALSE in case of a different
// run type
//
Bool_t MMcCalibrationCalc::CheckRunType(MParList *pList) const
{
    const MRawRunHeader *run = (MRawRunHeader*)pList->FindObject("MRawRunHeader");
    if (!run)
    {
        *fLog << warn << dbginf << "Warning - cannot check file type, MRawRunHeader not found." << endl;
        return kTRUE;
    }

    return  run->GetRunType() == kRTMonteCarlo;
}

// --------------------------------------------------------------------------
//
// Make sure, that there is an MCalibrationCam Object in the Parameter List.
//
Int_t MMcCalibrationCalc::PreProcess(MParList *pList)
{
  //
  // If it is no MC file skip this function...
  //
  if (!CheckRunType(pList))
    return kTRUE;

  if ( ! pList->FindCreateObj(AddSerialNumber("MCalibrationCam")))
    {
      *fLog << err << dbginf << "Cannot create MCalibrationCam... aborting." << endl;
      return kFALSE;
    }

  fSignalCam = (MExtractedSignalCam*) pList->FindObject(AddSerialNumber("MExtractedSignalCam"));
  if ( ! fSignalCam)
    {
      *fLog << err << dbginf << "Cannot find MExtractedSignalCam... aborting." << endl;
      return kFALSE;
    }

  return kTRUE;

}

// --------------------------------------------------------------------------
//
// Check for the runtype.
// Search for MCalibrationCam and MMcFadcHeader.
//
Bool_t MMcCalibrationCalc::ReInit(MParList *pList)
{
    //
    // If it is no MC file skip this function...
    //
    if (!CheckRunType(pList))
        return kTRUE;

    //
    // Now check the existence of all necessary containers. This has
    // to be done only if this is a MC file.
    //

    fHeaderFadc = (MMcFadcHeader*)pList->FindObject(AddSerialNumber("MMcFadcHeader"));
    if (!fHeaderFadc)
    {
        *fLog << err << dbginf << "MMcFadcHeader not found... aborting." << endl;
        return kFALSE;
    }

    fCalCam = (MCalibrationCam*)pList->FindObject(AddSerialNumber("MCalibrationCam"));

    if (!fCalCam)
    {
        *fLog << err << dbginf << "Could not create MCalibrationCam... aborting. " << endl;
        return kFALSE;
    }

    return kTRUE;
}


// --------------------------------------------------------------------------
//
// Fill the MCalibrationCam object
//
Int_t MMcCalibrationCalc::Process()
{
    const int num = fCalCam->GetSize();

    for (int i=0; i<num; i++)
    {
        MCalibrationPix &calpix = (*fCalCam)[i];
	MExtractedSignalPix &sigpix = (*fSignalCam)[i];

        const Float_t pedestmean = sigpix.IsLoGainUsed()? 
	  fSignalCam->GetNumUsedLoGainFADCSlices()*fHeaderFadc->GetPedestal(i) : 
	  fSignalCam->GetNumUsedHiGainFADCSlices()*fHeaderFadc->GetPedestal(i);

        const Float_t pedestrms  = sigpix.IsLoGainUsed()? 
	  sqrt((Double_t)(fSignalCam->GetNumUsedLoGainFADCSlices())) * fHeaderFadc->GetPedestalRmsLow(i) : 
	  sqrt((Double_t)(fSignalCam->GetNumUsedHiGainFADCSlices())) * fHeaderFadc->GetPedestalRmsHigh(i);

	// ped mean and rms per pixel, according to signal calculation:

        calpix.SetPedestal(pedestmean, pedestrms);

	if (i < 397) // FIXME!
	  calpix.SetConversionBlindPixelMethod(fADC2PheInner, 0., 0.);
	else
	  calpix.SetConversionBlindPixelMethod(fADC2PheOuter, 0., 0.);

    }

    return kTRUE;
}
