/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug  09/2003 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//   MCalibrate
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "MCalibrate.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"
#include "MH.h"

#include "MRawRunHeader.h"
#include "MRawEvtData.h"       // MRawEvtData::GetNumPixels
#include "MRawEvtPixelIter.h"

#include "MCalibrationCam.h"
#include "MCalibrationPix.h"

#include "MExtractedSignalCam.h"
#include "MExtractedSignalPix.h"

#include "MCerPhotEvt.h"

#include "MTime.h"
#include "TMath.h"

ClassImp(MCalibrate);

using namespace std;
// --------------------------------------------------------------------------
//
// Default constructor. b is the number of slices before the maximum slice,
// a the number of slices behind the maximum slice which is taken as signal.
//
MCalibrate::MCalibrate(const char *name, const char *title)
{

    fName  = name  ? name  : "MCalibrate";
    fTitle = title ? title : "Task to calculate the number of photons in one event";

}

// --------------------------------------------------------------------------
//
// The PreProcess searches for the following input containers:
//  - MCalibrationCam
//  - MExtractedSignalCam
//
// The following output containers are also searched and created if
// they were not found:
//
//  - MCerPhotEvt
//  - MPedPhotCam
//
Int_t MCalibrate::PreProcess(MParList *pList)
{

    fCalibrations = (MCalibrationCam*)pList->FindObject("MCalibrationCam");
    if (!fCalibrations)
      {
        *fLog << err << dbginf << "MCalibrationCam could not be created ... aborting." << endl;        
        return kFALSE;
      }

    fSignals = (MExtractedSignalCam*)pList->FindObject("MExtractedSignalCam");
    if (!fSignals)
      {
        *fLog << err << dbginf << "Cannot find MExtractedSignalCam ... aborting" << endl;
        return kFALSE;
      }

    fCerPhotEvt = (MCerPhotEvt*)pList->FindCreateObj("MCerPhotEvt");
    if (!fCerPhotEvt)
      {
        *fLog << err << dbginf << "Cannot find  MCerPhotEvt ... aborting" << endl;
        return kFALSE;
      }
    
    return kTRUE;
}


// --------------------------------------------------------------------------
//
//
Int_t MCalibrate::Process()
{

  for (UInt_t pixid = 0; pixid<577; pixid++)
    {

      MExtractedSignalPix &sig =  (*fSignals)[pixid];

      Bool_t logain = sig.IsLoGainUsed();

      Float_t signal;
      if (logain)
	signal = sig.GetExtractedSignalLoGain();
      else
	signal = sig.GetExtractedSignalHiGain();

      MCalibrationPix &pix = (*fCalibrations)[pixid];

      //      Float_t fCalibraitonConvertionFactor = pix.GetMeanConversionFFactorMethod();
      Float_t fCalibrationConvertionFactor = pix.GetMeanConversionBlindPixelMethod();
      

      Float_t nphot = signal*fCalibrationConvertionFactor;
      Float_t nphoterr = 0;
      
      cout << "DBG MCalibrate::Process logain " << logain << " signal " << signal << " fCalibrationConvertionFactor " << fCalibrationConvertionFactor << " nphot " << nphot << endl;
 
      cout << "Blind " << pix.GetMeanConversionBlindPixelMethod() << " " 
	 <<  pix.GetErrorConversionBlindPixelMethod()   << " " 
	 <<  pix.GetSigmaConversionBlindPixelMethod()   << " "
	
	   <<  "FFactor " << pix.GetMeanConversionFFactorMethod()       << " " 
   <<  pix.GetErrorConversionFFactorMethod()      << " " 
   <<  pix.GetSigmaConversionFFactorMethod()      << " " 
   <<  pix.GetPheFFactorMethod()                  << " " 
 
	   <<  "PIN " << pix.GetMeanConversionPINDiodeMethod()      << " " 
   <<  pix.GetErrorConversionPINDiodeMethod()     << " " 
	   <<  pix.GetSigmaConversionPINDiodeMethod()     << endl;

 
      fCerPhotEvt->AddPixel(pixid, nphot, nphoterr);

    }

    fCerPhotEvt->FixSize();
    fCerPhotEvt->SetReadyToSave();

    return kTRUE;
}

