----------------------------------------------------------------------------------
-- Company:        ETH Zurich, Institute for Particle Physics
-- Engineer:       P. Vogler, Q. Weitzel
-- 
-- Create Date:    11:59:40 01/19/2010 
-- Design Name:    
-- Module Name:    FTU_top - Behavioral 
-- Project Name: 
-- Target Devices: 
-- Tool versions: 
-- Description:    Top level entity of FACT FTU board 										
--
-- Dependencies: 
--
-- Revision: 
-- Revision 0.01 - File Created
-- Revision 0.02 - New design of FTU firmware, 12.07.2010, Q. Weitzel
-- Additional Comments: 
--
----------------------------------------------------------------------------------
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;
library ftu_definitions;
USE ftu_definitions.ftu_array_types.all;

---- Uncomment the following library declaration if instantiating
---- any Xilinx primitives in this code.
--library UNISIM;
--use UNISIM.VComponents.all;

entity FTU_top is
  port(    
    -- global control 
    ext_clk   : IN  STD_LOGIC;                      -- external clock from FTU board
    brd_add   : IN  STD_LOGIC_VECTOR(5 downto 0);   -- geographic board/slot address
    brd_id    : IN  STD_LOGIC_VECTOR(7 downto 0);   -- local solder-programmable board ID
    
    -- rate counters LVDS inputs
    -- use IBUFDS differential input buffer
    patch_A_p     : IN  STD_LOGIC;                  -- logic signal from first trigger patch
    patch_A_n     : IN  STD_LOGIC;           
    patch_B_p     : IN  STD_LOGIC;                  -- logic signal from second trigger patch
    patch_B_n     : IN  STD_LOGIC;
    patch_C_p     : IN  STD_LOGIC;                  -- logic signal from third trigger patch
    patch_C_n     : IN  STD_LOGIC;
    patch_D_p     : IN  STD_LOGIC;                  -- logic signal from fourth trigger patch
    patch_D_n     : IN  STD_LOGIC;
    trig_prim_p   : IN  STD_LOGIC;                  -- logic signal from n-out-of-4 circuit
    trig_prim_n   : IN  STD_LOGIC;
    
    -- DAC interface
    sck           : OUT STD_LOGIC;                  -- serial clock to DAC
    mosi          : OUT STD_LOGIC;                  -- serial data to DAC, master-out-slave-in
    clr           : OUT STD_LOGIC;                  -- clear signal to DAC
    cs_ld         : OUT STD_LOGIC;                  -- chip select or load to DAC
    
    -- RS-485 interface to FTM
    rx            : IN  STD_LOGIC;                  -- serial data from FTM
    tx            : OUT STD_LOGIC;                  -- serial data to FTM
    rx_en         : OUT STD_LOGIC;                  -- enable RS-485 receiver
    tx_en         : OUT STD_LOGIC;                  -- enable RS-485 transmitter

    -- analog buffer enable
    enables_A   : OUT STD_LOGIC_VECTOR(8 downto 0);  -- individual enables for analog inputs
    enables_B   : OUT STD_LOGIC_VECTOR(8 downto 0);  -- individual enables for analog inputs
    enables_C   : OUT STD_LOGIC_VECTOR(8 downto 0);  -- individual enables for analog inputs
    enables_D   : OUT STD_LOGIC_VECTOR(8 downto 0);  -- individual enables for analog inputs

    -- testpoints
    TP_A        : OUT STD_LOGIC_VECTOR(11 downto 0)   -- testpoints    
  );
end FTU_top;

architecture Behavioral of FTU_top is

  signal reset_sig   : STD_LOGIC := '0';  -- initialize reset to 0 at power up
  signal dac_clr_sig : STD_LOGIC := '1';  -- initialize dac_clr to 1 at power up
  
  signal config_start_sig : STD_LOGIC := '0';
  signal config_ready_sig : STD_LOGIC := '0';
  
  signal clk_50M_sig   : STD_LOGIC;         -- generated by internal DCM
  signal clk_ready_sig : STD_LOGIC := '0';  -- set high by FTU_clk_gen when DCMs have locked 

  signal ram_wea_sig  : STD_LOGIC_VECTOR(0 downto 0) := "0";  -- RAM write enable for port A
  signal ram_web_sig  : STD_LOGIC_VECTOR(0 downto 0) := "0";  -- RAM write enable for port B
  signal ram_ada_cntr : INTEGER range 0 to 2**5 - 1 := 0;
  signal ram_adb_cntr : INTEGER range 0 to 2**4 - 1 := 0;
  signal ram_ada_sig  : STD_LOGIC_VECTOR(4 downto 0);
  signal ram_adb_sig  : STD_LOGIC_VECTOR(3 downto 0);
  signal ram_dia_sig  : STD_LOGIC_VECTOR(7 downto 0);
  signal ram_dib_sig  : STD_LOGIC_VECTOR(15 downto 0);
  signal ram_doa_sig  : STD_LOGIC_VECTOR(7 downto 0);
  signal ram_dob_sig  : STD_LOGIC_VECTOR(15 downto 0);
    
  type FTU_top_StateType is (IDLE, INIT, RUNNING, RESET);
  signal FTU_top_State, FTU_top_NextState: FTU_top_StateType;

  component FTU_clk_gen
    port (
      clk    : IN  STD_LOGIC;
      rst    : IN  STD_LOGIC;
      clk_50 : OUT STD_LOGIC;
      ready  : OUT STD_LOGIC
    );
  end component;

  component FTU_spi_interface
    port(
      clk_50MHz      : IN     std_logic;
      config_start   : IN     std_logic;
      dac_array      : IN     dac_array_type;
      config_ready   : OUT    std_logic;
      config_started : OUT    std_logic;
      dac_cs         : OUT    std_logic;
      mosi           : OUT    std_logic;
      sclk           : OUT    std_logic
    );
  end component;

  component FTU_dual_port_ram
    port(
      clka  : IN  std_logic;
      ena   : IN  std_logic;
      wea   : IN  std_logic_VECTOR(0 downto 0);
      addra : IN  std_logic_VECTOR(4 downto 0);
      dina  : IN  std_logic_VECTOR(7 downto 0);
      douta : OUT std_logic_VECTOR(7 downto 0);
      clkb  : IN  std_logic;
      enb   : IN  std_logic;
      web   : IN  std_logic_VECTOR(0 downto 0);
      addrb : IN  std_logic_VECTOR(3 downto 0);
      dinb  : IN  std_logic_VECTOR(15 downto 0);
      doutb : OUT std_logic_VECTOR(15 downto 0)
    );
  end component;

  -- Synplicity black box declaration
  attribute syn_black_box : boolean;
  attribute syn_black_box of FTU_dual_port_ram: component is true;
  
begin

  clr <= dac_clr_sig;
  
  Inst_FTU_clk_gen : FTU_clk_gen
    port map (
      clk    => ext_clk,
      rst    => reset_sig,
      clk_50 => clk_50M_sig,
      ready  => clk_ready_sig
    );

  Inst_FTU_spi_interface : FTU_spi_interface
    port map(
      clk_50MHz      => clk_50M_sig,
      config_start   => config_start_sig,
      dac_array      => DEFAULT_DAC,  -- has to come from RAM
      config_ready   => config_ready_sig,
      config_started => open,
      dac_cs         => cs_ld,
      mosi           => mosi,
      sclk           => sck
    );

  Inst_FTU_dual_port_ram : FTU_dual_port_ram
    port map(
      clka  => clk_50M_sig,
      ena   => '1',
      wea   => ram_wea_sig,
      addra => ram_ada_sig,
      dina  => ram_dia_sig,
      douta => ram_doa_sig,
      clkb  => clk_50M_sig,
      enb   => '1',
      web   => ram_web_sig,
      addrb => ram_adb_sig,
      dinb  => ram_dib_sig,
      doutb => ram_dob_sig
    );
  
  --FTU main state machine (two-process implementation)
  --sensitive to external clock

  FTU_top_Registers: process (ext_clk)
  begin
    if Rising_edge(ext_clk) then
      FTU_top_State <= FTU_top_NextState;
    end if;
  end process FTU_top_Registers;

  FTU_top_C_logic: process (FTU_top_State, clk_ready_sig, config_ready_sig, ram_adb_cntr)
  begin
    FTU_top_NextState <= FTU_top_State;
    case FTU_top_State is
      when IDLE =>                      -- wait for DMCs to lock
        reset_sig <= '0';
        dac_clr_sig <= '1';
        config_start_sig <= '0';
        ram_web_sig <= "0";
        if (clk_ready_sig = '1') then
          FTU_top_NextState <= RUNNING;
        end if;          
      when INIT =>                      -- load default config data to RAM 
        reset_sig <= '0';
        dac_clr_sig <= '1';
        config_start_sig <= '0';
        ram_web_sig <= "1";
        ram_adb_cntr <= ram_adb_cntr + 1;
        ram_adb_sig <= conv_std_logic_vector(ram_adb_cntr, 4);        
        if (ram_adb_cntr < 4) then
          ram_dib_sig <= DEFAULT_ENABLE(ram_adb_cntr);
          FTU_top_NextState <= INIT;
        elsif (ram_adb_cntr < 4 + 5) then
          ram_dib_sig <= conv_std_logic_vector(DEFAULT_DAC(ram_adb_cntr - 4), 16);
          FTU_top_NextState <= INIT;
        elsif (ram_adb_cntr < 32) then
          ram_dib_sig <= (others => '0');
          FTU_top_NextState <= INIT;
        else
          ram_adb_cntr <= 0;
          ram_web_sig <= "0";
          FTU_top_NextState <= RUNNING;
        end if;
      when RUNNING =>                   -- count triggers and react to commands
        reset_sig <= '0';
        dac_clr_sig <= '1';
        config_start_sig <= '0';
        ram_web_sig <= "0";
      when RESET =>                     -- reset/clear and start from scratch
        reset_sig <= '1';
        dac_clr_sig <= '0';
        config_start_sig <= '0';
        ram_web_sig <= "0";
        FTU_top_NextState <= IDLE;
    end case;
  end process FTU_top_C_logic;
  
end Behavioral;
