/********************************************************************\

  Bridge between two networks

  Subscription to top-level server list DIS_DNS/SERVER_LIST not via
  AddService() ensures AddService() only called from infoHandler(),
  thus serialized by DIM and no Mutex is necessary.

  Remote procedure calls are bridged through their underlying services
  and commands.
  
  Configuraton changes are automatically tracked through ConfigChanged()
    
  Oliver Grimm, June 2010

\********************************************************************/

#define SERVER_NAME "Bridge"
#include "Evidence.h"

#include <string>
#include <vector>
#include <regex.h>

const int DEFAULT_PORT = 2505;

using namespace std;

// Class declaration
class Bridge: public DimClient, public EvidenceServer {

	struct Item {
	  DimCommand *Command;
	  DimStampedInfo *DataItem;
	  DimService *Service;
	  char *Data;
	};

	map<string, struct Item> Map;
	vector<regex_t> RegEx;
	DimInfo *ServerList;
	
    void infoHandler();
    void commandHandler();
	void AddService(string, char *, int);
	void RemoveService(string);
	void ConfigChanged();
	void BugFix();
	void Undo();

  public:
    Bridge();
    ~Bridge();
}; 

// Constructor (ConfigChanged() is automatically called at start-up)
Bridge::Bridge(): EvidenceServer(SERVER_NAME) {

  // Initialise
  ServerList = NULL;
  GetConfig("cmdallow", " ");
  
  ConfigChanged();
}


// Destructor
Bridge::~Bridge() {

  Undo();
}


// Undo: Delete all subscriptions and regular expressions
void Bridge::Undo() {

  while (Map.size() != 0) RemoveService((*Map.begin()).first);  
  delete ServerList;
  for (int i=0; i<RegEx.size(); i++) regfree(&RegEx[i]);
  RegEx.clear();
}


// Called by signal handler in case configuration changes and by constructor
void Bridge::ConfigChanged() {

  static string ExcludeString;

  // Check if configuratiomn changed
  string NewExcludeString = GetConfig("exclude");

  Lock();
  if (ExcludeString != NewExcludeString) ExcludeString.swap(NewExcludeString);
  Unlock();
  if (ExcludeString == NewExcludeString) return;
    
  // Compile regular expressions
  regex_t R;
  vector<regex_t> RegEx;

  vector<string> Exclude = Tokenize(ExcludeString, " \t");
  for (int i=0; i<Exclude.size(); i++) {
	int Ret = regcomp(&R, Exclude[i].c_str(), REG_EXTENDED|REG_NOSUB);
	if (Ret != 0) {
	  char Err[200];
	  regerror(Ret, &R, Err, sizeof(Err));
	  Message(ERROR, "Error compiling regular expression '%s' (%s)", Exclude[i].c_str(), Err);
	}
	else RegEx.push_back(R);
  }

  // Remove previous data, set new regular expressions and subscribe to top-level server list
  Lock();
  Undo();
  Bridge::RegEx = RegEx;
  ServerList = new DimInfo((char *) "DIS_DNS/SERVER_LIST", NO_LINK, this);
  Unlock();
}


// Service subscription and repeating
void Bridge::infoHandler() {

  DimInfo *I = getInfo();

  // Check if service available
  if (!ServiceOK(I)) return;

  // If service is DIS_DNS/SERVER_LIST, subscribe to all SERVICE_LIST services
  if (strcmp(I->getName(), "DIS_DNS/SERVER_LIST") == 0) {

	char *Token = strtok(I->getString(), "+-!@");	
	while (Token != NULL) {
	  if (*I->getString() == '-' || *I->getString() == '!') {
		RemoveService(string(Token)+"/SERVICE_LIST");
	  }
	  else AddService(string(Token)+"/SERVICE_LIST", (char *) "C", DimSERVICE);

	  // Skip server IP address and process ID
	  Token = strtok(NULL, "|");
	  Token = strtok(NULL, "@");
	}	
	return;
  }

  // If service is SERVICE_LIST, scan and subscribe/unsubscribe to services
  if (strstr(I->getName(), "/SERVICE_LIST") != NULL) {

	// Bug fix for empty SERVICE_LIST
	if (strlen(I->getString()) == 0) {
	  string Tmp(I->getName());
	  RemoveService(I->getName());
	  AddService(Tmp.c_str(), (char *) "C", DimSERVICE);
	  return;
	}

	char *Format, *Name = strtok(I->getString(), "+-!|");
	while (Name != NULL) {
      if ((Format = strtok(NULL, "\n")) != NULL) {
	    // Check if service added or removed/unavailable
	    if (*I->getString() == '-' || *I->getString() == '!') {
		  if (strstr(Format, "|RPC") != NULL) {
	  		RemoveService(string(Name)+"/RpcIn");
	  		RemoveService(string(Name)+"/RpcOut");
		  }
		  else RemoveService(Name);
		}
		else {
		  // Determine type of service
		  if (strstr(Format, "|CMD") != NULL) {
			*(strstr(Format, "|CMD")) = '\0';
			AddService(Name, Format, DimCOMMAND);
		  }
		  else if (strstr(Format, "|RPC") != NULL) {
			*(strstr(Format, "|RPC")) = '\0';
		    if (strchr(Format, ',') != NULL) {
	  		  *strchr(Format, ',') = '\0';
	  		  AddService(string(Name)+"/RpcIn", Format, DimCOMMAND);
	  		  AddService(string(Name)+"/RpcOut", Format+strlen(Format)+1, DimSERVICE);
			}
			
		  }
		  else {
			Format[strlen(Format)-1] = '\0';
			AddService(Name, Format, DimSERVICE);
		  }
		}
	  }
	  Name = strtok(NULL, "|");
	}
	return;
  }

  // Check if service known and repeat to secondary DNS
  if (Map.count(I->getName()) == 0) return;

  // Copy service data
  delete[] Map[I->getName()].Data;
  Map[I->getName()].Data = new char [I->getSize()];
  memcpy(Map[I->getName()].Data, I->getData(), I->getSize());

  // Set new service properties and update service
  Map[I->getName()].Service->setQuality(I->getQuality());
  Map[I->getName()].Service->setTimestamp(I->getTimestamp(), I->getTimestampMillisecs());	
  Map[I->getName()].Service->updateService(Map[I->getName()].Data, I->getSize());
}


// Command repeating (also handles requests for remote procedure calls)
void Bridge::commandHandler() {

  // Check if client allowed to send commands
  vector<string> Client = Tokenize(getClientName(), "@");
  if (Client.size() == 2 && GetConfig("cmdallow").find(Client[1]) == string::npos) {
	Message(INFO, "Rejected command/rpc from %s (ID %d)", getClientName(), getClientId());
	return;  
  } 

  // Send command to server
  sendCommandNB(getCommand()->getName(), getCommand()->getData(), getCommand()->getSize());
}


// Service subscription
void Bridge::AddService(string Name, char *Format, int Type) {

  // Check if already subscribed to this service
  if (Map.count(Name) != 0) return;

  // Should service be ignored?
  for (int i=0; i<RegEx.size(); i++) {
    if (regexec(&RegEx[i], Name.c_str(), (size_t) 0, NULL, 0) == 0) return;
  }

  // Create subscription and service to secondary DNS or new command
  Map[Name].Command = NULL;
  Map[Name].DataItem = NULL;
  Map[Name].Service = NULL;
  Map[Name].Data = NULL;
  
  if  (Type == DimSERVICE) {  
	Map[Name].Service = new DimService(Name.c_str(), (char *) Format, Map[Name].Data, 0);
	Map[Name].DataItem = new DimStampedInfo(Name.c_str(), NO_LINK, this);
  }
  else if (Type == DimCOMMAND) Map[Name].Command = new DimCommand(Name.c_str(), Format, this);
}


// Remove service from watch list (unused pointer are NULL)
void Bridge::RemoveService(string Name) {

  // Check if actually subscribed to service  
  if (Map.count(Name) == 0) return;

  delete Map[Name].DataItem;
  delete Map[Name].Service;
  delete[] Map[Name].Data;
  delete Map[Name].Command;

  Map.erase(Name);
}


// Main program
int main(int argc, char *argv[]) {

  // Check command line argument
  if (argc == 1) {
	printf("Usage: %s <address of primary DNS> [port] (default port %d)\n", argv[0], DEFAULT_PORT);
	exit(EXIT_FAILURE);
  }

  // Set primary DIM network to subscribe from
  DimClient::setDnsNode(argv[1], argc>2 ? atoi(argv[2]) : DEFAULT_PORT);

  // Static ensures calling of destructor by exit()
  static Bridge Class;
  
  // Sleep until signal caught
  while (!Class.ExitRequest) pause();
}
