/* Data organisation on disk:

                                  Board 1      Board 2      ...     Board 1      Board 2      ...
   RH  BS1 BS2 ... EH TC1 TC2 ... C1 C2 C3 ... C1 C2 C3 ... ...  EH C1 C2 C3 ... C1 C2 C3 ... ...
                   --------------------------------  --------------------------------
                   Event 1                           Event 2

  RH Run header   BSx Board structures   EH Event header   TCx Trigger cell of chip x (int)
  Cx Channel (0-19 for 2 chips), Channel data are written as shorts, length of event data is in event header

  Structures are defined using #pragma pack (1) to not include any padding. Note that
  using the gcc attribute __attribute__((__packed__)) is incompatible with root.

  The convention for the header structure is that exisitng structure entries
  should never be deleted. New items may only be added at the end.
*/

#ifndef FORMATRAW_H_
#define FORMATRAW_H_

#include <stdio.h>
//#include <time.h>

#define DATA_FORMAT 1
#define IDENTIFICATION 1    // Raw data file identification

typedef char I8;
typedef int I32;
typedef unsigned int U32;
typedef float F32;

#define MAGICNUM_OPEN 0xe0e1    // Magic number for run header while file open
#define MAGICNUM_CLOSED 0xe0e0  //    ... and when file is closed
#define MAGICNUM_ERROR 0xe0e2   //    ... and when an error occurred

// Error codes
enum CTX_ErrCode {CTX_OK, CTX_FOPEN, CTX_FCLOSE, CTX_NOTOPEN, CTX_RHEADER,
    	    	  CTX_BSTRUCT, CTX_EHEADER, CTX_DATA, CTX_SEEK, CTX_EOF, CTX_VERSION};


#pragma pack (1)  // Switch padding off

// Run header
typedef struct {
  U32 MagicNum;
  U32 DataFormat;       // Increasing whenever format changes

  U32 RunHeaderSize;
  U32 EventHeaderSize;
  U32 BoardStructureSize;

  I32 SoftwareRevision;	// Subversion revision number (negative for modified working copy)

  U32 Identification;
  U32 Type;          	// Run type: 0=data, 1=pedestal, 3=test
  U32 RunNumber;
  U32 FileNumber;
  I8  Description[100];
       
  U32 NBoards;	    	// Number of used mezzanine boards
  U32 NChips;		// Number of DRS chips per board
  U32 NChannels;	// Number of channels per chip

  U32 Samples;          // Number of samples
  U32 Offset;           // Offset from trigger

  U32 Events;           // Number of events in the file
  U32 NBytes;	    	// Bytes per sample
  
  U32 StartSecond;  	// Opening and closing time of the file
  U32 StartMicrosecond;
  U32 EndSecond;
  U32 EndMicrosecond;
} RunHeader;

// Board structure
typedef struct {
  I32 SerialNo;     // Board serial number
  F32 NomFreq;	    // Nominal sampling frequency [GHz]
  F32 BoardTemp;    // Board temperature [deg C]
  F32 ScaleFactor;  // Factor for conversion to mV
} BoardStructure;

// Event header
typedef struct {
  U32 EventNumber;
  U32 Second;          // Event time stamp (result of gettimeofday())
  U32 Microsecond;
  U32 TriggerType;
  U32 EventSize; 	// Size of following data in bytes
} EventHeader;

#pragma pack ()     // Set default padding

// Class definition
class RawDataCTX {
    FILE *Rawfile;
    bool FileOpen;
    bool Silent;    	// No textual output if true
    
  public:
    RunHeader   *RHeader;
    EventHeader *EHeader;
    BoardStructure *BStruct;
    char *Data;
    
    RawDataCTX(bool = false);
    ~RawDataCTX();
    
    CTX_ErrCode OpenDataFile(char*, FILE* = NULL);
    CTX_ErrCode CloseDataFile();
    CTX_ErrCode ReadEvent(unsigned int = 0, FILE* = NULL);
    bool IsFileOpen();
};
#endif
