/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Hengstebeck 3/2003 <mailto:hengsteb@alwa02.physik.uni-siegen.de>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
//  MRanForestCalc
//
//  Calculates the hadroness of an event. It calculates a mean value of all
//  classifications by the trees in a previously grown random forest.
//
//  To use only n trees for your calculation use:
//  MRanForestCalc::SetUseNumTrees(n);
//
////////////////////////////////////////////////////////////////////////////
#include "MRanForestCalc.h"

#include "MHMatrix.h" // must be before MLogManip.h
#include "MDataArray.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"

#include "MRanTree.h"
#include "MRanForest.h"

#include "MHadronness.h"

#include "MEvtLoop.h"
#include "MTaskList.h"
#include "MFillH.h"
#include "MStatusDisplay.h"
#include "MRanForestGrow.h"
#include "MRanForestFill.h"

#include "MWriteRootFile.h"
#include "MReadTree.h"

ClassImp(MRanForestCalc);

using namespace std;

static const TString gsDefName  = "MRanForestCalc";
static const TString gsDefTitle = "Tree Classification Loop 1/2";

// --------------------------------------------------------------------------
//
// Setup histograms and the number of distances which are used for
// avaraging in CalcDist
//
MRanForestCalc::MRanForestCalc(const char *name, const char *title)
    : fNum(100), fHadronnessName("MHadronness"), fData(NULL)
{
    //
    //   set the name and title of this object
    //
    fName  = name  ? name  : gsDefName.Data();
    fTitle = title ? title : gsDefTitle.Data();
}

// --------------------------------------------------------------------------
//
// Delete the data chains
//
MRanForestCalc::~MRanForestCalc()
{
    //    delete fData;
}

// --------------------------------------------------------------------------
//
// Needs:
//  - MatrixGammas  [MHMatrix]
//  - MatrixHadrons [MHMatrix]
//  - MHadronness
//  - all data containers used to build the matrixes
//
// The matrix object can be filles using MFillH. And must be of the same
// number of columns (with the same meaning).
//
Int_t MRanForestCalc::PreProcess(MParList *plist)
{
    fRanForest = (MRanForest*)plist->FindObject("MRanForest");
    if (!fRanForest)
    {
        *fLog << err << dbginf << "MRanForest not found... aborting." << endl;
        return kFALSE;
    }

    fRanTree = (MRanTree*)plist->FindObject("MRanTree");
    if (!fRanTree)
    {
        *fLog << err << dbginf << "MRanTree not found... aborting." << endl;
        return kFALSE;
    }

    /*if(!fRanForest->GetCurTree())
    {
        *fLog << err << dbginf << "MRanForest does not contain trees... aborting." << endl;
        return kFALSE;
    }*/

    fData = fRanTree->GetRules();

    if (!fData)
    {
        *fLog << err << dbginf << "Error matrix doesn't contain columns... aborting." << endl;
        return kFALSE;
    }

    if (!fData->PreProcess(plist))
    {
        *fLog << err << dbginf << "PreProcessing of the MDataArray failed for the columns failed... aborting." << endl;
        return kFALSE;
    }

    fHadroness = (MHadronness*)plist->FindCreateObj("MHadronness", fHadronnessName);
    if (!fHadroness)
        return kFALSE;

    return kTRUE;
}

// --------------------------------------------------------------------------
//
//
Int_t MRanForestCalc::Process()
{
    // first copy the data from the data array to a vector event
    TVector event;
    *fData >> event;

    Double_t hadroness=fRanForest->CalcHadroness(event);
    fHadroness->SetHadronness(hadroness);

    return kTRUE;
}

Bool_t MRanForestCalc::Grow(MHMatrix *matrixg,MHMatrix *matrixh,Int_t ntree,
                            Int_t numtry,Int_t ndsize,const char* treefile=0,
                            const char* treename=0,const char* contname=0,
                            const char* hgininame=0)
{

    treename  = treename  ? treename  : "Tree";
    contname  = contname  ? contname  : "MRanTree";
    hgininame  = hgininame  ? hgininame  : "MHRanForestGini";

    if (!matrixg->IsValid())
    {
        *fLog << err << dbginf << " MRanForestCalc::Grow - ERROR: matrixg not valid." << endl;
        return kFALSE;
    }
    if(!matrixh->IsValid())
    {
        *fLog << err << dbginf << " MRanForestCalc::Grow - ERROR: matrixh not valid." << endl;
        return kFALSE;
    }

    MEvtLoop run(GetName());
    MTaskList tlist;
    MParList plist;
    plist.AddToList(&tlist);
    plist.AddToList(matrixg);
    plist.AddToList(matrixh);

    // creating training task and setting parameters
    MRanForestGrow rfgrow;
    rfgrow.SetNumTrees(ntree); // number of trees
    rfgrow.SetNumTry(numtry);  // number of trials in random split selection
    rfgrow.SetNdSize(ndsize);  // limit for nodesize
    tlist.AddToList(&rfgrow);

    if(treefile){
        MWriteRootFile rfwrite(treefile);
        rfwrite.AddContainer(contname,treename);
        tlist.AddToList(&rfwrite);
    }

    MFillH fillh(hgininame);
    tlist.AddToList(&fillh);

    run.SetParList(&plist);
    
    // Execute tree growing
    if (!run.Eventloop())
    {
        *fLog << err << dbginf << "Evtloop in MRanForestCalc::Grow failed." << endl;
        return kFALSE;
    }
    tlist.PrintStatistics(0, kTRUE);

    if (TestBit(kEnableGraphicalOutput))
        plist.FindObject(hgininame)->DrawClone();

    return kTRUE;
}

Bool_t MRanForestCalc::Fill(Int_t ntree,const char* treefile=0,const char* treename=0)
{
    treefile  = treefile  ? treefile  : "RF.root";
    treename  = treename  ? treename  : "Tree";

    MParList  plist;

    MTaskList tlist;
    plist.AddToList(&tlist);

    MReadTree read(treename,treefile);
    read.DisableAutoScheme();

    MRanForestFill rffill;
    rffill.SetNumTrees(ntree);

    tlist.AddToList(&read);
    tlist.AddToList(&rffill);

    MEvtLoop run(GetName());
    run.SetParList(&plist);

    //
    // Execute tree reading
    //
    if (!run.Eventloop())
    {
        *fLog << err << dbginf << "Evtloop in MRanForestCalc::Fill failed." << endl;
        return kFALSE;
    }
    tlist.PrintStatistics(0, kTRUE);

    return kTRUE;
}
