/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Raquel de los Reyes, 01/2004 <mailto:reyes@gae.ucm.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== *//////////////////////////////////////////////////////////////////////////////
//
// This macro made the check of the central control files (.rep files).
// It displays 5 tabs:
//   - Drive system
//   - Power supplies
//   - Cooling system
//   - Trigger
//   - Weather station (?)
//
////////////////////////////////////////////////////////////////////////////

void CCDataCheck(const TString filename="CC_2003_11_30_20_29_20.root", const TString directory="../rootfiles/2003_12_01/")
{
  //
  // Check of the raw files
  //
  MStatusDisplay *d = new MStatusDisplay;
  d->SetLogStream(&gLog, kTRUE);            // Disables output to stdout

  //
  // Create a empty Parameter List and an empty Task List
  // The tasklist is identified in the eventloop by its name
  //
  MParList  plist;
  
  MTaskList tlist;
  plist.AddToList(&tlist);

  //
  // Now setup the tasks and tasklist:
  // ---------------------------------
  //

  // Set all the MHVsTime classes:
  // Drive system
  MHVsTime hZd("MReportDrive.fNominalZd");
  hZd.SetName("Zd");
  MHVsTime hAz("MReportDrive.fNominalAz");
  hAz.SetName("Az");
  MHVsTime hDState("MReportDrive.fState");
  hDState.SetName("State");
  // Power supplies
  MHVsTime hHVA("MCameraHV.fVoltageA");
  hHVA.SetName("HVA");
  MHVsTime hHVB("MCameraHV.fVoltageB");
  hHVB.SetName("HVB");
  MHVsTime hCA("MCameraHV.fCurrentA");
  hCA.SetName("CA");
  MHVsTime hCB("MCameraHV.fCurrentB");
  hCB.SetName("CB");
  // Cooling system
  MHVsTime hTCenter("MCameraCooling.fTempCenter");
  hTCenter.SetName("TCenter");
  MHVsTime hTWall("MCameraCooling.fTempWall");
  hTWall.SetName("TWall");
  MHVsTime hTWater("MCameraCooling.fTempWater");
  hTWater.SetName("TWater");
  MHVsTime hTOptLink("MCameraCooling.fTempOptLink");
  hTOptLink.SetName("TOptLink");
  MHVsTime hHWall("MCameraCooling.fHumWall");
  hHWall.SetName("HWall");
  MHVsTime hHCenter("MCameraCooling.fHumCenter");
  hHCenter.SetName("HCenter");
  // Weather station
  MHVsTime hCCHum("MReportCC.fHumidity");
  hCCHum.SetName("CCHum");
  MHVsTime hCCTemp("MReportCC.fTemperature");
  hCCTemp.SetName("CCTemp");
  MHVsTime hCCWS("MReportCC.fWindSpeed");
  hCCWS.SetName("CCWS");
  MHVsTime hCCSR("MReportCC.fSolarRadiation");
  hCCSR.SetName("CCSR");
  // Trigger system
  MHVsTime hTrigBL2("MReportTrigger.fL2BeforePrescaler");
  hTrigBL2.SetName("TrigBL2");
  MHVsTime hTrigAL2("MReportTrigger.fL2AfterPrescaler");
  hTrigAL2.SetName("TrigAL2");

  // Reads the trees of the root file and the analysed branches
  MReadReports read;
  read.AddTree("Drive");
  read.AddTree("Camera");
  read.AddTree("CC");
  read.AddTree("Trigger");
  read.AddFile(directory+filename);     // after the reading of the trees!!!
  read.AddToBranchList("MReportDrive.*");
  read.AddToBranchList("MCameraHV.*");
  read.AddToBranchList("MCameraCooling.*");
  read.AddToBranchList("MReportCC.*");
  read.AddToBranchList("MReportTrigger.*");

  // Set of MH3 classes
  MH3 hError("MReportDrive.GetAbsError*60");
  hError.SetName("DeltaH");
  MBinning bins("BinningDeltaH");
  bins.SetEdges(18, 0, 3.6);
  plist.AddToList(&bins);
  MH3 hError2("MReportDrive.fNominalZd","MReportDrive.GetAbsError*60");
  hError2.SetName("DeltaHvsZd");
  MBinning bins2("BinningDeltaHvsZdX");
  MBinning bins3("BinningDeltaHvsZdY");
  bins2.SetEdges(90, 0, 90);
  bins3.SetEdges(18, 0, 3.6);
  plist.AddToList(&bins2);
  plist.AddToList(&bins3);
  MH3 hTempOptLink("MCameraCooling.fTempOptLink");
  hTempOptLink.SetName("TOptLink");
  MBinning bins4("BinningTOptLinkX");
  bins4.SetEdges(50, 0, 50);
  plist.AddToList(&bins4);

  // Fill all the MH classes defined before
  MFillH fillZd(&hZd,             "MTimeDrive");
  MFillH fillAz(&hAz,             "MTimeDrive");
  MFillH fillError(&hError);
  MFillH fillDState(&hDState,     "MTimeDrive");
  MFillH fillError2(&hError2);
  MFillH fillHVA(&hHVA,           "MTimeCamera");
  MFillH fillHVB(&hHVB,           "MTimeCamera");
  MFillH fillCA(&hCA,             "MTimeCamera");
  MFillH fillCB(&hCB,             "MTimeCamera");
  MFillH fillTCenter(&hTCenter,   "MTimeCamera");
  MFillH fillTWall(&hTWall,       "MTimeCamera");
  MFillH fillTWater(&hTWater,     "MTimeCamera");
  MFillH fillTOptLink(&hTOptLink, "MTimeCamera");
  MFillH fillTempOptLink(&hTempOptLink);
  MFillH fillHWall(&hHWall,       "MTimeCamera");
  MFillH fillHCenter(&hHCenter,   "MTimeCamera");
  MFillH fillCCHum(&hCCHum,       "MTimeCC");
  MFillH fillCCTemp(&hCCTemp,     "MTimeCC");
  MFillH fillCCWS(&hCCWS,         "MTimeCC");
  MFillH fillCCSR(&hCCSR,         "MTimeCC");
  MFillH fillTrigBL2(&hTrigBL2,   "MTimeTrigger");
  MFillH fillTrigAL2(&hTrigAL2,   "MTimeTrigger");

  // Do not display the result given by the default draw function
  fillZd.SetBit(MFillH::kDoNotDisplay);
  fillAz.SetBit(MFillH::kDoNotDisplay);
  fillError.SetBit(MFillH::kDoNotDisplay);
  fillDState.SetBit(MFillH::kDoNotDisplay);
  fillError2.SetBit(MFillH::kDoNotDisplay);
  fillHVA.SetBit(MFillH::kDoNotDisplay);
  fillHVB.SetBit(MFillH::kDoNotDisplay);
  fillCA.SetBit(MFillH::kDoNotDisplay);
  fillCB.SetBit(MFillH::kDoNotDisplay);
  fillTCenter.SetBit(MFillH::kDoNotDisplay);
  fillTWall.SetBit(MFillH::kDoNotDisplay);
  fillTWater.SetBit(MFillH::kDoNotDisplay);
  fillTOptLink.SetBit(MFillH::kDoNotDisplay);
  fillTempOptLink.SetBit(MFillH::kDoNotDisplay);
  fillHWall.SetBit(MFillH::kDoNotDisplay);
  fillHCenter.SetBit(MFillH::kDoNotDisplay);
  fillCCHum.SetBit(MFillH::kDoNotDisplay);
  fillCCTemp.SetBit(MFillH::kDoNotDisplay);
  fillCCWS.SetBit(MFillH::kDoNotDisplay);
  fillCCSR.SetBit(MFillH::kDoNotDisplay);
  fillTrigBL2.SetBit(MFillH::kDoNotDisplay);
  fillTrigAL2.SetBit(MFillH::kDoNotDisplay);

  // Add all the task to the task list
  tlist.AddToList(&read);
  tlist.AddToList(&fillZd,       "Drive");
  tlist.AddToList(&fillAz,       "Drive");
  tlist.AddToList(&fillError,    "Drive");
  tlist.AddToList(&fillDState,   "Drive");
  tlist.AddToList(&fillError2,    "Drive");
  tlist.AddToList(&fillHVA,      "Camera");
  tlist.AddToList(&fillHVB,      "Camera");
  tlist.AddToList(&fillCA,       "Camera");
  tlist.AddToList(&fillCB,       "Camera");
  tlist.AddToList(&fillTCenter,  "Camera");
  tlist.AddToList(&fillTWall,    "Camera");
  tlist.AddToList(&fillTWater,   "Camera");
  tlist.AddToList(&fillTOptLink, "Camera");
  tlist.AddToList(&fillTempOptLink, "Camera");
  tlist.AddToList(&fillHWall,    "Camera");
  tlist.AddToList(&fillHCenter,  "Camera");
  tlist.AddToList(&fillCCHum,    "CC");
  tlist.AddToList(&fillCCTemp,   "CC");
  tlist.AddToList(&fillCCWS,     "CC");
  tlist.AddToList(&fillCCSR,     "CC");
  tlist.AddToList(&fillTrigBL2,  "Trigger");
  tlist.AddToList(&fillTrigAL2,  "Trigger");

  //
  // Create and setup the eventloop
  //
  MEvtLoop evtloop;
  evtloop.SetParList(&plist);
  evtloop.SetDisplay(d);
    
  //
  // Execute your analysis
  //
  if (!evtloop.Eventloop())
    return;
 
  tlist.PrintStatistics();

  //
  // Drive report (pointing.C from T. Bretz)
  //
  TCanvas &c1 = d->AddTab("Drive");
  // Plot the telescope tracking positions on the sky
  TGraph *g1 = hZd.GetGraph();
  TGraph *g2 = hAz.GetGraph();
  TPad *p = new TPad("", "",0,0.05,0.6,0.95);
  p->Draw();
  p->cd();
  gPad->SetTheta(-90);
  gPad->SetPhi(90);
  gPad->SetBorderMode(0);
  gStyle->SetOptStat(0);
  TH2F h("pol", "Telescope Tracking Positions on the Sky", 16, 0, 1, 9, 0, 1);
  h.DrawClone("surf1pol");
  gPad->Modified();
  gPad->Update();
  TView *view = gPad->GetView();
  if (!view)
    {
      cout << "No View!" << endl;
      return;
    }
  Double_t *zd=g1->GetY();
  Double_t *az=g2->GetY();
  Double_t old[2] = {0,0};
  for (int i=0; i<g1->GetN(); i++)
    {
      az[i] += 180;
      az[i] *= TMath::Pi()/180;
      Double_t x[3] = { zd[i]*cos(az[i])/90, zd[i]*sin(az[i])/90, 0};
      Double_t y[3];
      view->WCtoNDC(x, y);
      if (old[0]!=0 && old[1]!=1)
        {
	  TLine *l = new TLine(y[0], y[1], old[0], old[1]);
	  l->SetLineColor(kBlue);
	  l->Draw();
        }
      TMarker *m = new TMarker(y[0], y[1], kFullDotMedium);
      m->SetMarkerColor(i==g1->GetN()-1 ? kGreen : kRed);
      m->Draw();
      old[0] = y[0];
      old[1] = y[1];
    }
  // Control deviation of the motors
  c1.cd();
  p = new TPad("", "", 0.6, 0, 1, 0.29);
  p->Draw();
  p->cd();
  gStyle->SetOptStat(1110);
  gStyle->SetStatFormat(".2g");
  MH3 *mh3 = (MH3*)hError.DrawClone("nonew");
  mh3->GetHist()->SetXTitle("\\Delta [arcmin]");
  mh3->GetHist()->SetYTitle("");
  mh3->GetHist()->SetTitle("Control deviation of the motors");
  mh3->GetHist()->SetStats(1);
  TLine ln;
  ln.SetLineColor(kGreen);
  ln.DrawLine(0.5*360*60/16384., 0, 0.5*360*60/16384., hError.GetHist()->GetMaximum());
  ln.SetLineColor(kYellow);
  ln.DrawLine(1.0*360*60/16384., 0, 1.0*360*60/16384., hError.GetHist()->GetMaximum());
  ln.SetLineColor(kRed);
  ln.DrawLine(2.0*360*60/16384., 0, 2.0*360*60/16384., hError.GetHist()->GetMaximum());
  // Plot the drive status vs time
  c1.cd();
  p = new TPad("", "", 0.6, 0.29, 1, 0.42);
  p->Draw();
  p->cd();
  hvt = (MHVsTime*)hDState.DrawClone("nonew");
  hvt->GetGraph()->SetMarkerStyle(kFullDotSmall);
  TH1 *hist = hvt->GetGraph()->GetHistogram();
  TAxis *axey = hist->GetYaxis();
  TAxis *axex = hist->GetXaxis();
  hist->SetXTitle("Time");
  hist->SetYTitle("");
  hist->SetTitle("");//Drive Status vs. Time");
  hist->SetStats(0);
  hist->SetMinimum(-0.5);
  hist->SetMaximum(4.5);
  axey->Set(5, -0.5, 4.5);
  axey->SetBinLabel(axey->FindFixBin(0), "Error");
  axey->SetBinLabel(axey->FindFixBin(1), "Stopped");
  axey->SetBinLabel(axey->FindFixBin(3), "Moving");
  axey->SetBinLabel(axey->FindFixBin(4), "Tracking");
  axey->SetLabelSize(0.15);
  axex->SetLabelSize(0.08);
  axex->SetTitleSize(0.09);
  axex->SetTitleOffset(0.45);
  // Control deviation of the motors vs zenith angle
  c1.cd();
  p = new TPad("", "", 0.6, 0.71, 1, 1);
  p->Draw();
  p->cd();
  gStyle->SetOptStat(1110);
  gStyle->SetStatFormat(".2g");
  mh3 = (MH3*)hError2.DrawClone("nonew");
  mh3->GetHist()->SetXTitle("Zd [\\circ]");
  mh3->GetHist()->SetYTitle("\\Delta [arcmin]");
  mh3->GetHist()->SetTitle("Control deviation of the motors");
  mh3->GetHist()->SetStats(1);
  mh3->GetHist()->Draw("box");
  // Zenith angle vs time
  c1.cd();
  p = new TPad("", "", 0.6, 0.42, 1, 0.71);
  p->Draw();
  p->cd();
  gPad->SetBorderMode(0);
  hvt = (MHVsTime*)hZd.DrawClone("nonew");
  hvt->GetGraph()->SetMarkerStyle(kFullDotSmall);
  if (hvt->GetGraph()->GetN())
    {
      hvt->GetGraph()->GetHistogram()->SetXTitle("Time");
      hvt->GetGraph()->GetHistogram()->SetYTitle("Zd [\\circ]");
      hvt->GetGraph()->GetHistogram()->SetTitle("Zd vs. Time");
      hvt->GetGraph()->GetHistogram()->SetStats(0);
    }
  
  
  //
  // Camera report 
  // 
  // HV and currents
  TCanvas &c2 = d->AddTab("HV");
  c2->Divide(1,2);
  c2->cd(1);
  // High voltages of the power supplies
  TLegend *legHV = new TLegend(0.85,0.75,0.99,0.99);
  TGraph *g = hHVA.GetGraph();
  g->SetMarkerColor(2);
  g->SetLineColor(2);
  g->SetTitle("Voltages of power supplies");
  legHV->AddEntry(g,"Power supply A (hvps1)","l");
  g = hHVB.GetGraph();
  g->SetMarkerColor(3);
  g->SetLineColor(3);
  legHV->AddEntry(g,"Power supply B (hvps2)","l");
  MHVsTime *clone1 = (MHVsTime*)hHVA.DrawClone("nonew");
  MHVsTime *clone2 = (MHVsTime*)hHVB.DrawClone("nonewsame");
  TH1 *hist = clone1->GetGraph()->GetHistogram();
  hist->SetXTitle("Time");
  hist->SetYTitle("High voltage [V]");
  hist->SetMinimum(0);
  legHV->DrawClone();
  c2->cd(2);
  // Currents of power supplies
  TLegend *legC = new TLegend(0.85,0.75,0.99,0.99);
  TGraph *g = hCA.GetGraph();
  g->SetMarkerColor(2);
  g->SetLineColor(2);
  g->SetTitle("Currents of power supplies");
  legC->AddEntry(g,"Power supply A (curr1)","l");
  g = hCB.GetGraph();
  g->SetMarkerColor(3);
  g->SetLineColor(3);
  legC->AddEntry(g,"Power supply B (curr2)","l");
  MHVsTime *clone1 = (MHVsTime*)hCA.DrawClone("nonew");
  MHVsTime *clone2 = (MHVsTime*)hCB.DrawClone("nonewsame");
  TH1 *hist = clone1->GetGraph()->GetHistogram();
  hist->SetXTitle("Time");
  hist->SetYTitle("Current [mA]");
  hist->SetMinimum(0);
  legC->DrawClone();
  
  // Cooling
  TCanvas &c3 = d->AddTab("Cooling");
  gStyle->SetPadGridX(kTRUE);
  gStyle->SetPadGridY(kTRUE);
  c3->Divide(1,2);
  c3->cd(1);
  // Camera temperatures
  TLegend *legtemp = new TLegend(0.85,0.75,0.99,0.99);
  TGraph *g = hTCenter.GetGraph();
  g->SetMarkerColor(8);
  g->SetLineColor(8);
  g->SetTitle("Camera temperature");
  legtemp->AddEntry(g,"T at camera center","l");
  g = hTWall.GetGraph();
  g->SetMarkerColor(12);
  g->SetLineColor(12);
  legtemp->AddEntry(g,"T at camera wall","l");
  g = hTWater.GetGraph();
  g->SetMarkerColor(4);
  g->SetLineColor(4);
  legtemp->AddEntry(g,"T at water deposit","l");
  g = hTOptLink.GetGraph();
  g->SetMarkerColor(2);
  g->SetLineColor(2);
  legtemp->AddEntry(g,"T at optical links","l");
  MHVsTime *clone1 = (MHVsTime*)hTCenter.DrawClone("nonew");
  MHVsTime *clone2 = (MHVsTime*)hTWall.DrawClone("nonewsame");
  MHVsTime *clone3 = (MHVsTime*)hTWater.DrawClone("nonewsame");
  MHVsTime *clone4 = (MHVsTime*)hTOptLink.DrawClone("nonewsame");
  TH1 *hist = clone1->GetGraph()->GetHistogram();
  hist->SetXTitle("Time");
  hist->SetYTitle("Temperature [\\circ C]");
  hist->SetMinimum(0);
  legtemp->DrawClone();
  // Camera relative humidity
  c3->cd(2);
  gPad->SetBorderMode(0);
  gPad->Divide(2, 1);
  TVirtualPad *c = gPad;
  c->cd(1);
  TLegend *leghum = new TLegend(0.75,0.75,0.99,0.99);
  TGraph *g = hHCenter.GetGraph();
  g->SetMarkerColor(8);
  g->SetLineColor(8);
  g->SetTitle("Camera relative humidity");
  leghum->AddEntry(g,"RH at camera center","l");
  g = hHWall.GetGraph();
  g->SetMarkerColor(12);
  g->SetLineColor(12);
  leghum->AddEntry(g,"RH at camera wall","l");
  clone1 = (MHVsTime*)hHCenter.DrawClone("nonew");
  clone2 = (MHVsTime*)hHWall.DrawClone("nonewsame");
  hist = clone1->GetGraph()->GetHistogram();
  hist->SetXTitle("Time");
  hist->SetYTitle("Relative humidity [%]");
  hist->SetMaximum(50);
  hist->SetMinimum(0);
  leghum->DrawClone();
  // Temperature distribution
  c->cd(2);
  TH1F *h3 = hTempOptLink.GetHistByName("TOptLink");
  h3->SetXTitle("Temperature [\\circ C]");
  h3->SetYTitle("");
  h3->SetLineColor(2);
  h3->SetTitle("Distribution of opt. links temperature (31.6+/-3.8\\circ)");
  h3->SetStats(0);
  MH3 *mh3 = (MH3*)hTempOptLink.DrawClone("nonew");

  //
  // Trigger report
  //
  TCanvas &c4 = d->AddTab("Trigger");
  gStyle->SetPadGridX(kTRUE);
  gStyle->SetPadGridY(kTRUE);
  TLegend *legtrig = new TLegend(0.75,0.85,0.99,0.99);
  TGraph *g = hTrigBL2.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(2);
  g->SetLineColor(2);
  g->SetTitle("L2 trigger rate");
  legtrig->AddEntry(g,"Before prescaler","l");
  g = hTrigAL2.GetGraph();
  g->SetMarkerColor(4);
  g->SetLineColor(4);
  legtrig->AddEntry(g,"After prescaler","l");
  MHVsTime *clone1 = (MHVsTime*)hTrigBL2.DrawClone("nonew");
  MHVsTime *clone2 = (MHVsTime*)hTrigAL2.DrawClone("nonewsame");
  TH1 *hist = clone1->GetGraph()->GetHistogram();
  hist->SetXTitle("Time");
  hist->SetYTitle("L2 trigger rate [Hz]"); 
  legtrig->DrawClone();
  
  //
  // CC report (Weather station)
  //
  TCanvas &c5 = d->AddTab("Central Control");
  gStyle->SetPadGridX(kTRUE);
  gStyle->SetPadGridY(kTRUE);
  c5->Divide(2,2);
  // Relative humidity
  c5->cd(1);
  TGraph *g = hCCHum.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(4);
  g->SetLineColor(4);
  g->SetTitle("Relative humidity");
  MHVsTime *clone1 = (MHVsTime*)hCCHum.DrawClone("nonew");
  TH1 *hist = clone1->GetGraph()->GetHistogram();
  hist->SetXTitle("Time");
  hist->SetYTitle("Humidity [%]"); 
  // Temperature
  c5->cd(2);
  TGraph *g = hCCTemp.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(2);
  g->SetLineColor(2);
  g->SetTitle("Temperature");
  MHVsTime *clone2 = (MHVsTime*)hCCTemp.DrawClone("nonew");
  TH1 *hist = clone2->GetGraph()->GetHistogram();
  hist->SetXTitle("Time");
  hist->SetYTitle("Temperature [\\circ C]"); 
  // Wind speed
  c5->cd(3);
  TGraph *g = hCCWS.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(3);
  g->SetLineColor(3);
  g->SetTitle("Wind speed");
  MHVsTime *clone3 = (MHVsTime*)hCCWS.DrawClone("nonew");
  TH1 *hist = clone3->GetGraph()->GetHistogram();
  hist->SetXTitle("Time");
  hist->SetYTitle("Wind speed [km/h]"); 
  // Solar radiation
  c5->cd(4);
  TGraph *g = hCCSR.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(5);
  g->SetLineColor(5);
  g->SetTitle("Solar radiation");
  MHVsTime *clone4 = (MHVsTime*)hCCSR.DrawClone("nonew");
  TH1 *hist = clone4->GetGraph()->GetHistogram();
  hist->SetXTitle("Time");
  hist->SetYTitle("Solar radiation [W/m^2]"); 

  //
  // Make sure the display hasn't been deleted by the user while the
  // eventloop was running.
  //
  if ((d = evtloop.GetDisplay()))
    {
      TString file = filename.Remove(filename.Index("."),5);
      // Save data in a postscriptfile (status.ps)
      d->SaveAsPS(-1,Form("%s.ps",file));
    }
}














