/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Sebastian Raducci 12/2003 <mailto:raducci@fisica.uniud.it>
!
!   Copyright: MAGIC Software Development, 2002-2004
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
//   MArrivalTimeCalc
//
//   This is a task that calculates the arrival times of photons. 
//   It returns the absolute maximum of the spline that interpolates
//   the FADC slices 
//
// Input Containers:
//   MRawEvtData
//
// Output Containers:
//   MArrivalTime
//   MRawEvtData
//
//////////////////////////////////////////////////////////////////////////////
#include "MArrivalTimeCalc.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"

#include "MGeomCam.h"
#include "MArrivalTime.h"
#include "MRawEvtData.h"      
#include "MRawEvtPixelIter.h"

ClassImp(MArrivalTimeCalc);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor.
//
MArrivalTimeCalc::MArrivalTimeCalc(const char *name, const char *title)
{
    fName  = name  ? name  : "MArrivalTimeCalc";
    fTitle = title ? title : "Calculate photons arrival time";

}

// --------------------------------------------------------------------------
//
// The PreProcess searches for the following input containers:
//  - MRawEvtData
//  - MArrivalTime
//
// The following output containers are also searched and created if
// they were not found:
//  - MArrivalTime
//
Int_t MArrivalTimeCalc::PreProcess(MParList *pList)
{
    fRawEvt = (MRawEvtData*)pList->FindObject(AddSerialNumber("MRawEvtData"));
    if (!fRawEvt)
    {
        *fLog << err << "MRawEvtData not found... aborting." << endl;
        return kFALSE;
    }

    fArrTime = (MArrivalTime*)pList->FindCreateObj(AddSerialNumber("MArrivalTime"));
    if (!fArrTime)
        return kFALSE;

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// The ReInit searches for the following input containers:
//  - MGeomCam
//
Bool_t MArrivalTimeCalc::ReInit(MParList *pList)
{
    MGeomCam *cam = (MGeomCam*)pList->FindObject(AddSerialNumber("MGeomCam"));
    if (!cam)
    {
        *fLog << err << GetDescriptor() << ": No MGeomCam found... aborting." << endl;
        return kFALSE;
    }

    fArrTime->InitSize(cam->GetNumPixels());

    return kTRUE;
}


// --------------------------------------------------------------------------
//
// Evaluation of the mean arrival times (spline interpolation)
// per pixel and store them in the MArrivalTime container.
//
Int_t MArrivalTimeCalc::Process()
{
    MRawEvtPixelIter pixel(fRawEvt);

    while (pixel.Next())
    {
	const UInt_t idx = pixel.GetPixelId();

        // If pixel is saturated we use LoGains
        if (pixel.GetMaxHiGainSample() == 0xff && pixel.HasLoGain())
	{
	    const Byte_t *ptr = pixel.GetLoGainSamples();
	    const Short_t nSlice = fRawEvt->GetNumLoGainSamples();
	    fArrTime->Calc(ptr, nSlice, idx);
	}
        // Use HiGains
        else
	{
            const Byte_t *ptr = pixel.GetHiGainSamples();
	    const Short_t nSlice = fRawEvt->GetNumHiGainSamples();
	    fArrTime->Calc(ptr, nSlice, idx);
	}
        // If pixel is saturated and hasn't lo gains we do nothing, it's value remains -1
    }
    
    fArrTime->SetReadyToSave();

    return kTRUE;
}
