#ifndef MARS_MCalibrationPix
#define MARS_MCalibrationPix

#ifndef MARS_MParContainer
#include "MParContainer.h"
#endif

#include "MHCalibrationPixel.h"

class MCalibrationPix : public MParContainer
{
private:

  Int_t   fPixId;                     // the pixel Id
  
  const Float_t fElectronicPedRms;    // The pure electronic component of the RMS
  const Float_t fErrElectronicPedRms; // The error of the pure electronic component of the RMS

  const Float_t fFactor;              // The laboratory F-factor
  const Float_t fFactorError;         // The laboratory F-factor Error

  const Float_t fChargeLimit;         // The limit (in units of PedRMS) for acceptance of the fitted mean charge
  const Float_t fChargeErrLimit;      // The limit (in units of PedRMS) for acceptance of the fitted charge sigma
  const Float_t fChargeRelErrLimit;   // The limit (in units of Error of fitted charge) for acceptance of the fitted mean  
  Byte_t  fFlags;               // Flag for the set Bits
  
  Float_t fCharge;              // The mean reduced charge after the fit
  Float_t fErrCharge;           // The error of reduced mean charge after the fit
  Float_t fSigmaCharge;         // The sigma of the mean charge after the fit
  Float_t fErrSigmaCharge;      // The error of the sigma of the mean charge after the fit
  Float_t fRSigmaSquare;        // The reduced squares of sigmas after the fit
  Float_t fChargeProb;          // The probability of the fit function 

  Float_t fPed;                 // The mean pedestal (from MPedestalPix)
  Float_t fPedRms;              // The pedestal  RMS (from MPedestalPix)
  Float_t fErrPedRms;           // The error of the pedestal  RMS (from MPedestalPix)  

  Float_t fTime;                // The mean arrival time after the fit  
  Float_t fSigmaTime;           // The error of the mean arrival time after the fit
  Float_t fTimeChiSquare;       // The Chi Square of the fit function 
  Float_t fTimeProb;            // The probability of the fit function 
  
  Float_t fPheFFactorMethod;                // The number of Phe's calculated (F-factor method)
  Float_t fPheFFactorMethodError;           // The error on the number of Phe's calculated (F-factor method)

  Float_t fConversionFFactorMethod;         // The conversion factor to Phe's (F-factor method)
  Float_t fConversionBlindPixelMethod;      // The conversion factor to Ph's (Blind Pixel method)
  Float_t fConversionPINDiodeMethod;        // The conversion factor to Ph's (PIN Diode method)

  Float_t fConversionErrorFFactorMethod;    // The error of the conversion factor to Phe's (F-factor method)
  Float_t fConversionErrorBlindPixelMethod; // The error of the conversion factor to Ph's (Blind Pixel method)
  Float_t fConversionErrorPINDiodeMethod;   // The error of the conversion factor to Ph's (PIN Diode method)

  Float_t fConversionSigmaFFactorMethod;    // The sigma of conversion factor to Ph's (F-factor method)
  Float_t fConversionSigmaBlindPixelMethod; // The conversion factor to Ph's (Blind Pixel method)
  Float_t fConversionSigmaPINDiodeMethod;   // The conversion factor to Phd's (PIN Diode method)

  Float_t fConversionHiLo;                  // The conversion factor between Hi Gain and Lo Gain  
  Float_t fConversionHiLoError;             // The error of the conversion factor between Hi Gain and Lo Gain  
  
  enum  { kHiGainSaturation,
          kExcluded, kExcludeQualityCheck,
          kFitValid, kFitted,
          kBlindPixelMethodValid, kFFactorMethodValid, kPINDiodeMethodValid };
  
  MHCalibrationPixel *fHist;                // Pointer to the histograms performing the fits, etc.  

  Bool_t CheckChargeFitValidity();
  Bool_t CheckTimeFitValidity();
  Bool_t CheckOscillations();  
  
public:

  MCalibrationPix(const char *name=NULL, const char *title=NULL);
  ~MCalibrationPix();
  
  void Clear(Option_t *o="");

  // Getter
  MHCalibrationPixel *GetHist() const   { return fHist;     }

  // Charges
  Float_t GetCharge()              const { return fCharge;         }
  Float_t GetErrCharge()           const { return fErrCharge;      }
  Float_t GetChargeProb()          const { return fChargeProb;     }    
  Float_t GetSigmaCharge()         const { return fSigmaCharge;    }
  Float_t GetErrSigmaCharge()      const { return fErrSigmaCharge; }
  Float_t GetRSigmaSquare()        const { return fRSigmaSquare;   }

  // Times  
  Float_t GetTime()                const { return fTime;           }
  Float_t GetSigmaTime()           const { return fSigmaTime;      }
  Float_t GetTimeChiSquare()       const { return fTimeChiSquare;  }
  Float_t GetTimeProb()            const { return fTimeProb;  }      

  Float_t GetConversionHiLo()                 const  { return fConversionHiLo;        }
  Float_t GetConversionHiLoError()            const  { return fConversionHiLoError;   }

  Float_t GetMeanConversionBlindPixelMethod()  const { return fConversionBlindPixelMethod ; }
  Float_t GetErrorConversionBlindPixelMethod() const { return fConversionErrorBlindPixelMethod ; }
  Float_t GetSigmaConversionBlindPixelMethod() const { return fConversionSigmaBlindPixelMethod ; }

  Float_t GetMeanConversionFFactorMethod()     const { return fConversionFFactorMethod ;       }
  Float_t GetErrorConversionFFactorMethod()    const { return fConversionErrorFFactorMethod ;  }
  Float_t GetSigmaConversionFFactorMethod()    const { return fConversionSigmaFFactorMethod ;  }

  Float_t GetPheFFactorMethod()                const { return fPheFFactorMethod;               }
  Float_t GetPheFFactorMethodError()           const { return fPheFFactorMethodError;          }

  Float_t GetMeanConversionPINDiodeMethod()    const { return fConversionPINDiodeMethod ;      }
  Float_t GetErrorConversionPINDiodeMethod()   const { return fConversionErrorPINDiodeMethod ; }
  Float_t GetSigmaConversionPINDiodeMethod()   const { return fConversionSigmaPINDiodeMethod ; }

  Int_t   GetPixId()                           const  { return fPixId;   }

  Float_t GetPed()                             const { return fPed;    }
  Float_t GetPedRms()                          const { return fPedRms; }

  Bool_t IsExcluded()              const;
  Bool_t IsFitValid()              const;
  Bool_t IsFitted()                const;
  Bool_t IsBlindPixelMethodValid() const;
  Bool_t IsFFactorMethodValid()    const;
  Bool_t IsPINDiodeMethodValid()   const;

  // Setter
  void SetPedestal(Float_t ped, Float_t pedrms);
  void SetConversionHiLo(Float_t c)      { fConversionHiLo      = c;    }
  void SetConversionHiLoError(Float_t e)  { fConversionHiLoError = e;    }

  // Setters for MC
  void SetConversionFFactorMethod(Float_t c, Float_t err, Float_t sig);
  void SetConversionBlindPixelMethod(Float_t c, Float_t err, Float_t sig);
  void SetConversionPINDiodeMethod(Float_t c, Float_t err, Float_t sig);

  // Bit Setters
  void SetHiGainSaturation(Bool_t b = kTRUE);
  void SetExcluded(Bool_t b = kTRUE);
  void SetExcludeQualityCheck(Bool_t b = kTRUE);
  void SetFitValid(Bool_t b = kTRUE);
  void SetFitted(Bool_t b = kTRUE);
  void SetBlindPixelMethodValid(Bool_t b = kTRUE);
  void SetFFactorMethodValid(Bool_t b = kTRUE);
  void SetPINDiodeMethodValid(Bool_t b = kTRUE);
  
  // Fill histos
  Bool_t FillChargeHiGain(Float_t q) const                  { return fHist->FillChargeHiGain(q); }
  Bool_t FillTimeHiGain(Float_t t)   const                  { return fHist->FillTimeHiGain(t); }  
  Bool_t FillRChargevsTimeHiGain(Float_t rq, Int_t t) const { return fHist->FillChargevsNHiGain(rq,t); }    

  Bool_t FillChargeLoGain(Float_t q) const                  { return fHist->FillChargeLoGain(q); }
  Bool_t FillTimeLoGain(Float_t t)   const                  { return fHist->FillTimeLoGain(t); }  
  Bool_t FillRChargevsTimeLoGain(Float_t rq, Int_t t) const { return fHist->FillChargevsNLoGain(rq,t); }    
  
  Bool_t FillChargesInGraph(Float_t qhi,Float_t qlo) const  { return fHist->FillPointInGraph(qhi,qlo); }

  void   DefinePixId(Int_t i);

  // Fits
  Bool_t FitCharge();
  Bool_t FitTime();
  
  // Draws
  void Draw(Option_t *opt="")                    { fHist->Draw(opt); }
  TObject *DrawClone(Option_t *opt="") const    { return fHist->DrawClone(opt); }  
  
  // Miscellaneous
  void   ApplyLoGainConversion();

  ClassDef(MCalibrationPix, 1)	// Storage Container for Calibration information of one pixel
};

#endif

