#ifndef MARS_MCalibrationCalc
#define MARS_MCalibrationCalc

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// MCalibrationCalc                                                   //
//                                                                         //
// Integrates the time slices of the all pixels of a calibration event     //
// and substract the pedestal value                                        //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#ifndef MARS_MTask
#include "MTask.h"
#endif

#ifndef ROOT_TArrayI
#include "TArrayI.h"
#endif

#ifndef MARS_MCalibrationCam
#include "MCalibrationCam.h"
#endif

#include "TString.h"

class MRawEvtData;
class MRawRunHeader;

class MPedestalCam;
class MCalibrationCam;
class MExtractedSignalCam;
class MArrivalTime;

class MTime;

class MCalibrationCalc : public MTask
{
private:

  MPedestalCam             *fPedestals;    // Pedestals of all pixels in the camera
  MCalibrationCam          *fCalibrations; // Calibration events of all pixels in the camera
  MExtractedSignalCam      *fSignals;      // Calibration events of all pixels in the camera

  MRawEvtData              *fRawEvt;       // raw event data (time slices)
  MRawRunHeader            *fRunHeader;    // RunHeader information

  MArrivalTime             *fArrivalTime;  // Calculated Arrival Times
  
  MTime                    *fEvtTime;      // Time of the event

  enum { kBlindPixelId = 559,               // ID of the blind pixel
         kPINDiodeId = 9999};              // ID of the PIN Diode
  
  Int_t fEvents;                           // Number of events  
  Int_t fCosmics;                          // Number of events due to supposed cosmics

  Byte_t fNumHiGainSamples; 
  Byte_t fNumLoGainSamples; 
  Float_t fSqrtHiGainSamples; 
  
  Float_t fConversionHiLo;
  Byte_t fFlags;                           // Flag for the fits used

  TString fExcludedPixelsFile;
  UInt_t  fNumExcludedPixels;

  enum  { kUseTimes, kUseBlindPixelFit, kUsePinDiodeFit,
          kUseCosmicsRejection, kUseQualityChecks };

public:
  
  enum PulserColor_t  { kEGreen, kEBlue, kEUV, kECT1 };

private:

  PulserColor_t  fColor;
  
  Bool_t ReInit(MParList *pList); 
  Int_t PreProcess(MParList *pList);
  Int_t Process();
  Int_t PostProcess();
  
public:

  MCalibrationCalc(const char *name=NULL, const char *title=NULL);

  void Clear(const Option_t *o="");
  
  // Skipping parts of the work
  void SkipTime(Bool_t b=kTRUE)
      {b ? CLRBIT(fFlags, kUseTimes) : SETBIT(fFlags, kUseTimes);}
  void SkipBlindPixelFit(Bool_t b=kTRUE)
      {b ? CLRBIT(fFlags, kUseBlindPixelFit) : SETBIT(fFlags, kUseBlindPixelFit);}
  void SkipPinDiodeFit(Bool_t b=kTRUE)
      {b ? CLRBIT(fFlags, kUsePinDiodeFit) : SETBIT(fFlags, kUsePinDiodeFit);}
  void SkipCosmicsRejection(Bool_t b=kTRUE)
      {b ? CLRBIT(fFlags, kUseCosmicsRejection) : SETBIT(fFlags, kUseCosmicsRejection);}
  void SkipQualityChecks(Bool_t b=kTRUE)
      {b ? CLRBIT(fFlags, kUseQualityChecks) : SETBIT(fFlags, kUseQualityChecks);}


  // Setters 
  void SetPulserColor(PulserColor_t color)    { fColor = color; }
  void SetConversionHiLo(Float_t conv)         { fConversionHiLo = conv; }

  // Getters
  MCalibrationBlindPix *GetBlindPixel() const;
  MCalibrationPINDiode *GetPINDiode()   const;

  // Exclude pixels from configuration file
  void ExcludePixelsFromAsciiFile(const char *file) { fExcludedPixelsFile = file;  }
  
  ClassDef(MCalibrationCalc, 1)   // Task to fill the Calibration Containers from raw data
};

#endif
