/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug   11/2003 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// MCalibrationPix                                                         //
//                                                                         //
// This is the storage container to hold informations about the pedestal   //
// (offset) value of one Pixel (PMT).                                      //
//                                                                         //
// The following values are initialized to meaningful values:
//
// - The Electronic Rms to 1.5 per FADC slice
// - The uncertainty about the Electronic RMS to 0.3 per slice
// - The F-Factor is assumed to have been measured in Munich to 1.13 - 1.17. 
//   We use here the Square of the Munich definition, thus:
//   Mean F-Factor  = 1.15*1.15 = 1.32
//   Error F-Factor = 2.*0.02   = 0.04
//
/////////////////////////////////////////////////////////////////////////////
#include "MCalibrationPix.h"
#include "MCalibrationConfig.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MCalibrationPix);

using namespace std;

const Float_t MCalibrationPix::gkElectronicPedRms    = 1.5;
const Float_t MCalibrationPix::gkErrElectronicPedRms = 0.3;
const Float_t MCalibrationPix::gkFFactor             = 1.32;
const Float_t MCalibrationPix::gkFFactorError        = 0.04;
const Float_t MCalibrationPix::gkChargeLimit         = 3.;
const Float_t MCalibrationPix::gkChargeErrLimit      = 0.;
const Float_t MCalibrationPix::gkChargeRelErrLimit   = 1.;
const Float_t MCalibrationPix::gkTimeLimit           = 1.5;
const Float_t MCalibrationPix::gkTimeErrLimit        = 3.;

// --------------------------------------------------------------------------
//
// Default Constructor: 
//
MCalibrationPix::MCalibrationPix(const char *name, const char *title)
    : fPixId(-1),
      fFlags(0)
{

  fName  = name  ? name  : "MCalibrationPixel";
  fTitle = title ? title : "Container of the MHCalibrationPixels and the fit results";

  //
  // At the moment, we don't have a database, yet, 
  // so we get it from the configuration file
  //
  fConversionHiLo      = gkConversionHiLo;
  fConversionHiLoError = gkConversionHiLoError;  

  fHist = new MHCalibrationPixel("MHCalibrationPixel","Calibration Histograms Pixel ");

  if (!fHist)
    *fLog << warn << dbginf << " Could not create MHCalibrationPixel " << endl;

  Clear();
}

MCalibrationPix::~MCalibrationPix() 
{
  delete fHist;
}


// ------------------------------------------------------------------------
//
// Invalidate values
//
void MCalibrationPix::Clear(Option_t *o)
{

  fHist->Reset();

  CLRBIT(fFlags, kHiGainSaturation);
  CLRBIT(fFlags, kExcluded);
  CLRBIT(fFlags, kChargeFitValid);
  CLRBIT(fFlags, kTimeFitValid);
  CLRBIT(fFlags, kFitted);
  CLRBIT(fFlags, kBlindPixelMethodValid);
  CLRBIT(fFlags, kFFactorMethodValid);
  CLRBIT(fFlags, kPINDiodeMethodValid);

  fCharge                           =  -1.;
  fErrCharge                        =  -1.;
  fSigmaCharge                      =  -1.;
  fErrSigmaCharge                   =  -1.;
  fRSigmaSquare                     =  -1.;
  fChargeProb                       =  -1.;
  fPed                              =  -1.;
  fPedRms                           =  -1.;
  fErrPedRms                        =   0.;
  fTime                             =  -1.;
  fErrTime                          =  -1.;
  fSigmaTime                        =  -1.;
  fTimeProb                         =  -1.;
  fTimeFirstHiGain                  =   0 ;
  fTimeLastHiGain                   =   0 ;
  fTimeFirstLoGain                  =   0 ;
  fTimeLastLoGain                   =   0 ;

  fAbsTimeMean                      =  -1.;
  fAbsTimeRms                       =  -1.;

  fPheFFactorMethod                 =  -1.;
  fPheFFactorMethodError            =  -1.;
  fConversionFFactorMethod          =  -1.;
  fConversionBlindPixelMethod       =  -1.;
  fConversionPINDiodeMethod         =  -1.;
  fConversionErrorFFactorMethod     =  -1.;
  fConversionErrorBlindPixelMethod  =  -1.;
  fConversionErrorPINDiodeMethod    =  -1.;
  fConversionSigmaFFactorMethod     =  -1.;
  fConversionSigmaBlindPixelMethod  =  -1.;
  fConversionSigmaPINDiodeMethod    =  -1.;

}


void MCalibrationPix::DefinePixId(Int_t i)
{
  
  fPixId = i;
  fHist->ChangeHistId(i);
  
}


// --------------------------------------------------------------------------
//
// Set the pedestals from outside
//
void MCalibrationPix::SetPedestal(Float_t ped, Float_t pedrms)
{

  fPed    = ped;    
  fPedRms = pedrms;
  
}

// --------------------------------------------------------------------------
//
// Set the conversion factors from outside (only for MC)
//
void MCalibrationPix::SetConversionFFactorMethod(Float_t c, Float_t err, Float_t sig)
{
  fConversionFFactorMethod = c;
  fConversionErrorFFactorMethod = err;
  fConversionSigmaFFactorMethod = sig;
}


// --------------------------------------------------------------------------
//
// Set the conversion factors from outside (only for MC)
//
void MCalibrationPix::SetConversionBlindPixelMethod(Float_t c, Float_t err, Float_t sig)
{
  fConversionBlindPixelMethod = c;
  fConversionErrorBlindPixelMethod = err;
  fConversionSigmaBlindPixelMethod = sig;
}

// --------------------------------------------------------------------------
//
// Set the conversion factors from outside (only for MC)
//
void MCalibrationPix::SetConversionPINDiodeMethod(Float_t c, Float_t err, Float_t sig)
{
  fConversionPINDiodeMethod = c ;
  fConversionErrorPINDiodeMethod = err;
  fConversionSigmaPINDiodeMethod = sig;
}

// --------------------------------------------------------------------------
//
// Set the Hi Gain Saturation Bit from outside (only for MC)
//
void MCalibrationPix::SetHiGainSaturation(Bool_t b)
{

  if (b) 
    { 
      SETBIT(fFlags, kHiGainSaturation); 
      fHist->SetUseLoGain(1); 
    }
  else    
    {
      CLRBIT(fFlags, kHiGainSaturation); 
      fHist->SetUseLoGain(0); 
    }
}

// --------------------------------------------------------------------------
//
// Set the Excluded Bit from outside 
//
void MCalibrationPix::SetExcluded(Bool_t b )
{ 
  b ?  SETBIT(fFlags, kExcluded) : CLRBIT(fFlags, kExcluded); 
}


// --------------------------------------------------------------------------
//
// Set the Excluded Bit from outside 
//
void MCalibrationPix::SetExcludeQualityCheck(Bool_t b )
{ 
  b ?  SETBIT(fFlags, kExcludeQualityCheck) : CLRBIT(fFlags, kExcludeQualityCheck); 
}

// --------------------------------------------------------------------------
//
// Set the Excluded Bit from outside 
//
void MCalibrationPix::SetChargeFitValid(Bool_t b )    
{ 
  b ?  SETBIT(fFlags, kChargeFitValid) : CLRBIT(fFlags, kChargeFitValid); 
}

// --------------------------------------------------------------------------
//
// Set the Excluded Bit from outside 
//
void MCalibrationPix::SetTimeFitValid(Bool_t b )    
{ 
  b ?  SETBIT(fFlags, kTimeFitValid) : CLRBIT(fFlags, kTimeFitValid); 
}

// --------------------------------------------------------------------------
//
// Set the Excluded Bit from outside 
//
void MCalibrationPix::SetFitted(Bool_t b )
{ 
  b ?  SETBIT(fFlags, kFitted) : CLRBIT(fFlags, kFitted); 
}
    
// --------------------------------------------------------------------------
//
// Set the Excluded Bit from outside 
//
void MCalibrationPix::SetBlindPixelMethodValid(Bool_t b )
{ 
  b ?  SETBIT(fFlags, kBlindPixelMethodValid) : CLRBIT(fFlags, kBlindPixelMethodValid); 
}    

// --------------------------------------------------------------------------
//
// Set the Excluded Bit from outside 
//
void MCalibrationPix::SetFFactorMethodValid(Bool_t b )
{ 
  b ?  SETBIT(fFlags, kFFactorMethodValid) : CLRBIT(fFlags, kFFactorMethodValid); 
}    

// --------------------------------------------------------------------------
//
// Set the Excluded Bit from outside 
//
void MCalibrationPix::SetPINDiodeMethodValid(Bool_t b )  
{ 
  b ?  SETBIT(fFlags, kPINDiodeMethodValid) : CLRBIT(fFlags, kPINDiodeMethodValid); 
}

void MCalibrationPix::SetAbsTimeBordersHiGain(Byte_t f, Byte_t l)
{

  fTimeFirstHiGain = f;
  fTimeLastHiGain  = l;
  
}

void MCalibrationPix::SetAbsTimeBordersLoGain(Byte_t f, Byte_t l)
{

  fTimeFirstLoGain = f;
  fTimeLastLoGain  = l;
  
}



Bool_t MCalibrationPix::IsExcluded() const
 { 
   return TESTBIT(fFlags,kExcluded);  
 }

Bool_t MCalibrationPix::IsChargeFitValid() const 
{
  return TESTBIT(fFlags, kChargeFitValid);  
}

Bool_t MCalibrationPix::IsTimeFitValid() const 
{
  return TESTBIT(fFlags, kTimeFitValid);  
}

Bool_t MCalibrationPix::IsFitted() const 
{ 
  return TESTBIT(fFlags, kFitted);    
}

Bool_t MCalibrationPix::IsBlindPixelMethodValid() const 
{ 
  return TESTBIT(fFlags, kBlindPixelMethodValid); 
}

Bool_t MCalibrationPix::IsFFactorMethodValid()  const 
{ 
  return TESTBIT(fFlags, kFFactorMethodValid);     
}

Bool_t MCalibrationPix::IsPINDiodeMethodValid() const 
{ 
  return TESTBIT(fFlags, kPINDiodeMethodValid);    
}


// --------------------------------------------------------------------------
//
// 1) Return if the charge distribution is already succesfully fitted  
//    or if the histogram is empty
// 2) Set a lower Fit range according to 1.5 Pedestal RMS in order to avoid 
//    possible remaining cosmics to spoil the fit.
// 3) Decide if the LoGain Histogram is fitted or the HiGain Histogram
// 4) Fit the histograms with a Gaussian
// 5) In case of failure set the bit kFitted to false
// 6) Retrieve the results and store them in this class
// 7) Calculate the number of photo-electrons after the F-Factor method
// 8) Calculate the errors of the F-Factor method
//
// The fits are declared valid (fFitValid = kTRUE), if:
//
// 1) Pixel has a fitted charge greater than 3*PedRMS
// 2) Pixel has a fit error greater than 0.
// 3) Pixel has a fit Probability greater than 0.0001 
// 4) Pixel has a charge sigma bigger than its Pedestal RMS
// 5) If FitTimes is used, 
//    the mean arrival time is at least 1.0 slices from the used edge slices 
// (this stage is only performed in the times fit)
//
// If the histogram is empty, all values are set to -1.
//
// The conversion factor after the F-Factor method is declared valid, if:
//
// 1) fFitValid is kTRUE
// 2) Conversion Factor is bigger than 0.
// 3) The error of the conversion factor is smaller than 10%
//
Bool_t MCalibrationPix::FitCharge() 
{

  //
  // 1) Return if the charge distribution is already succesfully fitted  
  //    or if the histogram is empty
  //
  if (fHist->IsChargeFitOK() || fHist->IsEmpty())
    return kTRUE;

  //
  // 2) Set a lower Fit range according to 1.5 Pedestal RMS in order to avoid 
  //    possible remaining cosmics to spoil the fit.
  //
  //  if (fPed && fPedRms)
  //    fHist->SetLowerFitRange(1.5*fPedRms);
  //  else
  //    *fLog << warn << "WARNING: Cannot set lower fit range: Pedestals not available" << endl;

  //
  // 3) Decide if the LoGain Histogram is fitted or the HiGain Histogram
  //
  if (fHist->UseLoGain())
    SetHiGainSaturation();
      
  //
  // 4) Fit the Lo Gain histograms with a Gaussian
  //
  if(fHist->FitCharge())
    {
      SETBIT(fFlags,kFitted);
    }
  else
    {
      *fLog << warn << "WARNING: Could not fit charges of pixel " << fPixId << endl;
      //          
      // 5) In case of failure set the bit kFitted to false
      //
      CLRBIT(fFlags,kFitted);
    }

  //
  // 6) Retrieve the results and store them in this class
  //
  fCharge         = fHist->GetChargeMean();
  fErrCharge      = fHist->GetChargeMeanErr(); 
  fSigmaCharge    = fHist->GetChargeSigma();
  fErrSigmaCharge = fHist->GetChargeSigmaErr();
  fChargeProb     = fHist->GetChargeProb();

  if (CheckChargeFitValidity())
    SETBIT(fFlags,kChargeFitValid);
  else
    {
      CLRBIT(fFlags,kChargeFitValid);
      return kFALSE;
    }
  
  //
  // 7) Calculate the number of photo-electrons after the F-Factor method
  // 8) Calculate the errors of the F-Factor method
  //
  if ((fPed > 0.)  && (fPedRms > 0.))
    {
      
      //
      // Square all variables in order to avoid applications of square root
      //
      // First the relative error squares
      //
      const Float_t chargeSquare              =       fCharge*   fCharge;
      const Float_t chargeSquareRelErrSquare  = 4.*fErrCharge*fErrCharge / chargeSquare;

      const Float_t fFactorRelErrSquare       = gkFFactorError * gkFFactorError / (gkFFactor * gkFFactor);
      //
      // Now the absolute error squares
      //
      const Float_t sigmaSquare               =       fSigmaCharge*   fSigmaCharge;
      const Float_t sigmaSquareErrSquare      = 4.*fErrSigmaCharge*fErrSigmaCharge * sigmaSquare;

      const Float_t elecRmsSquare             =       gkElectronicPedRms*   gkElectronicPedRms;
      const Float_t elecRmsSquareErrSquare    = 4.*gkErrElectronicPedRms*gkErrElectronicPedRms * elecRmsSquare;

      Float_t pedRmsSquare                    =       fPedRms*   fPedRms;
      Float_t pedRmsSquareErrSquare           = 4.*fErrPedRms*fErrPedRms * pedRmsSquare;

      if (TESTBIT(fFlags,kHiGainSaturation))
	{
          
          //
          // We do not know the Lo Gain Pedestal RMS, so we have to retrieve it 
          // from the Hi Gain: 
          //
          // We extract the pure NSB contribution:
          //
          Float_t nsbSquare             =  pedRmsSquare          - elecRmsSquare;
          Float_t nsbSquareRelErrSquare = (pedRmsSquareErrSquare + elecRmsSquareErrSquare)
                                          / (nsbSquare * nsbSquare) ;

	  if (nsbSquare < 0.)
            nsbSquare = 0.;
          
          //
          // Now, we divide the NSB by the conversion factor and 
          // add it quadratically to the electronic noise
          //
          const Float_t conversionSquare             =    fConversionHiLo     *fConversionHiLo;
          const Float_t conversionSquareRelErrSquare = 4.*fConversionHiLoError*fConversionHiLoError/conversionSquare;

          //
          // Calculate the new "Pedestal RMS"
          //
          const Float_t convertedNsbSquare          =  nsbSquare             / conversionSquare;
          const Float_t convertedNsbSquareErrSquare = (nsbSquareRelErrSquare + conversionSquareRelErrSquare)
                                                      * convertedNsbSquare * convertedNsbSquare;

          pedRmsSquare           = convertedNsbSquare           + elecRmsSquare;
          pedRmsSquareErrSquare  = convertedNsbSquareErrSquare  + elecRmsSquareErrSquare;
          
	}    /* if (kHiGainSaturation) */
      
      //
      // Calculate the reduced sigmas
      //
      fRSigmaSquare = sigmaSquare - pedRmsSquare;
      if (fRSigmaSquare <= 0.)
        {
          *fLog << warn 
                << "WARNING: Cannot apply F-Factor calibration: Reduced Sigma smaller than 0 in pixel " 
                << fPixId << endl;
          if (TESTBIT(fFlags,kHiGainSaturation))
            ApplyLoGainConversion();
          return kFALSE;
        }

      const Float_t rSigmaSquareRelErrSquare = (sigmaSquareErrSquare + pedRmsSquareErrSquare)
                                               / (fRSigmaSquare * fRSigmaSquare) ;
  
      //
      // Calculate the number of phe's from the F-Factor method
      // (independent on Hi Gain or Lo Gain)
      //
      fPheFFactorMethod = gkFFactor * chargeSquare / fRSigmaSquare;

      const Float_t pheFFactorRelErrSquare =  fFactorRelErrSquare
                                            + chargeSquareRelErrSquare
                                            + rSigmaSquareRelErrSquare ;
          
      fPheFFactorMethodError        =  TMath::Sqrt(pheFFactorRelErrSquare) * fPheFFactorMethod;

      //
      // Calculate the conversion factors 
      //
      if (TESTBIT(fFlags,kHiGainSaturation))
        ApplyLoGainConversion();
      
      const Float_t chargeRelErrSquare = fErrCharge*fErrCharge / (fCharge * fCharge);
      
      fConversionFFactorMethod        =  fPheFFactorMethod / fCharge ;
      fConversionErrorFFactorMethod   =  ( pheFFactorRelErrSquare + chargeRelErrSquare )
                                         * fConversionFFactorMethod * fConversionFFactorMethod;
      
      if ( IsChargeFitValid()               &&
           (fConversionFFactorMethod > 0.) &&
           (fConversionErrorFFactorMethod/fConversionFFactorMethod < 0.1) )
        SETBIT(fFlags,kFFactorMethodValid);
      else 
        CLRBIT(fFlags,kFFactorMethodValid);
      
    } /*   if ((fPed > 0.)  && (fPedRms > 0.)) */

  return kTRUE;
  
}

//
// The check return kTRUE if:
//
// 0) No value is nan
// 1) Pixel has a fitted charge greater than 3*PedRMS
// 2) Pixel has a fit error greater than 0. 
// 3) Pixel has a fitted charge greater its charge error
// 4) Pixel has a fit Probability greater than 0.0001 
// 5) Pixel has a charge sigma bigger than its Pedestal RMS
// 
Bool_t MCalibrationPix::CheckChargeFitValidity()
{

  if (TMath::IsNaN(fCharge)
      || TMath::IsNaN(fErrCharge)
      || TMath::IsNaN(fSigmaCharge)
      || TMath::IsNaN(fErrSigmaCharge)
      || TMath::IsNaN(fChargeProb))
    {
      *fLog << warn << "WARNING: Some of the fit values are NAN in Pixel " << fPixId << endl;
      return kFALSE;
    }
  

  if (TESTBIT(fFlags,kExcludeQualityCheck))
    return kTRUE;

  Float_t equivpedestal = GetPedRms();

  if (TESTBIT(fFlags,kHiGainSaturation))
    equivpedestal /= fConversionHiLo;
      
  if (fCharge < gkChargeLimit*equivpedestal)
    {
      *fLog << warn << "WARNING: Fitted Charge is smaller than "
            << gkChargeLimit << " Pedestal RMS in Pixel " << fPixId << endl;
      return kFALSE;
    }
  
  if (fErrCharge < gkChargeErrLimit) 
    {
      *fLog << warn << "WARNING: Error of Fitted Charge is smaller than "
            << gkChargeErrLimit << " in Pixel " << fPixId << endl;
      return kFALSE;
    }
      
  if (fCharge < gkChargeRelErrLimit*fErrCharge) 
    {
      *fLog << warn << "WARNING: Fitted Charge is smaller than "
            << gkChargeRelErrLimit << "* its error in Pixel " << fPixId << endl;
      return kFALSE;
    }
      
  if (!fHist->IsChargeFitOK()) 
    {
      *fLog << warn << "WARNING: Probability of Fitted Charge too low in Pixel " 
            << fPixId << endl;
      return kFALSE;
    }

  if (fSigmaCharge < equivpedestal)
    {
      *fLog << warn << "WARNING: Sigma of Fitted Charge smaller than Pedestal RMS in Pixel " 
            << fPixId << endl;
      return kFALSE;
    }
  return kTRUE;
}

//
// The check return kTRUE if:
//
// 0) No value is nan
// 1) Pixel has a fitted rel. time smaller than 3*FADC slices
// 2) Pixel has a fit error greater than 0. 
// 4) Pixel has a fit Probability greater than 0.001 
// 5) The absolute arrival time is at least 1.0 slices from the used edge slices 
//
Bool_t MCalibrationPix::CheckTimeFitValidity()
{

  if (TMath::IsNaN(fTime)
      || TMath::IsNaN(fErrTime)
      || TMath::IsNaN(fSigmaTime)
      || TMath::IsNaN(fTimeProb))
    {
      *fLog << warn << "WARNING: Some of the time fit values are NAN in Pixel " 
            << fPixId << endl;
      return kFALSE;
    }
  
  if (TESTBIT(fFlags,kExcludeQualityCheck))
    return kTRUE;

  if (TMath::Abs(fTime) > gkTimeLimit) 
    {
      *fLog << warn << "WARNING: Abs(Fitted Rel. Time) is greater than "
            << gkTimeLimit << " in Pixel " << fPixId << endl;
      return kFALSE;
    }

  if (fErrTime > gkTimeErrLimit) 
    {
      *fLog << warn << "WARNING: Error of Fitted Time is smaller than "
            << gkTimeErrLimit << " in Pixel " << fPixId << endl;
      return kFALSE;
    }

  if (!fHist->IsTimeFitOK()) 
    {
      *fLog << warn << "WARNING: Probability of Fitted Time too low in Pixel " 
            << fPixId << endl;
      return kFALSE;
    }

  if (TESTBIT(fFlags,kHiGainSaturation))
    {

      if (fAbsTimeMean < (Float_t)fTimeFirstLoGain+1)
        {
          *fLog << warn 
                << "WARNING: Some absolute times smaller than limit in Pixel " 
                << fPixId << " time: " << fAbsTimeMean 
                << " Limit: " << (Float_t)fTimeFirstLoGain+1. << endl;
          return kFALSE;
        }

      if (fAbsTimeMean > (Float_t)fTimeLastLoGain-1)
        {
          *fLog << warn 
                << "WARNING: Some absolute times bigger than limit in Pixel " 
                << fPixId << " time: " << fAbsTimeMean 
                << " Limit: " << (Float_t)fTimeLastLoGain-1. << endl;
          return kFALSE;
        }

    }
  else
    {

      if (fAbsTimeMean < (Float_t)fTimeFirstHiGain+1.)
        {
          *fLog << warn 
                << "WARNING: Some absolute times smaller than limit in Pixel " 
                << fPixId << " time: " << fAbsTimeMean 
                << " Limit: " << (Float_t)fTimeFirstHiGain+1. << endl;
          //          return kFALSE;
        }

      if (fAbsTimeMean > (Float_t)fTimeLastHiGain-1.)
        {
          *fLog << warn 
                << "WARNING: Some absolute times bigger than limit in Pixel " 
                << fPixId << " time: " << fAbsTimeMean 
                << " Limit: " << (Float_t)fTimeLastHiGain-1. << endl;
          //          return kFALSE;
        }

    }



  return kTRUE;
}


//
// The check returns kTRUE if:
//
// 
//
Bool_t MCalibrationPix::CheckOscillations()
{


  return kTRUE;
}



void MCalibrationPix::ApplyLoGainConversion()
{
  
  const Float_t chargeRelErrSquare     =   fErrCharge*fErrCharge
                                         /(   fCharge *  fCharge);
  const Float_t sigmaRelErrSquare      =   fErrSigmaCharge*fErrSigmaCharge
                                         /(   fSigmaCharge *  fSigmaCharge);
  const Float_t conversionRelErrSquare =   fConversionHiLoError*fConversionHiLoError 
                                         /(fConversionHiLo    * fConversionHiLo);
  
  fCharge         *= fConversionHiLo;
  fErrCharge       = TMath::Sqrt(chargeRelErrSquare + conversionRelErrSquare) * fCharge;
  
  fSigmaCharge    *= fConversionHiLo;
  fErrSigmaCharge =  TMath::Sqrt(sigmaRelErrSquare + conversionRelErrSquare) * fSigmaCharge;
  
}



// --------------------------------------------------------------------------
//
// 1) Fit the arrival times
// 2) Retrieve the results
//
// This fit has to be done AFTER the Charges fit, 
// otherwise only the Hi Gain will be fitted, even if there are no entries
//
// 
Bool_t MCalibrationPix::FitTime() 
{


  if(!fHist->FitTime())
    {
      *fLog << warn << "WARNING: Could not fit relative times of pixel " << fPixId << endl;
      return kFALSE;
    }
  
  fTime          = fHist->GetRelTimeMean();
  fErrTime       = fHist->GetRelTimeMeanErr();
  fSigmaTime     = fHist->GetRelTimeSigma();
  fTimeProb      = fHist->GetRelTimeProb();

  fAbsTimeMean    = fHist->GetAbsTimeMean();
  fAbsTimeMeanErr = fHist->GetAbsTimeMeanErr();
  fAbsTimeRms     = fHist->GetAbsTimeRms();  

  if (CheckTimeFitValidity())
    SETBIT(fFlags,kTimeFitValid);
  else
    CLRBIT(fFlags,kTimeFitValid);

  return kTRUE;
}

