/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 12/2002 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MHPixVsTime
//
// A histogram to store the sum of camera events. This can be photons,
// currents or enything else derived from MPixVsTime
//
/////////////////////////////////////////////////////////////////////////////
#include "MHPixVsTime.h"

#include <TCanvas.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"
#include "MCamEvent.h"

#include "MGeomCam.h"

#include "MRawEvtHeader.h"
#include "MTime.h"

ClassImp(MHPixVsTime);

using namespace std;

// --------------------------------------------------------------------------
//
// Initialize the name and title of the task.
// Resets the sum histogram
//
MHPixVsTime::MHPixVsTime(Int_t idx, const char *name, const char *title)
    : fIndex(idx), fEvt(NULL), fType(0)
{
    //
    //   set the name and title of this object
    //
    fName  = name  ? name  : "MHPixVsTime";
    fTitle = title ? title : "Average of MPixVsTimes";

    TString t("Pixel Index #");
    t += idx;
    t += " vs Time";

    fGraph->SetName("MCamEvent");
    fGraph->SetTitle(t);
}

// --------------------------------------------------------------------------
//
// Delete the corresponding camera display if available
//
MHPixVsTime::~MHPixVsTime()
{
}

// --------------------------------------------------------------------------
//
// Get the event (MPixVsTime) the histogram might be filled with. If
// it is not given, it is assumed, that it is filled with the argument
// of the Fill function.
// Looks for the camera geometry MGeomCam and resets the sum histogram.
//
Bool_t MHPixVsTime::SetupFill(const MParList *plist)
{
    fEvt = (MCamEvent*)plist->FindObject(fNameEvt, "MCamEvent");
    if (!fEvt)
    {
        if (!fNameEvt.IsNull())
        {
            *fLog << err << GetDescriptor() << ": No " << fNameEvt <<" [MCamEvent] available..." << endl;
            return kFALSE;
        }
        *fLog << warn << GetDescriptor() << ": No MCamEvent available..." << endl;
    }

    fCam = (MGeomCam*)plist->FindObject("MGeomCam");
    if (!fCam)
    {
        *fLog << err << "MGeomCam not found... aborting." << endl;
        return kFALSE;
    }

    if (!fNameTime.IsNull())
    {
        fTime = (MTime*)plist->FindObject(fNameTime, "MTime");
        if (!fTime)
        {
            *fLog << err << fNameTime << " [MTime] not found... abort." << endl;
            return kFALSE;
        }
    }
    else
    {
        fHeader = (MRawEvtHeader*)plist->FindObject("MRawEvtHeader");
        if (!fHeader)
        {
            *fLog << err << "MRawEvtHeader not found... abort." << endl;
            return kFALSE;
        }
    }

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Fill the histograms with data from a MPixVsTime-Container.
//
Bool_t MHPixVsTime::Fill(const MParContainer *par, const Stat_t w)
{
    const MCamEvent *evt = par ? dynamic_cast<const MCamEvent*>(par) : fEvt;
    if (!evt)
    {
        *fLog << err << dbginf << "No MCamEvent found..." << endl;
        return kFALSE;
    }

    Double_t val = 0;
    evt->GetPixelContent(val, fIndex, *fCam, fType);
    if (TMath::IsNaN(val))
        return kCONTINUE;

    Double_t t = 0;
    if (!fNameTime.IsNull())
        t = fTime->GetAxisTime();
    else
        t = fHeader ? fHeader->GetDAQEvtNumber() : fGraph->GetN();

    fGraph->SetPoint(fGraph->GetN(), t, val);
    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Scale the sum container with the number of entries
//
Bool_t MHPixVsTime::Finalize()
{
    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Return fSum.
//
TH1 *MHPixVsTime::GetHistByName(const TString name)
{
    return fGraph->GetHistogram();
}

void MHPixVsTime::Draw(Option_t *opt)
{

    if (fGraph->GetN()==0)
        return;

    TVirtualPad *pad = gPad ? gPad : MakeDefCanvas(this);
    pad->SetBorderMode(0);

    AppendPad("");

    TString str(opt);

    fGraph->GetHistogram()->SetXTitle("Time");
    fGraph->GetHistogram()->GetXaxis()->SetTimeFormat("%H:%M:%S %F1995-01-01 00:00:00");
    fGraph->GetHistogram()->GetXaxis()->SetTimeDisplay(1);
    fGraph->GetHistogram()->GetXaxis()->SetLabelSize(0.033);

    fGraph->GetHistogram()->SetYTitle("");

    if (!str.Contains("A"))
        str += "A";
    if (!str.Contains("P"))
        str += "P";

    if (str.Contains("same", TString::kIgnoreCase))
    {
        str.ReplaceAll("same", "");
        str.ReplaceAll("A", "");
    }

    fGraph->Draw(str);

    pad->Modified();
    pad->Update();

}
