/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug 02/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
//  MHPedestalPix
//
//  Histogram class for pedestal analysis. Holds the histogrammed pedestals, 
//  derives from MHGausEvents, perform Fourier analysis
//
//////////////////////////////////////////////////////////////////////////////
#include "MHPedestalPix.h"

#include <TH1.h>

ClassImp(MHPedestalPix);

using namespace std;
//
const Int_t   MHPedestalPix::fgChargeNbins    = 450 ;
const Axis_t  MHPedestalPix::fgChargeFirst    = -0.5;
const Axis_t  MHPedestalPix::fgChargeLast     = 449.5;
// --------------------------------------------------------------------------
//
// Default Constructor. 
//
MHPedestalPix::MHPedestalPix(const char *name, const char *title) 
    : fPixId(-1)
{ 

  fName  = name  ? name  : "MHPedestalPix";
  fTitle = title ? title : "Histogrammed Pedestal events";

  SetChargeNbins();
  SetChargeFirst();
  SetChargeLast();

  // Create a large number of bins, later we will rebin
  fHGausHist.SetName("HPedestalCharge");
  fHGausHist.SetTitle("Distribution of Summed FADC Pedestal Slices Pixel ");
  fHGausHist.SetXTitle("Sum FADC Slices");
  fHGausHist.SetYTitle("Nr. of events");
  fHGausHist.Sumw2();

}

MHPedestalPix::~MHPedestalPix()
{
}


void MHPedestalPix::InitBins()
{

  fHGausHist.SetBins(fChargeNbins,fChargeFirst,fChargeLast);

}


void MHPedestalPix::ChangeHistId(Int_t id)
{

  fPixId = id;

  fHGausHist.SetName(Form("%s%d", fHGausHist.GetName(), id));
  fHGausHist.SetTitle(Form("%s%d", fHGausHist.GetTitle(), id));
}


void MHPedestalPix::Renorm(const Float_t nslices)
{

  if (!IsGausFitOK())
    return;

  Float_t sqslices = TMath::Sqrt(nslices);
  
  SetMean(GetMean()/nslices);
  //
  // Mean error goes with PedestalRMS/Sqrt(entries) -> scale with slices
  // 
  SetMeanErr(GetMeanErr()/nslices);
  //
  // Sigma goes like PedestalRMS -> scale with sqrt(slices)    
  //
  SetSigma(GetSigma()/sqslices);
  //
  // Sigma error goes like PedestalRMS/2.(entries) -> scale with slices
  //
  SetSigmaErr(GetSigmaErr()/nslices);
  
}

