/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Javier Lopez 12/2003 <mailto:jlopez@ifae.es>
!   Author(s): Javier Rico  01/2004 <mailto:jrico@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
//   MCalibrate
//
//   This task takes the integrated charge from MExtractedSignal and apply
//   the calibration constants from MCalibraitionCam to the charge. Then
//   stores number of photons obtained in MCerPhotEvt. Selection of different
//   calibration methods is allowed through SetCalibrationMode member function
//
//   Input Containers:
//    MExtractedSingal
//    MCalibrationCam
//
//   Output Containers:
//    MCerPhotEvt
//
//////////////////////////////////////////////////////////////////////////////
#include "MCalibrate.h"
#include "MCalibrationConfig.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"
#include "MH.h"

#include "MGeomCam.h"

#include "MCalibrationCam.h"
#include "MCalibrationPix.h"

#include "MExtractedSignalCam.h"
#include "MExtractedSignalPix.h"

#include "MCerPhotEvt.h"

ClassImp(MCalibrate);

using namespace std;
// --------------------------------------------------------------------------
//
// Default constructor. 
//
MCalibrate::MCalibrate(CalibrationMode_t calmode,const char *name, const char *title) : fCalibrationMode(calmode)
{
    fName  = name  ? name  : "MCalibrate";
    fTitle = title ? title : "Task to calculate the number of photons in one event";
}

// --------------------------------------------------------------------------
//
// The PreProcess searches for the following input containers:
//  - MGeomCam
//  - MCalibrationCam
//  - MExtractedSignalCam
//
// The following output containers are also searched and created if
// they were not found:
//
//  - MCerPhotEvt
//
Int_t MCalibrate::PreProcess(MParList *pList)
{

    fSignals = (MExtractedSignalCam*)pList->FindObject(AddSerialNumber("MExtractedSignalCam"));

    if (!fSignals)
    {
      *fLog << err << AddSerialNumber("MExtractedSignalCam") << " not found ... aborting" << endl;
        return kFALSE;
    }

   if(fCalibrationMode>kNone)
      {

	fCalibrations = (MCalibrationCam*)pList->FindObject(AddSerialNumber("MCalibrationCam"));
	if (!fCalibrations)
	  {
	    *fLog << err << AddSerialNumber("MCalibrationCam") << " not found ... aborting." << endl;
	    return kFALSE;
	  }

      }

    fCerPhotEvt = (MCerPhotEvt*)pList->FindCreateObj(AddSerialNumber("MCerPhotEvt"));
    if (!fCerPhotEvt)
        return kFALSE;
    
    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Check for validity of the selected calibration method, switch to a 
// different one in case of need
//
Bool_t MCalibrate::ReInit(MParList *pList)
{

  if(fCalibrationMode == kBlindPixel && !fCalibrations->IsBlindPixelMethodValid())
    {
      *fLog << warn << GetDescriptor() << "Warning: Blind pixel calibration method not valid, switching to F-factor method" << endl;
      fCalibrationMode = kFfactor;
    }

  if(fCalibrationMode == kPinDiode && !fCalibrations->IsPINDiodeMethodValid())
    { 
      *fLog << warn << GetDescriptor() << "Warning: PIN diode calibration method not valid, switching to F-factor method" << endl;
      fCalibrationMode = kFfactor;
    }

  switch(fCalibrationMode)
    {
    case kBlindPixel:
      break;
    case kFfactor:
      break;
    case kPinDiode:
      *fLog << err << GetDescriptor() 
                    << ": PIN Diode Calibration mode not yet available " << endl;
      return kFALSE;
      break;
    case kCombined:
      *fLog << err << GetDescriptor() 
                    << ": Combined Calibration mode not yet available " << endl;
      return kFALSE;
      break;
    case kDummy:
      *fLog << warn << GetDescriptor() 
                    << ": WARNING: Dummy calibration, no calibration applied!!" << endl;
      break;
    default:
      *fLog << warn << GetDescriptor() 
            << ": WARNING: Calibration mode value ("
            <<fCalibrationMode<<") not known" << endl;
      return kFALSE;
    }
  
  return kTRUE;
}
// --------------------------------------------------------------------------
//
// Apply the calibration factors to the extracted signal according to the 
// selected calibration method
//
Int_t MCalibrate::Process()
{
  /*
    if (fCalibrations->GetNumPixels() != (UInt_t)fSignals->GetSize())
    {
        // FIXME: MExtractedSignal must be of variable size -
        //        like MCerPhotEvt - because we must be able
        //        to reduce size by zero supression
        //        For the moment this check could be done in ReInit...
        *fLog << err << "MExtractedSignal and MCalibrationCam have different sizes... abort." << endl;
        return kFALSE;
    }
  */

  UInt_t npix = fSignals->GetSize();

  Float_t hiloconv      = 1.;
  Float_t hiloconverr   = 0.;
  Float_t calibrationConversionFactor    = 1.;
  Float_t calibrationConversionFactorErr = 0.;
  
  for (UInt_t pixidx=0; pixidx<npix; pixidx++)
    {
      if(fCalibrationMode!=kNone)
	{
	  MCalibrationPix &pix = (*fCalibrations)[pixidx];       
	  
	  if (!pix.IsChargeValid())
	    continue;
	  
	  hiloconv   = pix.GetConversionHiLo();
	  hiloconverr= pix.GetConversionHiLoErr();
	  
	  switch(fCalibrationMode)
	    {
	    case kBlindPixel:
	      calibrationConversionFactor      = pix.GetMeanConversionBlindPixelMethod();
	      calibrationConversionFactorErr   = pix.GetConversionBlindPixelMethodErr();
	      break;
	    case kPinDiode:
	      calibrationConversionFactor      = pix.GetMeanConversionPINDiodeMethod();
	      calibrationConversionFactorErr   = pix.GetConversionPINDiodeMethodErr();
	      break;
	    case kFfactor:
	      calibrationConversionFactor      = pix.GetMeanConversionFFactorMethod();
	      calibrationConversionFactorErr   = pix.GetConversionFFactorMethodErr();
	      break;
	    case kCombined:
	      calibrationConversionFactor      = pix.GetMeanConversionCombinedMethod();
	      calibrationConversionFactorErr   = pix.GetConversionCombinedMethodErr();
	      break;
            case kDummy:
	      calibrationConversionFactor      = 1.;
	      calibrationConversionFactorErr   = 0.;
              break;
	    }
	}
      
      MExtractedSignalPix &sig =  (*fSignals)[pixidx];
      
      Float_t signal;
      Float_t signalErr = 0.;
      Float_t nphot,nphotErr;
            
      if (sig.IsLoGainUsed())
        {
          signal    = sig.GetExtractedSignalLoGain()*hiloconv;
          signalErr = signal*hiloconverr;
        }
      else
        {
	  if (sig.GetExtractedSignalHiGain() > 9999.)
	    {
	      signal = 0.;
	      signalErr = 0.;
	    }
	  else
	    signal = sig.GetExtractedSignalHiGain();
        }
      
      nphot    = signal*calibrationConversionFactor;
      nphotErr = signal*calibrationConversionFactorErr
        	*signal*calibrationConversionFactorErr
	       + signalErr*calibrationConversionFactor
        	*signalErr*calibrationConversionFactor;
      
      nphotErr  = TMath::Sqrt(nphotErr);
      
      MCerPhotPix *cpix = fCerPhotEvt->AddPixel(pixidx, nphot, nphotErr);

      if (sig.GetNumLoGainSaturated() > 0)
	cpix->SetPixelSaturated();
    }
  
  fCerPhotEvt->FixSize();
  fCerPhotEvt->SetReadyToSave();
  
  return kTRUE;
}
