/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.

! *
!
!
!   Author(s): Markus Gaug   02/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                               
// MCalibrationQECam                                               
//                                                               
// Hold the calibrated QE information of the camera:
//                                                               
// 1) MCalibrationQECam initializes a TClonesArray whose elements are 
//    pointers to MCalibrationQEPix Containers
//
/////////////////////////////////////////////////////////////////////////////
#include "MCalibrationQECam.h"

#include <TClonesArray.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MCalibrationQEPix.h"

ClassImp(MCalibrationQECam);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor. 
//
// Creates a TClonesArray of MCalibrationPix containers, initialized to 1 entry
// Later, a call to MCalibrationQECam::InitSize(Int_t size) has to be performed
//
// Creates an MCalibrationBlindPix container 
//
MCalibrationQECam::MCalibrationQECam(const char *name, const char *title)
{
    fName  = name  ? name  : "MCalibrationQECam";
    fTitle = title ? title : "Storage container for the calibrated Quantrum Efficiency of the camera";

    fPixels     = new TClonesArray("MCalibrationQEPix",1);

    Clear();
}

// --------------------------------------------------------------------------
//
MCalibrationQECam::~MCalibrationQECam()
{

  //
  // delete fPixels should delete all Objects stored inside
  // 
  delete fPixels;

}

// -------------------------------------------------------------------
//
void MCalibrationQECam::InitSize(const UInt_t i)
{
  fPixels->ExpandCreate(i);
}

// --------------------------------------------------------------------------
//
// This function returns the current size of the TClonesArray 
// independently if the MCalibrationPix is filled with values or not.
//
// It is the size of the array fPixels.
//
Int_t MCalibrationQECam::GetSize() const
{
  return fPixels->GetEntriesFast();
}


// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
MCalibrationQEPix &MCalibrationQECam::operator[](UInt_t i)
{
  return *static_cast<MCalibrationQEPix*>(fPixels->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
const MCalibrationQEPix &MCalibrationQECam::operator[](UInt_t i) const
{
  return *static_cast<MCalibrationQEPix*>(fPixels->UncheckedAt(i));
}


// --------------------------------------
//
void MCalibrationQECam::Clear(Option_t *o)
{

  fPixels->ForEach(TObject, Clear)();

  return;
}


// --------------------------------------------------------------------------
//
// Print first the well fitted pixels 
// and then the ones which are not FitValid
//
void MCalibrationQECam::Print(Option_t *o) const
{

  *fLog << all << GetDescriptor() << ":" << endl;
  int id = 0;
  
  TIter Next(fPixels);
  MCalibrationQEPix *pix;
  while ((pix=(MCalibrationQEPix*)Next()))
    {
      
      if (!pix->IsExcluded() && pix->IsQEValid()) 
	{
	    *fLog << all << "Pix " << pix->GetPixId() 
		<< ":  QE: "                   << pix->GetQE(kCT1)        << " +- " << pix->GetQEErr(kCT1) 
		<< endl;
          id++;
	}
    }
  
  *fLog << all << id << " succesful pixels :-))" << endl;
  id = 0;
  
  *fLog << all << endl;
  *fLog << all << "Pixels with errors:" << endl;
  *fLog << all << endl;
  
  TIter Next2(fPixels);
    while ((pix=(MCalibrationQEPix*)Next2()))
      {
        
        if (!pix->IsExcluded() && !pix->IsQEValid())
          {
	    *fLog << all << "Pix " << pix->GetPixId() 
		<< ":  QE: "                   << pix->GetQE(kCT1)        << " +- " << pix->GetQEErr(kCT1) 
		<< endl;
            id++;
          }
      }
    *fLog << all << id << " pixels with errors :-((" << endl;

    
  *fLog << all << endl;
  *fLog << all << "Excluded pixels:" << endl;
  *fLog << all << endl;
  
  id = 0;

  TIter Next4(fPixels);
  while ((pix=(MCalibrationQEPix*)Next4()))
  {
      if (pix->IsExcluded())
      {
	  *fLog << all << pix->GetPixId() << endl;
	  id++;
      }
  }
  *fLog << all << id << " Excluded pixels " << endl;
}

//
Bool_t MCalibrationQECam::GetPixelContent(Double_t &val, Int_t idx, const MGeomCam &cam, Int_t type) const
{

  if (idx > GetSize())
    return kFALSE;

  switch (type)
    {
    case 0:
      if ((*this)[idx].IsExcluded())
        return kFALSE;
      val = (*this)[idx].GetQE(kCT1);
      break;
    case 1:
      if ((*this)[idx].IsExcluded())
        return kFALSE;
      val = (*this)[idx].GetQEErr(kCT1);
      break;
    default:
      return kFALSE;
    }
  return val!=-1.;
}

// --------------------------------------------------------------------------
//
// What MHCamera needs in order to draw an individual pixel in the camera
//
void MCalibrationQECam::DrawPixelContent(Int_t idx) const
{
    return;
}









