/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 <mailto:tbretz@uni-sw.gwdg.de>
!   Author(s): Harald Kornmayer 1/2001
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MGeomCam
//
// This is the base class of different camera geometries. It creates
// a pixel object for a given number of pixels and defines the
// interface of how to acccess the geometry information.
//
// We use a TObjArray for possible future usage (it is much more flexible
// than a TClonesArray so that it can store more types of pixels (eg
// fake pixels which are not really existing)
//
// Version 1:
// ----------
//  - first implementation
//
// Version 2:
// ----------
//  - added fPixRatio
//  - added fPixRatioSqrt
//
/////////////////////////////////////////////////////////////////////////////
#include "MGeomCam.h"

#include <TClass.h>     // IsA()->New()

#include "MLog.h"
#include "MLogManip.h"

#include "MGeomPix.h"

ClassImp(MGeomCam);

using namespace std;

// --------------------------------------------------------------------------
//
// Default Constructor
//
MGeomCam::MGeomCam()
    : fNumPixels(0), fCamDist(0), fConvMm2Deg(0)
{
}

// --------------------------------------------------------------------------
//
// Initializes a Camera Geometry with npix pixels. All pixels
// are deleted when the corresponding array is deleted.
//
MGeomCam::MGeomCam(UInt_t npix, Float_t dist, const char *name, const char *title)
    : fNumPixels(npix), fCamDist(dist), fConvMm2Deg(kRad2Deg/(dist*1000)), fPixels(npix), fPixRatio(npix), fPixRatioSqrt(npix)
{
    fName  = name  ? name  : "MGeomCam";
    fTitle = title ? title : "Storage container for  a camera geometry";

    //
    // make sure that the destructor delete all contained objects
    //
    fPixels.SetOwner();

    for (UInt_t i=0; i<npix; i++)
      {
	MGeomPix* dummy = new MGeomPix();
	fPixels.Add(dummy);
      }

    SetReadyToSave();
}

// --------------------------------------------------------------------------
//
// Returns a reference of the i-th entry (the pixel with the software idx i)
// The access is unchecked (for speed reasons!) accesing non existing
// entries may crash the program!
//
MGeomPix &MGeomCam::operator[](Int_t i)
{
    return *static_cast<MGeomPix*>(fPixels.UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Returns a reference of the i-th entry (the pixel with the software idx i)
// The access is unchecked (for speed reasons!) accesing non existing
// entries may crash the program!
//
MGeomPix &MGeomCam::operator[](Int_t i) const
{
    return *static_cast<MGeomPix*>(fPixels.UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Calculate and fill the arrays storing the ratio of the area of a pixel
// i to the pixel 0 and its square root.
// The precalculation is done for speed reasons. Having an event the
// ratio would be calculated at least once for each pixel which is
// an enormous amount of numerical calculations, which are time
// consuming and which can be avoided doing the precalculation.
//
void MGeomCam::CalcPixRatio()
{
    const Double_t a0 = (*this)[0].GetA();

    for (UInt_t i=0; i<fNumPixels; i++)
    {
        fPixRatio[i] = a0/(*this)[i].GetA();
        fPixRatioSqrt[i] = TMath::Sqrt(fPixRatio[i]);
    }
}

// --------------------------------------------------------------------------
//
//  Set the kIsOuterRing flag for all pixels which have a outermost pixel
//  as Next Neighbor and don't have the kIsOutermostRing flag itself.
//
void MGeomCam::InitOuterRing()
{
    fPixels.ForEach(MGeomPix, CheckOuterRing)(*this);
}

// --------------------------------------------------------------------------
//
// Calculate the highest sector index+1 of all pixels, please make sure
// the the sector numbers are continous.
//
void MGeomCam::CalcNumSectors()
{
    fNumSectors = 0;

    for (UInt_t i=0; i<fNumPixels; i++)
    {
        const UInt_t s = (*this)[i].GetSector();

        if (s>fNumSectors)
            fNumSectors = s;
    }

    fNumSectors++;
}

// --------------------------------------------------------------------------
//
// Calculate the maximum radius of the camera. This is ment for GUI layout.
//
void MGeomCam::CalcMaxRadius()
{
    fMaxRadius = 0;

    for (UInt_t i=0; i<fNumPixels; i++)
    {
        const MGeomPix &pix = (*this)[i];

        const Float_t x = pix.GetX();
        const Float_t y = pix.GetY();
        const Float_t d = pix.GetD();

        const Float_t maxr = sqrt(x*x+y*y) + d;

        if (maxr>fMaxRadius)
            fMaxRadius = maxr;
    }
}

// --------------------------------------------------------------------------
//
//  returns the ratio of the area of the pixel with index 0 to the pixel
//  with the specified index i. 0 Is returned if the index argument is
//  out of range.
//
Float_t MGeomCam::GetPixRatio(UInt_t i) const
{
    // Former: (*this)[0].GetA()/(*this)[i].GetA();
    // The const_cast is necessary to support older root version
    return i<fNumPixels ? const_cast<TArrayF&>(fPixRatio)[i] : 0;
}

// --------------------------------------------------------------------------
//
//  returns the square root of the ratio of the area of the pixel with
//  index 0 to the pixel with the specified index i. 0 Is returned if
//  the index argument is out of range.
//
Float_t MGeomCam::GetPixRatioSqrt(UInt_t i) const
{
    // The const_cast is necessary to support older root version
    return i<fNumPixels ? const_cast<TArrayF&>(fPixRatioSqrt)[i] : 0;
}

// --------------------------------------------------------------------------
//
//  Prints the Geometry information of all pixels in the camera
//
void MGeomCam::Print(Option_t *) const
{
    //
    //   Print Information about the Geometry of the camera
    //
    *fLog << all << " Number of Pixels (" << GetTitle() << "): " << fNumPixels << endl;

    fPixels.Print();
} 

// --------------------------------------------------------------------------
//
//  Create a clone of this container. This is very easy, because we
//  simply have to create a new object of the same type.
//
TObject *MGeomCam::Clone(const char *newname) const
{
    return (TObject*)IsA()->New();
}
