// MagicMirror.cpp: Implementierung der Klasse MagicMirror.
//
//////////////////////////////////////////////////////////////////////

#include "amcmirrorpanel.h"
#include "amcstate.h"
#include "magicmirror.h"
#include <qfile.h>
#include <qmessagebox.h>
#include <qtextstream.h>

#define PANELFILENAME "/home/amc/Panels.txt"
#define POSFILENAME   "/home/amc/PanelPos.txt"

extern AMCState*	g_pAMCState;

//////////////////////////////////////////////////////////////////////
// Konstruktion/Destruktion
//////////////////////////////////////////////////////////////////////

MagicMirror::MagicMirror()
{
	readPanelDescription();
	readPanelPositions();
}

MagicMirror::~MagicMirror()
{
	/* Make sure that when the list is deleted all AMCMirrorPanels
	 * are also deleted.
   */
  for (int i=0; i<8; i++)
		m_panelList[i].setAutoDelete( true );
}

/** Read he panel descriptions from the file "Panels.txt".
	This file contains for each panel a line with the following contents:
	i	j	type	port	box		driver	installed
	
	i: is the first panel index (counting from left to right)
	j: is the second panel index (counting from bottom to top)
	type: the panel type (see the file for a description of types)
	port: the serial port this box is connected to.
		  (The numbering corresponds to the numbers on the cables.
		   If only 1 Meilhaus ME9100 card is installed on linux boxes
		   we have the following correspondance to device files
		   port 1 == /dev/ttyS4 ... port 8 = /dev/ttyS11)
	box: the number of the AMC box.
	driver: The internal AMC driver to which this panel is connected.
	installed: Flag indicating if this panel is installed (0=no, 1=yes)
*/

void MagicMirror::readPanelDescription()
{
	
	m_iNumPanels = 0;
	QFile file( "/home/amc/Panels.txt" );
	if( file.open( IO_ReadOnly ) )
	{
		QTextStream stream( &file );
		QString line;
		while( !stream.eof() )
		{
			line = stream.readLine();
			if( line[0] == '#')
				continue;
			int ii, ij, iType, iPort, iBox, iDriver, iInst;
			sscanf( line.data(),"%d %d %d %d %d %d %d",
								&ii, &ij, &iType, &iPort,
								&iBox, &iDriver, &iInst );
			AMCMirrorPanel* pPanel = new AMCMirrorPanel( ii, ij, iType, iPort, iBox, iDriver );
			pPanel->setInstalled(iInst != 0);
			m_arrPanels[ii+8][ij+8] = pPanel;
			if(iInst != 0)
			{
				m_iNumPanels++;
				m_panelList[iPort-1].append( pPanel );
				m_allPanelList.append( pPanel );
			}
		}
		g_pAMCState->setInstalledPanels(m_iNumPanels);
	}
	else
		qDebug("Can't open file Panels.txt");
	file.close();
}

/** Read the stored panel positions from the file "PanelPos.txt"
	This file contains for each panel a line with the following contents:
	i	j	X	Y
	
	i: is the first panel index (counting from left to right)
	j: is the second panel index (counting from bottom to top)
	X: position of the X motor relative to the center position
	Y: position of the Y motor relative to the center position
 */
void MagicMirror::readPanelPositions()
{
	QFile file( "/home/amc/PanelPos.txt" );
	if( file.open( IO_ReadOnly ) )
	{
		QTextStream stream( &file );
		QString line;
		while( !stream.eof() )
		{
			line = stream.readLine();
			
			// Skip comment lines
			if( line[0] == '#')
				continue;
				
			int ii, ij, iX, iY, iLaserX, iLaserY;
			double dAxisX, dSlopeX, dConvX;
			double dAxisY, dSlopeY, dConvY;
			sscanf( line.data(),"%d %d %d %d %d %d %lf %lf %lf %lf %lf %lf",
								&ii, &ij, &iX, &iY, &iLaserX, &iLaserY,
								&dAxisX, &dSlopeX, &dConvX, &dAxisY, &dSlopeY, &dConvY );
      if( dAxisX == 0.0)
			{
        dSlopeX = 0.0;
        dConvX = 0.0;
				dAxisY = 0.0;
        dSlopeY = 0.0;
        dConvY = 0.0;
 			}
			m_arrPanels[ii+8][ij+8]->setRefX( iX );
			m_arrPanels[ii+8][ij+8]->setRefY( iY );
			m_arrPanels[ii+8][ij+8]->setLaserX( iLaserX );
			m_arrPanels[ii+8][ij+8]->setLaserY( iLaserY );
			m_arrPanels[ii+8][ij+8]->setAxisX( dAxisX );
			m_arrPanels[ii+8][ij+8]->setSlopeX( dSlopeX );
			m_arrPanels[ii+8][ij+8]->setConversionX( dConvX );
			m_arrPanels[ii+8][ij+8]->setAxisY( dAxisY );
			m_arrPanels[ii+8][ij+8]->setSlopeY( dSlopeY );
			m_arrPanels[ii+8][ij+8]->setConversionY( dConvY );
		}
	}
	else
		qDebug("Can't open file PanelsPos.txt");

	file.close();
	
	return;
}

/** Save the actual panel positions to the file "PanelPos.txt"
	This file contains for each panel a line with the following contents:
	i	j	X	Y
	
	i: is the first panel index (counting from left to right)
	j: is the second panel index (counting from bottom to top)
	X: position of the X motor relative to the center position
	Y: position of the Y motor relative to the center position
 */
void MagicMirror::saveMirrorPanels()
{
	QCString line;
	
	QFile file( POSFILENAME );
	if( file.open( IO_WriteOnly ) )
	{
		QTextStream stream( &file );
		for(int j=0; j<17; j++)
		{
			line.sprintf("# Column %d\n",j-8);
			stream << line;
			for(int i=0; i<17; i++)
			{
				AMCMirrorPanel* pPanel = m_arrPanels[i][j];
				line.sprintf( " %2d %2d\t%5d\t%5d\t%5d\t%5d\t%9.2f\t%7.2f\t%10.5f\t%9.2f\t%7.2f\t%10.5f\n",
											pPanel->i(),
											pPanel->j(),
											pPanel->getRefX(),
											pPanel->getRefY(),
											pPanel->getLaserX(),
											pPanel->getLaserY(),
											pPanel->getAxisX(),
											pPanel->getSlopeX(),
											pPanel->getConversionX(),
											pPanel->getAxisY(),
											pPanel->getSlopeY(),
											pPanel->getConversionY() );
				stream << line;
			}
		}
					
	}
	else
	{
  	int iRes = QMessageBox::critical( 0,
 										 "Can't open panel position description file",
										 "The programm tried to open the file \'PanelPos.txt\'\n"
										 "in the directory \'/home/amc/\'.\n"
										 "The file may not have the right permissions.\n",
										 "Continue" );
	}
	file.close();
					
}

