#ifndef MARS_MAstroCatalog
#define MARS_MAstroCatalog

#ifndef ROOT_TVector3
#include <TVector3.h>
#endif
#ifndef ROOT_TList
#include <TList.h>
#endif

class MTime;
class MObservatory;
class TArrayI;

class MVector3 : public TVector3
{
    enum VectorType_t
    {
        kIsInvalid,
        kIsRaDec,
        kIsZdAz,
        kIsAltAz,
        kIsArbitrary
    };

    VectorType_t fType;

public:
    /*
     MVector3(Double_t theta=0, Double_t phi=0, Double_t mag=1)
     {
     SetMagThetaPhi(exp(mag), theta, phi);
     }*/
    MVector3() { fType=kIsInvalid; }
    MVector3(const TVector3 &v3) : TVector3(v3) { fType=kIsArbitrary; }
    Double_t Magnitude() const { return TMath::Log(Mag()); }

    void SetRaDec(Double_t ra, Double_t dec, Double_t mag)
    {
        fType = kIsRaDec;
        SetMagThetaPhi(exp(mag), TMath::Pi()/2-dec, ra);
    }
    void SetZdAz(Double_t zd, Double_t az, Double_t mag)
    {
        fType = kIsZdAz;
        SetMagThetaPhi(exp(mag), zd, az);
    }
    void SetAltAz(Double_t alt, Double_t az, Double_t mag)
    {
        fType = kIsAltAz;
        SetMagThetaPhi(exp(mag), TMath::Pi()/2-alt, az);
    }

    MVector3 GetZdAz(const MObservatory &obs, Double_t gmst) const;
    MVector3 GetZdAz(const MTime &time, MObservatory &obs) const;

    ClassDef(MVector3, 0)
};

class MAstroCatalog : public TObject
{
private:
    Double_t fLimMag;    // [1]   Limiting Magnitude
    Double_t fRadiusFOV; // [deg] Radius of Field of View
    MVector3 fRaDec;     // pointing position

    TList fList;

    TString FindToken(TString &line, Char_t tok=',');

    Int_t   atoi(const TSubString &sub);
    Float_t atof(const TSubString &sub);
    Int_t   atoi(const TString &s);
    Float_t atof(const TString &s);

public:
    MAstroCatalog() : fLimMag(99), fRadiusFOV(99)
    {
        fList.SetOwner();
    }

    void SetLimMag(Double_t mag)             { fLimMag=mag; }
    void SetRadiusFOV(Double_t deg)          { fRadiusFOV=deg; }
    void SetRaDec(Double_t ra, Double_t dec) { fRaDec.SetRaDec(ra, dec, 1); }
    void SetRaDec(const TVector3 &v)         { fRaDec=v; }
    void Delete(Option_t *o="")              { fList.Delete(o); }

    Int_t ReadXephem(TString catalog = "/usr/X11R6/lib/xephem/catalogs/YBS.edb");
    Int_t ReadNGC2000(TString catalog = "ngc2000.dat");
    Int_t ReadBSC(TString catalog = "bsc5.dat");

    void Print(Option_t *o="") const { fList.Print(); }

    TList *GetList() { return &fList; }

    virtual Bool_t Convert(const TRotation &rot, TVector2 &v, Int_t type);
    virtual Bool_t PaintLine(const TVector2 &v0, Double_t dx, Double_t dy, const TRotation &rot, Int_t type);

    void Paint(const TVector2 &v0, const TRotation &rot, TArrayI &dx, TArrayI &dy, Byte_t type);
    void PaintNet(const TVector2 &v0, const TRotation &rot, Int_t type=0);
    void Paint(Option_t *o="");

    ClassDef(MAstroCatalog, 0)
};
#endif
