/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz 1/2004 <mailto:tbretz@astro.uni-wuerzburg.de>
!              Markus Gaug  3/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MBadPixelsCam                                                                                                                              //
//
// Storage container to store bad pixel of the camera...
//
/////////////////////////////////////////////////////////////////////////////
#include "MBadPixelsCam.h"

#include <fstream>

#include <TClonesArray.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MBadPixelsPix.h"

ClassImp(MBadPixelsCam);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor.
//
MBadPixelsCam::MBadPixelsCam(const char *name, const char *title)
{
    fName  = name  ? name  : "MBadPixelsCam";
    fTitle = title ? title : "";

    fArray = new TClonesArray("MBadPixelsPix", 1);
}

// --------------------------------------------------------------------------
//
// Delete the array conatining the bad pixel information
//
MBadPixelsCam::~MBadPixelsCam()
{
    delete fArray;
}

// --------------------------------------------------------------------------
//
// Set the size of the camera
//
void MBadPixelsCam::InitSize(const UInt_t i)
{
    fArray->ExpandCreate(i);
}

// --------------------------------------------------------------------------
//
// Get the size of the MBadPixelsCam
//
Int_t MBadPixelsCam::GetSize() const
{
    return fArray->GetEntriesFast();
}

// --------------------------------------------------------------------------
//
// Copy 'constructor'
//
void MBadPixelsCam::Copy(TObject &object) const
{
    MBadPixelsCam &cam = (MBadPixelsCam&)object;

    const Int_t n = GetSize();
    if (n==0)
        return;

    cam.InitSize(n);
    for (int i=0; i<n; i++)
        (*this)[i].Copy(cam[i]);
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
MBadPixelsPix &MBadPixelsCam::operator[](Int_t i)
{
    return *static_cast<MBadPixelsPix*>(fArray->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
const MBadPixelsPix &MBadPixelsCam::operator[](Int_t i) const
{
    return *static_cast<MBadPixelsPix*>(fArray->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Merge two MBadPixelsCam together, see also MBadPixelsPix::Merge
//
void MBadPixelsCam::Merge(const MBadPixelsCam &cam)
{
    const Int_t n = cam.GetSize();
    if (n==0)
    {
        *fLog << warn << "MBadPixelsCam::Merge: Container empty." << endl;
        return;
    }

    if (GetSize()==0)
        InitSize(n);

    if (n!=GetSize())
    {
        *fLog << warn << "MBadPixelsCam::Merge: Size mismatch... ignored." << endl;
        return;
    }

    for (int i=0; i<n; i++)
        (*this)[i].Merge(cam[i]);
}

// --------------------------------------------------------------------------
//
// Clear the contents of all bad pixels (set=0 means Ok)
//
void MBadPixelsCam::Clear(Option_t *o)
{
    fArray->ForEach(TObject, Clear)();
}

// --------------------------------------------------------------------------
//
// Print the contents of all bad pixels
//
void MBadPixelsCam::Print(Option_t *o) const
{
    *fLog << all << GetDescriptor() << ":" << endl;
    
    *fLog << "Pixels without problems:" << endl;
    *fLog << endl;

    Int_t count = 0;

    for (Int_t i=0; i<GetSize(); i++)
    {
        if (!(*this)[i].IsUnsuitable(MBadPixelsPix::kUnsuitableRun))
        {
            *fLog << i << " ";
            count ++;
        }

        if (count == 0)
            continue;

        if (!(count % 25))
            *fLog << endl;
    }
    *fLog << endl;
    *fLog << count << " normal pixels :-))" << endl;
    *fLog << endl;
    count = 0;


    *fLog << "Pixels unsuited for the whole run:" << endl;
    *fLog << endl;

    for (Int_t i=0; i<GetSize(); i++)
    {
        if ((*this)[i].IsUnsuitable(MBadPixelsPix::kUnsuitableRun))
        {
            *fLog << i << " ";
            count ++;
        }

        if (count == 0)
            continue;

        if (!(count % 25))
            *fLog << endl;
    }
    *fLog << endl;
    *fLog << count << " unsuited pixels :-(" << endl;
    *fLog << endl;

    count = 0;

    *fLog << all << "Pixels unreliable for the whole run:" << endl;
    *fLog << all << endl;

    for (Int_t i=0; i<GetSize(); i++)
    {
        if ((*this)[i].IsUnsuitable(MBadPixelsPix::kUnreliableRun))
        {
            *fLog << i << " ";
            count ++;
        }

        if (count == 0)
            continue;

        if (!(count % 25))
          *fLog << endl;
    }

    *fLog << endl;
    *fLog << count << " unreliable pixels :-(" << endl;
    *fLog << endl;
}

// --------------------------------------------------------------------------
//
// Read from an ascii file of the format:
//    pixel1 pixel2 pixel3 pixel4
// while pixel1,2,3,4 are the pixel indices used in the software.
//
// To read the pixels corresponding to a given run you can give run!=0
// and a file of the format:
//   1234: 17 193 292
//   1235: 17 193 292 293
//
void MBadPixelsCam::AsciiRead(ifstream &fin, UInt_t run=0)
{
    Int_t len;
    TString str;

    while (1)
    {
        str.ReadLine(fin);
        if (!fin)
            return;

        Int_t r;

        const Int_t n = sscanf(str.Data(), " %d : %n", &r, &len);
        if (n!=1)
            return;

        if (run==0 || run && (UInt_t)r==run)
            break;
    }

    str.Remove(0, len);

    while (1)
    {
        Int_t idx;
        const Int_t n = sscanf(str.Data(), " %d %n", &idx, &len);

        if (n!=1)
            break;

        str.Remove(0, len);

        if (idx>=GetSize())
            InitSize(idx+1);

        (*this)[idx].SetUnsuitable(MBadPixelsPix::kUnsuitableRun);
    }
}

// --------------------------------------------------------------------------
//
// Write the information into an ascii file. If a run-number is given the
// run-number will lead the line.
//
Bool_t MBadPixelsCam::AsciiWrite(ostream &fout, UInt_t run=0) const
{
    if (run)
        fout << run << ":";

    for (int i=0; i<GetSize(); i++)
        if ((*this)[i].IsUnsuitable(MBadPixelsPix::kUnsuitableRun))
            fout << " " << i;

    if (run && GetSize())
        fout << endl;

    return kTRUE;
}

Bool_t MBadPixelsCam::GetPixelContent(Double_t &val, Int_t idx, const MGeomCam &cam, Int_t type) const
{

  if (idx >= GetSize())
    return kFALSE;

  switch (type)
    {
    case 0:
      val = (*this)[idx].GetInfo()[0];
      return (*this)[idx].IsOK();
      break;
    case 1:
      val = (*this)[idx].IsUnsuitable(MBadPixelsPix::kUnsuitableRun);
      return val;
      break;
    case 2:
      val = (*this)[idx].IsUnsuitable(MBadPixelsPix::kUnsuitableEvt);
      return val;
      break;
    case 3:
      val = (*this)[idx].IsUnsuitable(MBadPixelsPix::kUnreliableRun);
      return val;
      break;
    case 4:
      val = (*this)[idx].IsHiGainBad();
      return val;
      break;
    case 5:
      val = (*this)[idx].IsLoGainBad();
      return val;
      break;
    case 6:
      val = (*this)[idx].IsCalibrationSignalOK();
      return val;
      break;
    case 7:
      val = (*this)[idx].IsCalibrationFitOK();
      return val;
      break;
    case 8:
      val = (*this)[idx].IsCalibrationOscillating();
      return val;
      break;
    case 9:
      val = (*this)[idx].IsCalibrationResultOK();
      return val;
      break;
    case 10:
      val = (*this)[idx].IsHiGainSaturation();
      return val;
      break;
    case 11:
      val = (*this)[idx].IsLoGainSaturation();
      return val;
      break;
    case 12:
      val = (*this)[idx].IsNotCalibrated();
      return val;
      break;
    case 13:
      val = (*this)[idx].IsMeanTimeInLastBin();
      return val;
      break;
    case 14:
      val = (*this)[idx].IsMeanTimeInFirstBin();      
      return val;
      break;
    case 15:
      val = (*this)[idx].IsLoGainOscillating();
      return val;
      break;
    case 16:
      val = (*this)[idx].IsHiGainOscillating();
      return val;
      break;
    case 17:
      val = (*this)[idx].IsConvHiLoNotValid();
      return val;
      break;
    case 18:
      val = (*this)[idx].IsChargeSigmaNotValid();
      return val;
      break;
    case 19:
      val = (*this)[idx].IsChargeRelErrNotValid();
      return val;
      break;
    case 20:
      val = (*this)[idx].IsChargeErrNotValid();
      return val;
    case 21:
      val = (*this)[idx].IsChargeIsPedestal();
      return val;
    case 22:
      val = (*this)[idx].IsLoGainNotFitted();
      return val;
      break;
    case 23:
      val = (*this)[idx].IsHiGainNotFitted();
      return val;
      break;
    default:
      return kFALSE;
    }
  
  return kFALSE;
}

void MBadPixelsCam::DrawPixelContent(Int_t num) const
{
    *fLog << warn << "MBadPixelsCam::DrawPixelContent - not available." << endl;
}
