/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Robert Wagner  10/2002 <mailto:magicsoft@rwagner.de>
!   Author(s): Thomas Bretz   2/2003 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2002-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MObservatory
//
// BE EXTREMLY CARFEFULL CHANGING THIS CLASS! THE TRACKING SYSTEM IS BASED
// ON IT!
//
/////////////////////////////////////////////////////////////////////////////
#include "MObservatory.h"

#include <TVector3.h>

#include "MTime.h"
#include "MAstro.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MObservatory);

using namespace std;

void MObservatory::Init(const char *name, const char *title)
{
    fName  = name  ? name  : "MObservatory";
    fTitle = title ? title : "Storage container for coordinates of an observatory";   
}

MObservatory::MObservatory(const char *name, const char *title)
{
    Init(name, title);

    SetLocation(kMagic1);
}

MObservatory::MObservatory(LocationName_t key, const char *name, const char *title)
{
    Init(name, title);

    SetLocation(key);
}

// --------------------------------------------------------------------------
//
// BE EXTREMLY CARFEFULL CHANGING THIS CLASS! THE TRACKING SYSTEM IS BASED
// ON IT!
//
void MObservatory::SetLocation(LocationName_t name)
{
    switch (name)
    {
        // BE EXTREMLY CARFEFULL CHANGING THIS CLASS! THE TRACKING SYSTEM
        // IS BASED ON IT!
    case kMagic1:
        // Values taken from the GPS Receiver (avg 20h)
        // on 26/11/2003 at 17h30 in the counting house
        fLatitude  = MAstro::Dms2Rad(28, 45, 42.576, '+');
        fLongitude = MAstro::Dms2Rad(17, 53, 26.460, '-');
        fHeight    = 2196.5; // m
        fObservatoryName = "Observatorio del Roque de los Muchachos (Magic1)";
        break;

    case kWuerzburgCity:
        fLatitude  = MAstro::Dms2Rad(51, 38, 48.0);
        fLongitude = MAstro::Dms2Rad( 9, 56, 36.0);
        fHeight    = 300;
        fObservatoryName = "Wuerzburg City";
        break;
    }

    fSinLatitude = TMath::Sin(fLatitude);
    fCosLatitude = TMath::Cos(fLatitude);
}

void MObservatory::Print(Option_t *) const
{
    *fLog << all;
    *fLog << fObservatoryName << endl;
    *fLog << "Latitude " << (fLatitude > 0 ? (fLatitude*kRad2Deg) : -(fLatitude*kRad2Deg)) << " deg " << (fLatitude > 0 ? "W" : "E") << endl;
    *fLog << "Longitude " << (fLongitude > 0 ? (fLongitude*kRad2Deg) : -(fLongitude*kRad2Deg)) <<" deg " << (fLongitude < 0 ? "N" : "S") << endl;
    *fLog << "Height " << fHeight << "m" << endl;
}

// --------------------------------------------------------------------------
//
// RotationAngle 
// 
// calculates the angle for the rotation of the sky image in the camera;
// this angle is a function of the local coordinates
//
//  theta [rad]: polar angle/zenith distance
//  phi   [rad]: rotation angle/azimuth
//
// Return sin/cos component of angle
//
// calculate rotation angle alpha of sky image in camera
// (see TDAS 00-11, eqs. (18) and (20))
//
void MObservatory::RotationAngle(Double_t theta, Double_t phi, Double_t &sin, Double_t &cos) const
{
    const Double_t sint = TMath::Sin(theta);
    const Double_t cost = TMath::Cos(theta);

    const Double_t sinl = fSinLatitude*sint;
    const Double_t cosl = fCosLatitude*cost;

    const Double_t sinp = TMath::Sin(phi);
    const Double_t cosp = TMath::Cos(phi);

    const Double_t v1 = sint*sinp;
    const Double_t v2 = cosl - sinl*cosp;

    const Double_t denom = TMath::Sqrt(v1*v1 + v2*v2);

    sin = (fCosLatitude*sinp) / denom;
    cos = sinl + cosl*cosp / denom;
}

// --------------------------------------------------------------------------
//
// RotationAngle 
// 
// calculates the angle for the rotation of the sky image in the camera;
// this angle is a function of the local coordinates
//
//  theta [rad]: polar angle/zenith distance
//  phi   [rad]: rotation angle/azimuth
//
// Return RotationAngle in rad
//
// calculate rotation angle alpha of sky image in camera
// (see TDAS 00-11, eqs. (18) and (20))
//
Double_t MObservatory::RotationAngle(Double_t theta, Double_t phi) const
{
    const Double_t sint = TMath::Sin(theta);
    const Double_t cost = TMath::Cos(theta);

    const Double_t sinp = TMath::Sin(phi);
    const Double_t cosp = TMath::Cos(phi);

    const Double_t v1 = sint*sinp;
    const Double_t v2 = fCosLatitude*cost - fSinLatitude*sint*cosp;

    const Double_t denom = TMath::Sqrt(v1*v1 + v2*v2);

    return TMath::ASin((fCosLatitude*sinp) / denom);
}

// --------------------------------------------------------------------------
//
// RotationAngle 
// 
// calculates the angle for the rotation of the sky image in the camera;
// this angle is a function of the sky coordinates, the observatory
// location and the time
//
//  ra  [rad]: Right ascension
//  dec [rad]: Declination
//
// Return RotationAngle in rad
//
Double_t MObservatory::RotationAngle(Double_t ra, Double_t dec, const MTime &t) const
{
    const Double_t alpha = t.GetGmst() + GetElong();

    TVector3 v;
    v.SetMagThetaPhi(1, TMath::Pi()/2-dec, alpha-ra);
    v.RotateY(GetPhi()-TMath::Pi()/2);

    return RotationAngle(v.Theta(), v.Phi());
}
