
/**************************************************************\

  Main program for the CTX G-APD HV supply, sends commands, 
  reads and monitors status of the G-APD HV supply

  Name:         hvcontrol.cpp

  Actions:      Do global initialization, start threads  

  Created by:   Sebastian Commichau, November 2008
                commichau@phys.ethz.ch

\**************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>
#include <signal.h>

#include "ConsoleCommand.h"
#include "CCCommand.h"
#include "ProcessIO.h"
#include "HVMonitor.h"
#include "Types.h"

#define Sleep(x) usleep(x*1000)

#define DEFAULT_CONFIG "HV.conf"     // Default configuration file
#define LOCKFILE "/tmp/CTX_HV_LOCK"

void CrashHandler(int Signal);


int main(int argc, char *argv[]) {


  char config_file[] = DEFAULT_CONFIG, str[MAX_COM_SIZE];

  pthread_t thread_ConsoleCommand; // Read commands from console
  pthread_t thread_HVMonitor;      // Reads continuously from HV board(s)
  pthread_t thread_CCCommand;      // Thread listening to commands from Central Control

  int LockDescriptor;

  // Assure only one instance of the HV control program runs
  // The flag O_EXCL together with O_CREAT assure that the lock 
  // file cannot be opened by another instance, i.e. there are no parallel write accesses
  /*  if ((LockDescriptor = open(LOCKFILE,O_RDONLY|O_CREAT|O_EXCL)) == -1) {  
    sprintf(str, "Could not create lock file %s", LOCKFILE);
    perror(str);
    exit(EXIT_FAILURE);
  }
  close(LockDescriptor);
  */
  if((LockDescriptor = open(LOCKFILE,O_WRONLY|O_CREAT|O_EXCL,S_IWUSR|S_IRUSR|S_IRGRP|S_IROTH)) == -1) {
    if(errno==EEXIST) {
      printf("Error: Lock file already existing\n");
      sprintf(str,"paste %s -s -d ' '",LOCKFILE);
      system(str);
    }
    else {
      sprintf(str, "Could not create lock file %s", LOCKFILE);
      perror(str);
    }
    exit(EXIT_FAILURE);
  }
  close(LockDescriptor);
  sprintf(str,"echo Created >%s; date >>%s; echo by $USER@$HOSTNAME>>%s",LOCKFILE,LOCKFILE,LOCKFILE);
  system(str);


  for (int i=1; i<argc; i++) {
    if (argv[i][0] == '-') {
      if (argv[i][1] == 'h') {
	printf("Usage: %s [-c <ConfigFile>]  default file is \"%s\"\n", argv[0], config_file);
	exit(1);
      }
      if (argv[i][1] == 'c') {
	if (argc<i+2) {
	  printf("Error: missing configuration file after -c option\n");
	  exit(1);
	} else {
	  sprintf(config_file, "%s", argv[i+1]);
	}
      }
    }
  }
  
  system("clear");                   
  printf("\n****************** HV Control %s built %s, %s by S. Commichau ******************\n\n",HV_CONTROL_VERSION,__DATE__,__TIME__);


  // Construct main instance
  ProcessIO pio(config_file);
  
  // Install signal handler and set signal SIGUSR1 to interrupt blocking system calls
  signal(SIGUSR1, &SignalHandler);
  siginterrupt (SIGUSR1, true);

  // Install signals to assure that the lock file is deleted in case of a program crash
  signal(SIGQUIT, &CrashHandler);
  signal(SIGILL, &CrashHandler);
  signal(SIGABRT, &CrashHandler);
  signal(SIGFPE, &CrashHandler);
  signal(SIGSEGV, &CrashHandler);
  signal(SIGBUS, &CrashHandler);
  signal(SIGTERM, &CrashHandler);
  signal(SIGINT, &CrashHandler);
  signal(SIGHUP, &CrashHandler);


  // Create threads
  if ((pthread_create(&thread_ConsoleCommand, NULL, (void * (*)(void *)) ConsoleCommand,(void *) &pio)) != 0) {
    fprintf(stderr, "pthread_create failed with ConsoleCommand\n");
    exit(1);
  }

  if ((pthread_create(&thread_HVMonitor, NULL, (void * (*)(void *)) HVMonitor,(void *) &pio)) != 0) {
    fprintf(stderr, "pthread_create failed with HVMonitor\n");
    exit(1);
  }

  if ((pthread_create(&thread_CCCommand, NULL, (void * (*)(void *)) CCCommand,(void *) &pio)) != 0) {
    fprintf(stderr, "pthread_create failed with CCCommand\n");
    exit(1);
  }

  // Threads should be accessible for sending signals
  pio.status->HVMonitor = thread_HVMonitor;
  pio.status->SocketThread = thread_CCCommand;

  // Wait for threads to quit
  pthread_join(thread_CCCommand, NULL);
  pthread_join(thread_ConsoleCommand, NULL);
  pthread_join(thread_HVMonitor, NULL);

  // Remove lockfile
  if (remove(LOCKFILE)==-1) {
    sprintf(str, "Could not remove lock file %s", LOCKFILE);
    perror(str);
    exit(EXIT_FAILURE);
  }

  return 0;
}

// Remove lock file before running default signal code
void CrashHandler(int Signal) {
  remove(LOCKFILE);
  printf("Caught signal number %d. Removing lockfile and performing standard signal action. Good luck.\n",Signal);
  signal(Signal, SIG_DFL);
  raise(Signal);
}
