/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 3/2004 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
// MHFalseSource
//
// Create a false source plot. For the Binning in x,y the object MBinning
// "BinningFalseSource" is taken from the paremeter list.
//
// The binning in alpha is currently fixed as 18bins from 0 to 90deg.
//
// If MTime, MObservatory and MPointingPos is available in the paremeter
// list each image is derotated.
//
// MHFalseSource fills a 3D histogram with alpha distribution for
// each (derotated) x and y position.
//
// Only a radius of 1.2deg around the camera center is taken into account.
//
// The displayed histogram is the projection of alpha<fAlphaCut into
// the x,y plain and the projection of alpha>90-fAlphaCut
//
// By using the context menu (find it in a small region between the single
// pads) you can change the AlphaCut 'online'
//
// Here is a slightly simplified version of the algorithm:
// ------------------------------------------------------
//    MHillas hil; // Taken as second argument in MFillH
//
//    MSrcPosCam src;
//    MHillasSrc hsrc;
//    hsrc.SetSrcPos(&src);
//
//    for (int ix=0; ix<nx; ix++)
//        for (int iy=0; iy<ny; iy++)
//        {
//            TVector2 v(cx[ix], cy[iy]); //cx center of x-bin ix
//            if (rho!=0)                 //cy center of y-bin iy
//                v.Rotate(-rho);         //image rotation angle
//
//            src.SetXY(v);               //source position in the camera
//
//            if (!hsrc.Calc(hil))        //calc source dependant hillas
//                return;
//
//            //fill absolute alpha into histogram
//            const Double_t alpha = hsrc.GetAlpha();
//            fHist.Fill(cx[ix], cy[iy], TMath::Abs(alpha), w);
//        }
//    }
//
// Use MHFalse Source like this:
// -----------------------------
//    MFillH fill("MHFalseSource", "MHillas");
// or
//    MHFalseSource hist;
//    hist.SetAlphaCut(12.5);  // The default value
//    hist.SetBgmean(55);      // The default value
//    MFillH fill(&hist, "MHillas");
//
// To be implemented:
// ------------------
//  - a switch to use alpha or |alpha|
//  - taking the binning for alpha from the parlist (binning is
//    currently fixed)
//  - a possibility to change the fit-function (eg using a different TF1)
//  - a possibility to change the fit algorithm (eg which paremeters
//    are fixed at which time)
//  - use a different varaible than alpha
//  - a possiblity to change the algorithm which is used to calculate
//    alpha (or another parameter) is missing (this could be done using
//    a tasklist somewhere...)
//  - a more clever (and faster) algorithm to fill the histogram, eg by
//    calculating alpha once and fill the two coils around the mean
//  - more drawing options...
//  - Move Significance() to a more 'general' place and implement
//    also different algorithms like (Li/Ma)
//  - implement fit for best alpha distribution -- online (Paint)
//
//////////////////////////////////////////////////////////////////////////////
#include "MHFalseSource.h"

#include <TF1.h>
#include <TH2.h>
#include <TGraph.h>
#include <TStyle.h>
#include <TCanvas.h>
#include <TPaveText.h>
#include <TStopwatch.h>

#include "MGeomCam.h"
#include "MSrcPosCam.h"
#include "MHillasSrc.h"
#include "MTime.h"
#include "MObservatory.h"
#include "MPointingPos.h"
#include "MAstroSky2Local.h"
#include "MStatusDisplay.h"
#include "MMath.h"

#include "MBinning.h"
#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MHFalseSource);

using namespace std;

// --------------------------------------------------------------------------
//
// Default Constructor
//
MHFalseSource::MHFalseSource(const char *name, const char *title)
    : fMm2Deg(-1), fUseMmScale(kTRUE), fAlphaCut(12.5), fBgMean(55)
{
    //
    //   set the name and title of this object
    //
    fName  = name  ? name  : "MHFalseSource";
    fTitle = title ? title : "3D-plot of Alpha vs x, y";

    fHist.SetDirectory(NULL);

    fHist.SetName("Alpha");
    fHist.SetTitle("3D-plot of Alpha vs x, y");
    fHist.SetXTitle("y [\\circ]");
    fHist.SetYTitle("x [\\circ]");
    fHist.SetZTitle("\\alpha [\\circ]");
}

// --------------------------------------------------------------------------
//
// Set the alpha cut (|alpha|<fAlphaCut) which is use to estimate the
// number of excess events
//
void MHFalseSource::SetAlphaCut(Float_t alpha)
{
    if (alpha<0)
        *fLog << warn << "Alpha<0... taking absolute value." << endl;

    fAlphaCut = TMath::Abs(alpha);

    Modified();
}

// --------------------------------------------------------------------------
//
// Set mean alpha around which the off sample is determined
// (fBgMean-fAlphaCut/2<|fAlpha|<fBgMean+fAlphaCut/2) which is use
// to estimate the number of off events
//
void MHFalseSource::SetBgMean(Float_t alpha)
{
    if (alpha<0)
        *fLog << warn << "Alpha<0... taking absolute value." << endl;

    fBgMean = TMath::Abs(alpha);

    Modified();
}

// --------------------------------------------------------------------------
//
// Use this function to setup your own conversion factor between degrees
// and millimeters. The conversion factor should be the one calculated in
// MGeomCam. Use this function with Caution: You could create wrong values
// by setting up your own scale factor.
//
void MHFalseSource::SetMm2Deg(Float_t mmdeg)
{
    if (mmdeg<0)
    {
        *fLog << warn << dbginf << "Warning - Conversion factor < 0 - nonsense. Ignored." << endl;
        return;
    }

    if (fMm2Deg>=0)
        *fLog << warn << dbginf << "Warning - Conversion factor already set. Overwriting" << endl;

    fMm2Deg = mmdeg;
}

// --------------------------------------------------------------------------
//
// With this function you can convert the histogram ('on the fly') between
// degrees and millimeters.
//
void MHFalseSource::SetMmScale(Bool_t mmscale)
{
    if (fUseMmScale == mmscale)
        return;

    if (fMm2Deg<0)
    {
        *fLog << warn << dbginf << "Warning - Sorry, no conversion factor for conversion available." << endl;
        return;
    }

    if (fUseMmScale)
    {
        fHist.SetXTitle("y [mm]");
        fHist.SetYTitle("x [mm]");

        fHist.Scale(1./fMm2Deg);
    }
    else
    {
        fHist.SetXTitle("y [\\circ]");
        fHist.SetYTitle("x [\\circ]");

        fHist.Scale(1./fMm2Deg);
    }

    fUseMmScale = mmscale;
}

// --------------------------------------------------------------------------
//
// Calculate Significance as
// significance = (s-b)/sqrt(s+k*k*b) mit k=s/b
//
// s: total number of events in signal region
// b: number of background events in signal region
// 
Double_t MHFalseSource::Significance(Double_t s, Double_t b)
{
    return MMath::Significance(s, b);
}

// --------------------------------------------------------------------------
//
//  calculates the significance according to Li & Ma
//  ApJ 272 (1983) 317, Formula 17
//
//  s                    // s: number of on events
//  b                    // b: number of off events
//  alpha = t_on/t_off;  // t: observation time
//
Double_t MHFalseSource::SignificanceLiMa(Double_t s, Double_t b, Double_t alpha)
{
    return MMath::SignificanceLiMa(s, b);
}

// --------------------------------------------------------------------------
//
// Set binnings (takes BinningFalseSource) and prepare filling of the
// histogram.
//
// Also search for MTime, MObservatory and MPointingPos
// 
Bool_t MHFalseSource::SetupFill(const MParList *plist)
{
    const MGeomCam *geom = (MGeomCam*)plist->FindObject("MGeomCam");
    if (geom)
    {
        fMm2Deg = geom->GetConvMm2Deg();
        fUseMmScale = kFALSE;

        fHist.SetXTitle("y [\\circ]");
        fHist.SetYTitle("x [\\circ]");
    }

    MBinning binsa;
    binsa.SetEdges(18, 0, 90);

    const MBinning *bins = (MBinning*)plist->FindObject("BinningFalseSource");
    if (!bins)
    {
        Float_t r = geom ? geom->GetMaxRadius() : 600;
        r /= 3;
        if (!fUseMmScale)
            r *= fMm2Deg;

        MBinning b;
        b.SetEdges(20, -r, r);
        SetBinning(&fHist, &b, &b, &binsa);
    }
    else
        SetBinning(&fHist, bins, bins, &binsa);

    fPointPos = (MPointingPos*)plist->FindObject(AddSerialNumber("MPointingPos"));
    if (!fPointPos)
        *fLog << warn << "MPointingPos not found... no derotation." << endl;

    fTime = (MTime*)plist->FindObject(AddSerialNumber("MTime"));
    if (!fTime)
        *fLog << warn << "MTime not found... no derotation." << endl;

    fObservatory = (MObservatory*)plist->FindObject(AddSerialNumber("MObservatory"));
    if (!fObservatory)
        *fLog << err << "MObservatory not found...  no derotation." << endl;


    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Fill the histogram. For details see the code or the class description
// 
Bool_t MHFalseSource::Fill(const MParContainer *par, const Stat_t w)
{
    MHillas *hil = (MHillas*)par;

    const Double_t maxr = 0.98*TMath::Abs(fHist.GetBinCenter(1));

    // Get camera rotation angle
    Double_t rho = 0;
    if (fTime && fObservatory && fPointPos)
        rho = fPointPos->RotationAngle(*fObservatory, *fTime);
    //if (fPointPos)
    //    rho = fPointPos->RotationAngle(*fObservatory);

    MSrcPosCam src;
    MHillasSrc hsrc;
    hsrc.SetSrcPos(&src);

    // This is because a 3D histogram x and y are exchanged...
    const Int_t nx = fHist.GetNbinsY();
    const Int_t ny = fHist.GetNbinsX();
    Axis_t cx[nx];
    Axis_t cy[ny];
    fHist.GetYaxis()->GetCenter(cx);
    fHist.GetXaxis()->GetCenter(cy);

    for (int ix=0; ix<nx; ix++)
    {
        for (int iy=0; iy<ny; iy++)
        {
            if (TMath::Hypot(cx[ix], cy[iy])>maxr)
                continue;

            TVector2 v(cx[ix], cy[iy]);
            if (rho!=0)
                v=v.Rotate(-rho);

            if (!fUseMmScale)
                v *= 1./fMm2Deg;

            src.SetXY(v);

            if (!hsrc.Calc(hil))
            {
                *fLog << warn << "Calculation of MHillasSrc failed for x=" << cx[ix] << " y=" << cy[iy] << endl;
                return kFALSE;
            }

            const Double_t alpha = hsrc.GetAlpha();

            // This is because a 3D histogram x and y are exchanged...
            fHist.Fill(cy[iy], cx[ix], TMath::Abs(alpha), w);
        }
    }

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Create projection for off data, taking sum of bin contents of
// range (fBgMean-fAlphaCut/2, fBgMean+fAlphaCut) Making sure to take
// the same number of bins than for on-data
//
void MHFalseSource::ProjectOff(TH2D *h2)
{
    TAxis &axe = *fHist.GetZaxis();

    // Find range to cut (left edge and width)
    const Int_t f = axe.FindBin(fBgMean-fAlphaCut/2);
    const Int_t l = axe.FindBin(fAlphaCut)+f-1;

    axe.SetRange(f, l);
    const Float_t cut1 = axe.GetBinLowEdge(f);
    const Float_t cut2 = axe.GetBinUpEdge(l);
    h2->SetTitle(Form("Distribution of %.1f\\circ<|\\alpha|<%.1f\\circ in x,y", cut1, cut2));

    // Get projection for range
    TH2D *p = (TH2D*)fHist.Project3D("xy_off");

    // Reset range
    axe.SetRange(0,9999);

    // Move contents from projection to h2
    h2->Reset();
    h2->Add(p);

    // Delete p
    delete p;

    // Set Minimum as minimum value Greater Than 0
    h2->SetMinimum(GetMinimumGT(*h2));
}

// --------------------------------------------------------------------------
//
// Create projection for on data, taking sum of bin contents of
// range (0, fAlphaCut)
//
void MHFalseSource::ProjectOn(TH2D *h3)
{
    TAxis &axe = *fHist.GetZaxis();

    // Find range to cut
    axe.SetRangeUser(0, fAlphaCut);
    const Float_t cut = axe.GetBinUpEdge(axe.GetLast());
    h3->SetTitle(Form("Distribution of |\\alpha|<%.1f\\circ in x,y", cut));

    // Get projection for range
    TH2D *p = (TH2D*)fHist.Project3D("xy_on");

    // Reset range
    axe.SetRange(0,9999);

    // Move contents from projection to h3
    h3->Reset();
    h3->Add(p);
    delete p;

    // Set Minimum as minimum value Greater Than 0
    h3->SetMinimum(GetMinimumGT(*h3));
}

// --------------------------------------------------------------------------
//
// Update the projections
//
void MHFalseSource::Paint(Option_t *opt)
{
    // sigma = (s-b)/sqrt(s+k*k*b) mit k=s/b

    gStyle->SetPalette(1, 0);

    TVirtualPad *padsave = gPad;

    TH1D* h1;
    TH2D* h2;
    TH2D* h3;
    TH2D* h4;

    padsave->cd(3);
    if ((h3 = (TH2D*)gPad->FindObject("Alpha_xy_on")))
        ProjectOn(h3);

    padsave->cd(4);
    if ((h2 = (TH2D*)gPad->FindObject("Alpha_xy_off")))
        ProjectOff(h2);

    padsave->cd(2);
    if (h2 && h3 && (h4 = (TH2D*)gPad->FindObject("Alpha_xy_sig")))
    {
        const Int_t nx = h4->GetXaxis()->GetNbins();
        const Int_t ny = h4->GetYaxis()->GetNbins();

        Int_t maxx=nx/2;
        Int_t maxy=ny/2;

        Int_t max = h4->GetBin(maxx, maxy);

        for (int ix=0; ix<nx; ix++)
            for (int iy=0; iy<ny; iy++)
            {
                const Int_t n = h4->GetBin(ix+1, iy+1);

                const Double_t s = h3->GetBinContent(n);
                const Double_t b = h2->GetBinContent(n);

                const Double_t sig = Significance(s, b);

                h4->SetBinContent(n, sig);

                if (sig>h4->GetBinContent(max) && sig!=0)
                {
                    max = n;
                    maxx=ix;
                    maxy=iy;
                }
            }

        padsave->cd(1);
        if ((h1 = (TH1D*)gPad->FindObject("Alpha")))
        {
            const Double_t x = h4->GetXaxis()->GetBinCenter(maxx);
            const Double_t y = h4->GetYaxis()->GetBinCenter(maxy);
            const Double_t s = h4->GetBinContent(max);

            // This is because a 3D histogram x and y are vice versa
            // Than for their projections
            TH1 *h = fHist.ProjectionZ("Alpha", maxy, maxy, maxx, maxx);
            h->SetTitle(Form("Distribution of \\alpha for x=%.2f y=%.2f (\\sigma_{max}=%.1f)", x, y, s));
        }
    }

    gPad = padsave;
}

// --------------------------------------------------------------------------
//
// Draw the histogram
//
void MHFalseSource::Draw(Option_t *opt)
{
    TVirtualPad *pad = gPad ? gPad : MakeDefCanvas(this);
    pad->SetBorderMode(0);

    AppendPad("");

    pad->Divide(2, 2);

    // draw the 2D histogram Sigmabar versus Theta
    pad->cd(1);
    gPad->SetBorderMode(0);
    TH1 *h1 = fHist.ProjectionZ("Alpha");
    h1->SetDirectory(NULL);
    h1->SetTitle("Distribution of \\alpha");
    h1->SetXTitle(fHist.GetZaxis()->GetTitle());
    h1->SetYTitle("Counts");
    h1->Draw(opt);
    h1->SetBit(kCanDelete);

    pad->cd(4);
    gPad->SetBorderMode(0);
    fHist.GetZaxis()->SetRangeUser(fBgMean-fAlphaCut/2, fBgMean+fAlphaCut/2);
    TH1 *h2 = fHist.Project3D("xy_off");
    h2->SetDirectory(NULL);
    h2->SetXTitle(fHist.GetYaxis()->GetTitle());
    h2->SetYTitle(fHist.GetXaxis()->GetTitle());
    h2->Draw("colz");
    h2->SetBit(kCanDelete);

    pad->cd(3);
    gPad->SetBorderMode(0);
    fHist.GetZaxis()->SetRangeUser(0,fAlphaCut);
    TH1 *h3 = fHist.Project3D("xy_on");
    fHist.GetZaxis()->SetRange(0,9999);
    h3->SetDirectory(NULL);
    h3->SetXTitle(fHist.GetYaxis()->GetTitle());
    h3->SetYTitle(fHist.GetXaxis()->GetTitle());
    h3->Draw("colz");
    h3->SetBit(kCanDelete);

    pad->cd(2);
    gPad->SetBorderMode(0);
    fHist.GetZaxis()->SetRange(0,0);
    TH1 *h4 = fHist.Project3D("xy_sig"); // Do this to get the correct binning....
    fHist.GetZaxis()->SetRange(0,9999);
    h4->SetTitle("Significance");
    h4->SetDirectory(NULL);
    h4->SetXTitle(fHist.GetYaxis()->GetTitle());
    h4->SetYTitle(fHist.GetXaxis()->GetTitle());
    h4->Draw("colz");
    h4->SetBit(kCanDelete);
}

// --------------------------------------------------------------------------
//
// Everything which is in the main pad belongs to this class!
//
Int_t MHFalseSource::DistancetoPrimitive(Int_t px, Int_t py)
{
    return 0;
}

// --------------------------------------------------------------------------
//
// Set all sub-pads to Modified()
//
void MHFalseSource::Modified()
{
    if (!gPad)
        return;

    TVirtualPad *padsave = gPad;
    padsave->Modified();
    padsave->cd(1);
    gPad->Modified();
    padsave->cd(2);
    gPad->Modified();
    padsave->cd(3);
    gPad->Modified();
    padsave->cd(4);
    gPad->Modified();
    gPad->cd();
}

// --------------------------------------------------------------------------
//
// This is a preliminary implementation of a alpha-fit procedure for
// all possible source positions. It will be moved into its own
// more powerfull class soon.
//
// The fit function is "gaus(0)+pol2(3)" which is equivalent to:
//   [0]*exp(-0.5*((x-[1])/[2])^2) + [3] + [4]*x + [5]*x^2
// or
//   A*exp(-0.5*((x-mu)/sigma)^2) + a + b*x + c*x^2
//
// Parameter [1] is fixed to 0 while the alpha peak should be
// symmetric around alpha=0.
//
// Parameter [4] is fixed to 0 because the first derivative at
// alpha=0 should be 0, too.
//
// In a first step the background is fitted between bgmin and bgmax,
// while the parameters [0]=0 and [2]=1 are fixed.
//
// In a second step the signal region (alpha<sigmax) is fittet using
// the whole function with parameters [1], [3], [4] and [5] fixed.
//
// The number of excess and background events are calculated as
//   s = int(0, sigint, gaus(0)+pol2(3))
//   b = int(0, sigint,         pol2(3))
//
// The Significance is calculated using the Significance() member
// function.
//
void MHFalseSource::FitSignificance(Float_t sigint, Float_t sigmax, Float_t bgmin, Float_t bgmax, Byte_t polynom)
{
    TH1D h0a("A",          "", 50,   0, 4000);
    TH1D h4a("chisq1",     "", 50,   0,   35);
    //TH1D h5a("prob1",      "", 50,   0,  1.1);
    TH1D h6("signifcance", "", 50, -20,   20);

    TH1D h1("mu",    "Parameter \\mu",    50,   -1,    1);
    TH1D h2("sigma", "Parameter \\sigma", 50,    0,   90);
    TH1D h3("b",     "Parameter b",       50, -0.1,  0.1);

    TH1D h0b("a",         "Parameter a (red), A (blue)", 50, 0, 4000);
    TH1D h4b("\\chi^{2}", "\\chi^{2} (red, green) / significance (black)", 50, 0, 35);
    //TH1D h5b("prob",      "Fit probability: Bg(red), F(blue)", 50, 0, 1.1);

    h0a.SetLineColor(kBlue);
    h4a.SetLineColor(kBlue);
    //h5a.SetLineColor(kBlue);
    h0b.SetLineColor(kRed);
    h4b.SetLineColor(kRed);
    //h5b.SetLineColor(kRed);

    TH1 *hist  = fHist.Project3D("xy_fit");
    hist->SetDirectory(0);
    TH1 *hists = fHist.Project3D("xy_fit");
    hists->SetDirectory(0);
    TH1 *histb = fHist.Project3D("xy_fit");
    histb->SetDirectory(0);
    hist->Reset();
    hists->Reset();
    histb->Reset();
    hist->SetNameTitle("Significance",
                       Form("Fit Region: Signal<%.1f\\circ, %.1f\\circ<Bg<%.1f\\circ",
                            sigmax, bgmin, bgmax));
    hists->SetNameTitle("Excess",     Form("Number of excess events for \\alpha<%.0f\\circ", sigint));
    histb->SetNameTitle("Background", Form("Number of background events for \\alpha<%.0f\\circ", sigint));
    hist->SetXTitle(fHist.GetYaxis()->GetTitle());
    hists->SetXTitle(fHist.GetYaxis()->GetTitle());
    histb->SetXTitle(fHist.GetYaxis()->GetTitle());
    hist->SetYTitle(fHist.GetXaxis()->GetTitle());
    hists->SetYTitle(fHist.GetXaxis()->GetTitle());
    histb->SetYTitle(fHist.GetXaxis()->GetTitle());

    const Double_t w = fHist.GetZaxis()->GetBinWidth(1);

    //                      xmin, xmax, npar
    //TF1 func("MyFunc", fcn, 0, 90, 6);
    // Implementing the function yourself is only about 5% faster
    TF1 func("", Form("gaus(0) + pol%d(3)", polynom), 0, 90);
    TArrayD maxpar(func.GetNpar());

    /*
     func.SetParName(0, "A");
     func.SetParName(1, "mu");
     func.SetParName(2, "sigma");
     func.SetParName(3, "a");
     func.SetParName(4, "b");
     func.SetParName(5, "c");
    */

    func.FixParameter(1, 0);
    func.FixParameter(4, 0);
    func.SetParLimits(2, 0, 90);
    func.SetParLimits(3, -1, 1);

    const Int_t nx = hist->GetXaxis()->GetNbins();
    const Int_t ny = hist->GetYaxis()->GetNbins();
    const Int_t nr = nx*nx+ny*ny;

    Double_t maxalpha0=0;
    Double_t maxs=3;

    Int_t maxx=0;
    Int_t maxy=0;

    TStopwatch clk;
    clk.Start();

    *fLog << inf;
    *fLog << "Signal fit:     alpha < " << sigmax << endl;
    *fLog << "Integration:    alpha < " << sigint << endl;
    *fLog << "Background fit: " << bgmin << " < alpha < " << bgmax << endl;
    *fLog << "Polynom order:  " << (int)polynom << endl;
    *fLog << "Fitting False Source Plot..." << flush;

    TH1 *h=0;
    for (int ix=1; ix<=nx; ix++)
        for (int iy=1; iy<=ny; iy++)
        {
            // This is because a 3D histogram x and y are vice versa
            // Than for their projections
            h = fHist.ProjectionZ("AlphaFit", iy, iy, ix, ix);

            const Double_t alpha0 = h->GetBinContent(1);

            // Check for the regios which is not filled...
            if (alpha0==0)
                continue;

            if (alpha0>maxalpha0)
                maxalpha0=alpha0;

            // First fit a polynom in the off region
            func.FixParameter(0, 0);
            func.FixParameter(2, 1);
            func.ReleaseParameter(3);

            for (int i=5; i<func.GetNpar(); i++)
                func.ReleaseParameter(i);

            h->Fit(&func, "N0Q", "", bgmin, bgmax);
            //*fLog << dbg << ix << "/" << iy << ":  " << func.GetParameter(3) << "    " << func.GetParameter(4) << endl;

            h4a.Fill(func.GetChisquare());
            //h5a.Fill(func.GetProb());

            //func.SetParLimits(0, 0.5*h->GetBinContent(1), 1.5*h->GetBinContent(1));
            //func.SetParLimits(2, 5, 90);

            func.ReleaseParameter(0);
            //func.ReleaseParameter(1);
            func.ReleaseParameter(2);
            func.FixParameter(3, func.GetParameter(3));
            for (int i=5; i<func.GetNpar(); i++)
                func.FixParameter(i, func.GetParameter(i));

            // Do not allow signals smaller than the background
            const Double_t A  = alpha0-func.GetParameter(3);
            const Double_t dA = TMath::Abs(A);
            func.SetParLimits(0, -dA*4, dA*4);
            func.SetParLimits(2, 0, 90);

            // Now fit a gaus in the on region on top of the polynom
            func.SetParameter(0, A);
            func.SetParameter(2, sigmax*0.75);

            h->Fit(&func, "N0Q", "", 0, sigmax);
            //*fLog << dbg << "     " << func.GetParameter(0) << "    " << func.GetParameter(1) << "    " << func.GetParameter(2) << endl;

            TArrayD p(func.GetNpar(), func.GetParameters());

            // Fill results into some histograms
            h0a.Fill(p[0]);
            h0b.Fill(p[3]);
            h1.Fill(p[1]);
            h2.Fill(p[2]);
            if (polynom>1)
                h3.Fill(p[5]);
            h4b.Fill(func.GetChisquare());
            //h5b.Fill(func.GetProb());

            // Implementing the integral as analytical function
            // gives the same result in the order of 10e-5
            // and it is not faster at all...
            //const Bool_t ok = /*p[0]>=0 && /*p[0]<alpha0*2 &&*/ p[2]>1.75;// && p[2]<88.5;
            /*
            if (p[0]<-fabs(alpha0-p[3])*7 && p[2]<alphaw/3)
            {
                func.SetParameter(0, alpha0-p[3]);
                cout << p[2] << " " << p[0] << " " << alpha0-p[3] << endl;
            }
            */

            // The fitted function returned units of
            // counts bin binwidth. To get the correct number
            // of events we must adapt the functions by dividing
            // the result of the integration by the bin-width
            const Double_t s = func.Integral(0, sigint)/w;

            func.SetParameter(0, 0);
            func.SetParameter(2, 1);

            const Double_t b   = func.Integral(0, sigint)/w;
            const Double_t sig = Significance(s, b);

            // This is because a 3D histogram x and y are exchanged...
            const Int_t n = hist->GetBin(ix, iy);
            hists->SetBinContent(n, s-b);
            histb->SetBinContent(n, b);

            hist->SetBinContent(n, sig);
            if (sig!=0)
                h6.Fill(sig);

            if (sig>maxs && ix*ix+iy*iy<nr*nr/9)
            {
                maxs = sig;
                maxx = ix;
                maxy = iy;
                maxpar = p;
            }
        }

    *fLog << inf << "done." << endl;

    h0a.GetXaxis()->SetRangeUser(0, maxalpha0*1.5);
    h0b.GetXaxis()->SetRangeUser(0, maxalpha0*1.5);

    //hists->SetMinimum(GetMinimumGT(*hists));
    histb->SetMinimum(GetMinimumGT(*histb));

    clk.Stop();
    clk.Print();

    TCanvas *c=new TCanvas;

    gStyle->SetPalette(1, 0);

    c->Divide(3,2, 0, 0);
    c->cd(1);
    gPad->SetBorderMode(0);
    hists->Draw("colz");
    hists->SetBit(kCanDelete);
    c->cd(2);
    gPad->SetBorderMode(0);
    hist->Draw("colz");
    hist->SetBit(kCanDelete);
    c->cd(3);
    gPad->SetBorderMode(0);
    histb->Draw("colz");
    histb->SetBit(kCanDelete);
    c->cd(4);
    gPad->Divide(1,3, 0, 0);
    TVirtualPad *p=gPad;
    p->SetBorderMode(0);
    p->cd(1);
    gPad->SetBorderMode(0);
    h0b.DrawCopy();
    h0a.DrawCopy("same");
    p->cd(2);
    gPad->SetBorderMode(0);
    h3.DrawCopy();
    p->cd(3);
    gPad->SetBorderMode(0);
    h2.DrawCopy();
    c->cd(6);
    gPad->Divide(1,2, 0, 0);
    TVirtualPad *q=gPad;
    q->SetBorderMode(0);
    q->cd(1);
    gPad->SetBorderMode(0);
    gPad->SetBorderMode(0);
    h4b.DrawCopy();
    h4a.DrawCopy("same");
    h6.DrawCopy("same");
    q->cd(2);
    gPad->SetBorderMode(0);
    //h5b.DrawCopy();
    //h5a.DrawCopy("same");
    c->cd(5);
    gPad->SetBorderMode(0);
    if (maxx>0 && maxy>0)
    {
        const char *title = Form(" \\alpha for x=%.2f y=%.2f (\\sigma_{max}=%.1f) ",
                                 hist->GetXaxis()->GetBinCenter(maxx),
                                 hist->GetYaxis()->GetBinCenter(maxy), maxs);

        TH1 *result = fHist.ProjectionZ("AlphaFit", maxy, maxy, maxx, maxx);
        result->SetDirectory(NULL);
        result->SetNameTitle("Result \\alpha", title);
        result->SetBit(kCanDelete);
        result->SetXTitle("\\alpha [\\circ]");
        result->SetYTitle("Counts");
        result->Draw();

        TF1 f1("", func.GetExpFormula(), 0, 90);
        TF1 f2("", func.GetExpFormula(), 0, 90);
        f1.SetParameters(maxpar.GetArray());
        f2.SetParameters(maxpar.GetArray());
        f2.FixParameter(0, 0);
        f2.FixParameter(1, 0);
        f2.FixParameter(2, 1);
        f1.SetLineColor(kGreen);
        f2.SetLineColor(kRed);

        f2.DrawCopy("same");
        f1.DrawCopy("same");

        TPaveText *leg = new TPaveText(0.35, 0.10, 0.90, 0.35, "brNDC");
        leg->SetBorderSize(1);
        leg->SetTextSize(0.04);
        leg->AddText(0.5, 0.82, Form("A * exp(-(\\frac{x-\\mu}{\\sigma})^{2}/2) + pol%d", polynom))->SetTextAlign(22);
        //leg->AddText(0.5, 0.82, "A * exp(-(\\frac{x-\\mu}{\\sigma})^{2}/2) + b*x^{2} + a")->SetTextAlign(22);
        leg->AddLine(0, 0.65, 0, 0.65);
        leg->AddText(0.06, 0.54, Form("A=%.2f", maxpar[0]))->SetTextAlign(11);
        leg->AddText(0.06, 0.34, Form("\\sigma=%.2f", maxpar[2]))->SetTextAlign(11);
        leg->AddText(0.06, 0.14, Form("\\mu=%.2f (fix)", maxpar[1]))->SetTextAlign(11);
        leg->AddText(0.60, 0.54, Form("a=%.2f", maxpar[3]))->SetTextAlign(11);
        leg->AddText(0.60, 0.34, Form("b=%.2f (fix)", maxpar[4]))->SetTextAlign(11);
        if (polynom>1)
            leg->AddText(0.60, 0.14, Form("c=%.2f", maxpar[5]))->SetTextAlign(11);
        leg->SetBit(kCanDelete);
        leg->Draw();

        q->cd(2);

        TGraph *g = new TGraph;
        g->SetPoint(0, 0, 0);

        Int_t max=0;
        Float_t maxsig=0;
        for (int i=1; i<89; i++)
        {
            const Double_t s = f1.Integral(0, (float)i)/w;
            const Double_t b = f2.Integral(0, (float)i)/w;

            const Double_t sig = Significance(s, b);

            g->SetPoint(g->GetN(), i, sig);

            if (sig>maxsig)
            {
                max = i;
                maxsig = sig;
            }
        }

        g->SetNameTitle("SigVs\\alpha", "Significance vs \\alpha");
        g->GetHistogram()->SetXTitle("\\alpha_{0} [\\circ]");
        g->GetHistogram()->SetYTitle("Significance");
        g->SetBit(kCanDelete);
        g->Draw("AP");

        leg = new TPaveText(0.35, 0.10, 0.90, 0.25, "brNDC");
        leg->SetBorderSize(1);
        leg->SetTextSize(0.1);
        leg->AddText(Form("\\sigma_{max}=%.1f at \\alpha_{max}=%d\\circ", maxsig, max));
        leg->SetBit(kCanDelete);
        leg->Draw();
    }
}
