/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug 02/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
//  MHCalibrationRelTimePix
//
//  Histogram class for the relative arrival time calibration.
//  Holds the histogrammed arrival times,
//  derives from MHGausEvents, perform Fourier analysis
//
//////////////////////////////////////////////////////////////////////////////
#include "MHCalibrationRelTimePix.h"

#include <TH1.h>

ClassImp(MHCalibrationRelTimePix);

using namespace std;
//
const Int_t   MHCalibrationRelTimePix::fgChargeNbins    = 900;
const Axis_t  MHCalibrationRelTimePix::fgChargeFirst    = -13.;
const Axis_t  MHCalibrationRelTimePix::fgChargeLast     =  13.;
// --------------------------------------------------------------------------
//
// Default Constructor. 
//
MHCalibrationRelTimePix::MHCalibrationRelTimePix(const char *name, const char *title) 
    : fPixId(-1)
{ 

  fName  = name  ? name  : "MHCalibrationRelTimePix";
  fTitle = title ? title : "Histogrammed Calibration Relative Arrival Time events";

  SetChargeNbins();
  SetChargeFirst();
  SetChargeLast();

  // Create a large number of bins, later we will rebin
  fHGausHist.SetName("HCalibrationRelTimeCharge");
  fHGausHist.SetTitle("Distribution of Relative Arrival Times Pixel ");
  fHGausHist.SetXTitle("FADC Slice");
  fHGausHist.SetYTitle("Nr. of events");
  fHGausHist.Sumw2();

}

MHCalibrationRelTimePix::~MHCalibrationRelTimePix()
{
}

void MHCalibrationRelTimePix::Clear(Option_t *o)
{

  fPixId = -1;
  MHGausEvents::Clear();
  return;
}



void MHCalibrationRelTimePix::InitBins()
{

  fHGausHist.SetBins(fChargeNbins,fChargeFirst,fChargeLast);

}


void MHCalibrationRelTimePix::ChangeHistId(Int_t id)
{

  fPixId = id;

  fHGausHist.SetName(  Form("%s%d", fHGausHist.GetName(),  id));
  fHGausHist.SetTitle( Form("%s%d", fHGausHist.GetTitle(), id));

}


void  MHCalibrationRelTimePix::BypassFit()
{

  //
  // In case, we do not obtain reasonable values 
  // with the fit, we take the histogram values
  //
  SetMean(      fHGausHist.GetMean()                              );
  SetMeanErr(   fHGausHist.GetRMS() / fHGausHist.GetEntries()     );
  SetSigma(     fHGausHist.GetRMS()                               );
  SetSigmaErr(  fHGausHist.GetRMS() / fHGausHist.GetEntries() /2. );

}

// ----------------------------------------------------------------------
//
// Renorm the results from FADC slices to times in ns.
//
void MHCalibrationRelTimePix::Renorm(const Float_t slicewidth)
{

  SetMean(     GetMean()    * slicewidth  );
  SetMeanErr(  GetMeanErr() * slicewidth  );
  SetSigma(    GetSigma()   * slicewidth  );
  SetSigmaErr( GetSigmaErr()* slicewidth  );
  
}

