/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug   02/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                               
// MHCalibrationChargeCam                                               
//
// Fills the extracted signals of MExtractedSignalCam into the MH-classes: 
//
// MHCalibrationChargeHiGainPix, MHCalibrationChargeLoGainPix for each pixel 
// and additionally for an average of the inner and the outer pixels, respectively.
// 
// By default, subsequently the hi-gain classes are treated unless 
// more than fNumHiGainSaturationLimit (default: 1%) of the overall FADC 
// slices show saturation. In that case, the low-gain classes are treated. 
// If more than fNumLoGainSaturationLimit (default: 1%) of the overall 
// low-gain FADC slices saturate, the pixel is declared not valid and no further 
// treatment is pursued. 
// 
// The filled histograms are fitted to a Gaussian and the mean and sigma with 
// its errors and the fit probability are extracted. If none of these values are 
// NaN's and if the probability is bigger than fProbLimit (default: 0.5%), the fit 
// is declared valid. Otherwise, the fit is repeated within ranges of the previous mean 
// +- 5 sigma. In case that this does not help, the histogram means and RMS's are taken directly,
// but the flag kFitValid is set to FALSE. Outliers of more than fPickUpLimit (default: 5) sigmas 
// from the mean are counted as PickUp events. 
//
// Additionally, the slice number with the highest value is stored and a corresponding 
// histogram is filled. This histogram serves only for a rough cross-check if the 
// signal does not lie at the edges of chose extraction window. 
//
// The class also fills arrays with the signal vs. event number, creates a fourier 
// spectrum out of it and investigates if the projected frequencies follow an exponential 
// distribution. In case that the probability of the exponential fit is less than 
// fProbLimit, the pixel is declared HiGainOscillating or LoGainOscillating, respectively.
// 
// The results are written into MCalibrationChargeCam.
//                                                               
/////////////////////////////////////////////////////////////////////////////
#include "MHCalibrationChargeCam.h"

#include <TVirtualPad.h>
#include <TCanvas.h>
#include <TPad.h>
#include <TText.h>
#include <TPaveText.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"

#include "MHCalibrationChargeHiGainPix.h"
#include "MHCalibrationChargeLoGainPix.h"
#include "MHCalibrationChargePix.h"

#include "MCalibrationChargeCam.h"
#include "MCalibrationChargePix.h"

#include "MGeomCam.h"
#include "MGeomPix.h"

#include "MBadPixelsCam.h"
#include "MBadPixelsPix.h"

#include "MRawEvtData.h"
#include "MRawEvtPixelIter.h"

#include "MExtractedSignalCam.h"
#include "MExtractedSignalPix.h"

ClassImp(MHCalibrationChargeCam);

using namespace std;

const Float_t MHCalibrationChargeCam::fgNumHiGainSaturationLimit = 0.01;
const Float_t MHCalibrationChargeCam::fgNumLoGainSaturationLimit = 0.005;
const Int_t   MHCalibrationChargeCam::fgPulserFrequency          = 500;
//
//
MHCalibrationChargeCam::MHCalibrationChargeCam(const char *name, const char *title)
    : fRawEvt(NULL), fGeom(NULL), fBadPixels(NULL)
{
    fName  = name  ? name  : "MHCalibrationChargeCam";
    fTitle = title ? title : "Class to fill the calibration histograms ";

    fHiGainArray = new TObjArray;
    fHiGainArray->SetOwner();
    
    fLoGainArray = new TObjArray;
    fLoGainArray->SetOwner();

    fAverageHiGainAreas = new TObjArray;
    fAverageHiGainAreas->SetOwner();

    fAverageLoGainAreas = new TObjArray;
    fAverageLoGainAreas->SetOwner();

    fAverageHiGainSectors = new TObjArray;
    fAverageHiGainSectors->SetOwner();

    fAverageLoGainSectors = new TObjArray;
    fAverageLoGainSectors->SetOwner();

    SetNumHiGainSaturationLimit();
    SetNumLoGainSaturationLimit();
    SetPulserFrequency();
}

// --------------------------------------------------------------------------
//
// Delete the TClonesArray of MHCalibrationChargePix containers
// Delete the average pixels
//
MHCalibrationChargeCam::~MHCalibrationChargeCam()
{
  delete fHiGainArray;
  delete fLoGainArray;

  delete fAverageHiGainAreas;
  delete fAverageLoGainAreas;

  delete fAverageHiGainSectors;
  delete fAverageLoGainSectors;
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
MHCalibrationChargeHiGainPix &MHCalibrationChargeCam::operator[](UInt_t i)
{
  return *static_cast<MHCalibrationChargeHiGainPix*>(fHiGainArray->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
const MHCalibrationChargeHiGainPix &MHCalibrationChargeCam::operator[](UInt_t i) const
{
  return *static_cast<MHCalibrationChargeHiGainPix*>(fHiGainArray->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
MHCalibrationChargeLoGainPix &MHCalibrationChargeCam::operator()(UInt_t i)
{
  return *static_cast<MHCalibrationChargeLoGainPix*>(fLoGainArray->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
const MHCalibrationChargeLoGainPix &MHCalibrationChargeCam::operator()(UInt_t i) const
{
  return *static_cast<MHCalibrationChargeLoGainPix*>(fLoGainArray->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
MHCalibrationChargeHiGainPix &MHCalibrationChargeCam::GetAverageHiGainArea(UInt_t i)
{
  return *static_cast<MHCalibrationChargeHiGainPix*>(fAverageHiGainAreas->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
const MHCalibrationChargeHiGainPix &MHCalibrationChargeCam::GetAverageHiGainArea(UInt_t i) const
{
  return *static_cast<MHCalibrationChargeHiGainPix*>(fAverageHiGainAreas->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
MHCalibrationChargeLoGainPix &MHCalibrationChargeCam::GetAverageLoGainArea(UInt_t i)
{
  return *static_cast<MHCalibrationChargeLoGainPix*>(fAverageLoGainAreas->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
const MHCalibrationChargeLoGainPix &MHCalibrationChargeCam::GetAverageLoGainArea(UInt_t i) const
{
  return *static_cast<MHCalibrationChargeLoGainPix*>(fAverageLoGainAreas->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
MHCalibrationChargeHiGainPix &MHCalibrationChargeCam::GetAverageHiGainSector(UInt_t i)
{
  return *static_cast<MHCalibrationChargeHiGainPix*>(fAverageHiGainSectors->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
const MHCalibrationChargeHiGainPix &MHCalibrationChargeCam::GetAverageHiGainSector(UInt_t i) const
{
  return *static_cast<MHCalibrationChargeHiGainPix*>(fAverageHiGainSectors->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
MHCalibrationChargeLoGainPix &MHCalibrationChargeCam::GetAverageLoGainSector(UInt_t i)
{
  return *static_cast<MHCalibrationChargeLoGainPix*>(fAverageLoGainSectors->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
const MHCalibrationChargeLoGainPix &MHCalibrationChargeCam::GetAverageLoGainSector(UInt_t i) const
{
  return *static_cast<MHCalibrationChargeLoGainPix*>(fAverageLoGainSectors->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Our own clone function is necessary since root 3.01/06 or Mars 0.4
// I don't know the reason
//
TObject *MHCalibrationChargeCam::Clone(const char *) const
{

  const Int_t nhi   = fHiGainArray->GetEntries();
  const Int_t nlo   = fLoGainArray->GetEntries();
  const Int_t navhi = fAverageHiGainAreas->GetEntries();
  const Int_t navlo = fAverageLoGainAreas->GetEntries();
  const Int_t nsehi = fAverageHiGainSectors->GetEntries();
  const Int_t nselo = fAverageLoGainSectors->GetEntries();
  
  //
  // FIXME, this might be done faster and more elegant, by direct copy.
  //
  MHCalibrationChargeCam *cam = new MHCalibrationChargeCam();

  cam->fHiGainArray->Expand(nhi);
  cam->fLoGainArray->Expand(nlo);
  cam->fAverageHiGainAreas->Expand(navhi);
  cam->fAverageLoGainAreas->Expand(navlo);
  cam->fAverageHiGainSectors->Expand(nsehi);
  cam->fAverageLoGainSectors->Expand(nselo);

  for (int i=0; i<nhi; i++)
    {
      delete (*cam->fHiGainArray)[i];
      (*cam->fHiGainArray)[i] = (MHCalibrationChargeHiGainPix*)(*fHiGainArray)[i]->Clone();
    }
  for (int i=0; i<nlo; i++)
    {
      delete (*cam->fLoGainArray)[i];
      (*cam->fLoGainArray)[i] = (MHCalibrationChargeLoGainPix*)(*fLoGainArray)[i]->Clone();
    }
  for (int i=0; i<navhi; i++)
    {
      delete (*cam->fAverageHiGainAreas)[i];
      (*cam->fAverageHiGainAreas)[i] = (MHCalibrationChargeHiGainPix*)(*fAverageHiGainAreas)[i]->Clone();
    }
  for (int i=0; i<navlo; i++)
    {
      delete (*cam->fAverageLoGainAreas)[i];
      (*cam->fAverageLoGainAreas)[i] = (MHCalibrationChargeLoGainPix*)(*fAverageLoGainAreas)[i]->Clone();
    }
  for (int i=0; i<nsehi; i++)
    {
      delete (*cam->fAverageHiGainSectors)[i];
      (*cam->fAverageHiGainSectors)[i] = (MHCalibrationChargeHiGainPix*)(*fAverageHiGainSectors)[i]->Clone();
    }
  for (int i=0; i<nselo; i++)
    {
      delete (*cam->fAverageLoGainSectors)[i];
      (*cam->fAverageLoGainSectors)[i] = (MHCalibrationChargeLoGainPix*)(*fAverageLoGainSectors)[i]->Clone();
    }

  cam->fAverageAreaSat         = fAverageAreaSat;            
  cam->fAverageAreaSigma       = fAverageAreaSigma;      
  cam->fAverageAreaSigmaErr    = fAverageAreaSigmaErr;   
  cam->fAverageAreaRelSigma    = fAverageAreaRelSigma;
  cam->fAverageAreaRelSigmaErr = fAverageAreaRelSigmaErr;   

  return cam;
}

// --------------------------------------------------------------------------
//
// To setup the object we get the number of pixels from a MGeomCam object
// in the Parameter list.
//
Bool_t MHCalibrationChargeCam::SetupFill(const MParList *pList)
{
  
  fRawEvt = (MRawEvtData*)pList->FindObject("MRawEvtData");
  if (!fRawEvt)
  {
      *fLog << err << dbginf << "MRawEvtData not found... aborting." << endl;
      return kFALSE;
  }

  fGeom = (MGeomCam*)pList->FindObject("MGeomCam");
  if (!fGeom)
  {
      *fLog << err << "MGeomCam not found... aborting." << endl;
      return kFALSE;
  }

  fHiGainArray->Delete();
  fLoGainArray->Delete();

  fAverageHiGainAreas->Delete();
  fAverageLoGainAreas->Delete();

  fAverageHiGainSectors->Delete();
  fAverageLoGainSectors->Delete();

  return kTRUE;
}

Bool_t MHCalibrationChargeCam::ReInit(MParList *pList)
{

  fBadPixels = (MBadPixelsCam*)pList->FindCreateObj("MBadPixelsCam");
  if (!fBadPixels)
      return kFALSE;


  fCam = (MCalibrationChargeCam*)pList->FindCreateObj("MCalibrationChargeCam");
  if (!fCam)
      return kFALSE;

  MExtractedSignalCam *signal = (MExtractedSignalCam*)pList->FindObject("MExtractedSignalCam");
  if (!signal)
  {
      *fLog << err << "MExtractedSignalCam not found... abort." << endl;
      return kFALSE;
  }

  const Int_t n = signal->GetSize();

  if (fHiGainArray->GetEntries()==0)
  {
      fHiGainArray->Expand(n);
  
      for (Int_t i=0; i<n; i++)
      {
	  //
	  // Oscillating pixels
	  //
	  (*fHiGainArray)[i] = new MHCalibrationChargeHiGainPix;
          MHCalibrationChargeHiGainPix &hist = (*this)[i];

          if ((*fBadPixels)[i].IsBad())
	  {
	    *fLog << warn << "Excluded pixel: " << i << " from calibration " << endl;
	    hist.SetExcluded();
	  }
	  hist.Init();
	  hist.ChangeHistId(i);
          hist.SetEventFrequency(fPulserFrequency);
      }
  }

 
  if (fLoGainArray->GetEntries()==0)
  {
      fLoGainArray->Expand(n);
      
      for (Int_t i=0; i<n; i++)
      {
	  (*fLoGainArray)[i] = new MHCalibrationChargeLoGainPix;
          MHCalibrationChargeLoGainPix &hist = (*this)(i);
	  //
	  // Pixels with non-valid behavior
	  //
          if ((*fBadPixels)[i].IsBad())
            hist.SetExcluded();

	  hist.Init();
	  hist.ChangeHistId(i);
          hist.SetEventFrequency(fPulserFrequency);
      }
      
  }

  const Int_t nareas = fGeom->GetNumAreas();

  if (fAverageHiGainAreas->GetEntries()==0)
  {
      fAverageHiGainAreas->Expand(nareas);

      for (Int_t j=0; j<nareas; j++)
      {
	  //
	  // Oscillating pixels
	  //
	  (*fAverageHiGainAreas)[j] = 
            new MHCalibrationChargeHiGainPix("AverageHiGainArea",
                                             "Average HiGain FADC sums of pixel area idx ");
          MHCalibrationChargeHiGainPix &hist = GetAverageHiGainArea(j);

          hist.GetHGausHist()->SetTitle("Summed FADC slices average Hi Gain pixels Area Idx ");
          hist.GetHAbsTime()->SetTitle("Absolute Arrival Time average Hi Gain pixels Area Idx ");
          hist.SetChargeFirst(-1000.);
          hist.SetChargeLast(4000.);
          hist.SetChargeNbins(4000);
	  hist.Init();
	  hist.ChangeHistId(j);
          hist.SetEventFrequency(fPulserFrequency);
      }
  }

  if (fAverageLoGainAreas->GetEntries()==0)
  {
      fAverageLoGainAreas->Expand(nareas);
  
      for (Int_t j=0; j<nareas; j++)
      {
	  //
	  // Oscillating pixels
	  //
	  (*fAverageLoGainAreas)[j] = 
            new MHCalibrationChargeLoGainPix("AverageLoGainArea",
                                             "Average LoGain FADC sums of pixel area idx ");
          MHCalibrationChargeLoGainPix &hist = GetAverageLoGainArea(j);

          hist.GetHGausHist()->SetTitle("Summed FADC slices average Lo Gain pixels Area Idx ");
          hist.GetHAbsTime()->SetTitle("Absolute Arrival Time average Lo Gain pixels Area Idx ");
          hist.SetChargeFirst(-1000.);
          hist.SetChargeLast(4000.);
          hist.SetChargeNbins(4000);
	  hist.Init();
	  hist.ChangeHistId(j);
          hist.SetEventFrequency(fPulserFrequency);
      }
  }

  fAverageAreaNum.        Set(nareas);
  fAverageAreaSat.        Set(nareas);           
  fAverageAreaSigma.      Set(nareas);      
  fAverageAreaSigmaErr.   Set(nareas);   
  fAverageAreaRelSigma.   Set(nareas);   
  fAverageAreaRelSigmaErr.Set(nareas);
    
  for (Int_t i=0; i<n; i++)
    {
      if ((*this)[i].IsExcluded())
        continue;

      const Int_t aidx  = (*fGeom)[i].GetAidx();
      fAverageAreaNum[aidx]++;
    }


  const Int_t nsectors = fGeom->GetNumSectors();

  if (fAverageHiGainSectors->GetEntries()==0)
  {
      fAverageHiGainSectors->Expand(nsectors);

      for (Int_t j=0; j<nsectors; j++)
      {
	  //
	  // Oscillating pixels
	  //
	  (*fAverageHiGainSectors)[j] = 
            new MHCalibrationChargeHiGainPix("AverageHiGainSector",
                                             "Average HiGain FADC sums of pixel sector ");
          MHCalibrationChargeHiGainPix &hist = GetAverageHiGainSector(j);

          hist.GetHGausHist()->SetTitle("Summed FADC slices average Hi Gain pixels Sector ");
          hist.GetHAbsTime()->SetTitle("Absolute Arrival Time average Hi Gain pixels Sector ");
          hist.SetChargeFirst(-1000.);
          hist.SetChargeLast(4000.);
          hist.SetChargeNbins(4000);
	  hist.Init();
	  hist.ChangeHistId(j);
          hist.SetEventFrequency(fPulserFrequency);
      }
  }

  if (fAverageLoGainSectors->GetEntries()==0)
  {
      fAverageLoGainSectors->Expand(nsectors);
  
      for (Int_t j=0; j<nsectors; j++)
      {
	  //
	  // Oscillating pixels
	  //
	  (*fAverageLoGainSectors)[j] = 
            new MHCalibrationChargeLoGainPix("AverageLoGainSector",
                                             "Average LoGain FADC sums of pixel sector ");
          MHCalibrationChargeLoGainPix &hist = GetAverageLoGainSector(j);

          hist.GetHGausHist()->SetTitle("Summed FADC slices average Lo Gain pixels Sector ");
          hist.GetHAbsTime()->SetTitle("Absolute Arrival Time average Lo Gain pixels Sector ");
          hist.SetChargeFirst(-1000.);
          hist.SetChargeLast(4000.);
          hist.SetChargeNbins(4000);
	  hist.Init();
	  hist.ChangeHistId(j);
          hist.SetEventFrequency(fPulserFrequency);
      }
  }

  fAverageSectorNum.        Set(nsectors);

  for (Int_t i=0; i<n; i++)
    {
      if ((*this)[i].IsExcluded())
        continue;

      const Int_t sector  = (*fGeom)[i].GetSector();
      fAverageSectorNum[sector]++;
    }

  
  return kTRUE;
}
  

// --------------------------------------------------------------------------
Bool_t MHCalibrationChargeCam::Fill(const MParContainer *par, const Stat_t w)
{

  MExtractedSignalCam *signal = (MExtractedSignalCam*)par;
  if (!signal)
    {
      *fLog << err << "No argument in MExtractedSignalCam::Fill... abort." << endl;
      return kFALSE;
    }
  
  const Int_t npixels  = signal->GetSize();
  const Int_t lofirst  = signal->GetFirstUsedSliceLoGain();
  const Int_t nareas   = fGeom->GetNumAreas();
  const Int_t nsectors = fGeom->GetNumSectors();

  if (fHiGainArray->GetEntries() != npixels)
    {
      *fLog << err << "ERROR - Size mismatch in number of pixels ... abort." << endl;
      return kFALSE;
    }

  if (fLoGainArray->GetEntries() != npixels)
    {
      *fLog << err << "ERROR - Size mismatch in number of pixels ... abort." << endl;
      return kFALSE;
    }

  if (fAverageHiGainAreas->GetEntries() != nareas)
    {
      *fLog << err << "ERROR - Size mismatch in number of areas ... abort." << endl;
      return kFALSE;
    }

  if (fAverageLoGainAreas->GetEntries() != nareas)
    {
      *fLog << err << "ERROR - Size mismatch in number of areas ... abort." << endl;
      return kFALSE;
    }

  if (fAverageHiGainSectors->GetEntries() != nsectors)
    {
      *fLog << err << "ERROR - Size mismatch in number of sectors ... abort." << endl;
      return kFALSE;
    }

  if (fAverageLoGainSectors->GetEntries() != nsectors)
    {
      *fLog << err << "ERROR - Size mismatch in number of sectors ... abort." << endl;
      return kFALSE;
    }

  Float_t sumhiarea  [nareas],   sumloarea  [nareas],   timehiarea  [nareas],   timeloarea  [nareas];
  Float_t sumhisector[nsectors], sumlosector[nsectors], timehisector[nsectors], timelosector[nsectors];
  Int_t   sathiarea  [nareas],   satloarea  [nareas];
  Int_t   sathisector[nsectors], satlosector[nsectors];

  for (UInt_t j=0; j<nareas; j++)
    {
      sumhiarea  [j] = sumloarea  [j] = timehiarea  [j] =  timeloarea  [j] = 0.;
      sumhisector[j] = sumlosector[j] = timehisector[j] =  timelosector[j] = 0.;
      sathiarea  [j] = satloarea  [j] = 0;
      sathisector[j] = satlosector[j] = 0;
    }
  
  for (Int_t i=0; i<npixels; i++)
    {

      if ((*this)[i].IsExcluded())
	continue;

      const MExtractedSignalPix &pix = (*signal)[i];
      
      const Float_t sumhi  = pix.GetExtractedSignalHiGain();
      const Float_t sumlo  = pix.GetExtractedSignalLoGain();
      
      (*this)[i].FillHistAndArray(sumhi);
      (*this)(i).FillHistAndArray(sumlo);

      const Int_t sathi = (Int_t)pix.GetNumHiGainSaturated();
      const Int_t satlo = (Int_t)pix.GetNumLoGainSaturated();

      (*this)[i].SetSaturated(sathi); 
      (*this)(i).SetSaturated(satlo); 

      const Int_t aidx   = (*fGeom)[i].GetAidx();
      const Int_t sector = (*fGeom)[i].GetSector();

      sumhiarea[aidx]  += sumhi;
      sumloarea[aidx]  += sumlo;
      sathiarea[aidx]  += sathi;
      satloarea[aidx]  += satlo;

      sumhisector[sector]  += sumhi;
      sumlosector[sector]  += sumlo;
      sathisector[sector]  += sathi;
      satlosector[sector]  += satlo;
    }

  MRawEvtPixelIter pixel(fRawEvt);
  while (pixel.Next())
    {
      
      const UInt_t pixid = pixel.GetPixelId();
       if ((*this)[pixid].IsExcluded())
         continue;

      const Float_t timehi = (Float_t)pixel.GetIdxMaxHiGainSample();
      const Float_t timelo = (Float_t)pixel.GetIdxMaxLoGainSample(lofirst);

      (*this)[pixid].FillAbsTime(timehi);
      (*this)(pixid).FillAbsTime(timelo);

      const Int_t aidx   = (*fGeom)[pixid].GetAidx();
      const Int_t sector = (*fGeom)[pixid].GetSector();

      timehiarea[aidx] += timehi;
      timeloarea[aidx] += timelo;

      timehisector[sector] += timehi;
      timelosector[sector] += timelo;
    }
  

  for (UInt_t j=0; j<nareas; j++)
    {

      const Int_t npix = fAverageAreaNum[j];

      MHCalibrationChargeHiGainPix &hipix = GetAverageHiGainArea(j);
      MHCalibrationChargeLoGainPix &lopix = GetAverageLoGainArea(j);

      hipix.FillHistAndArray(sumhiarea[j]/npix);
      lopix.FillHistAndArray(sumloarea[j]/npix);

      hipix.SetSaturated((Float_t)sathiarea[j]/npix); 
      lopix.SetSaturated((Float_t)satloarea[j]/npix); 

      hipix.FillAbsTime(timehiarea[j]/npix);
      lopix.FillAbsTime(timeloarea[j]/npix);

    }

  for (UInt_t j=0; j<nsectors; j++)
    {

      const Int_t npix = fAverageSectorNum[j];

      MHCalibrationChargeHiGainPix &hipix = GetAverageHiGainSector(j);
      MHCalibrationChargeLoGainPix &lopix = GetAverageLoGainSector(j);

      hipix.FillHistAndArray(sumhisector[j]/npix);
      lopix.FillHistAndArray(sumlosector[j]/npix);

      hipix.SetSaturated((Float_t)sathisector[j]/npix); 
      lopix.SetSaturated((Float_t)satlosector[j]/npix); 

      hipix.FillAbsTime(timehisector[j]/npix);
      lopix.FillAbsTime(timelosector[j]/npix);

    }

  return kTRUE;
}

// --------------------------------------------------------------------------
//
// 1) Return if the charge distribution is already succesfully fitted  
//    or if the histogram is empty
// 2) Fit the histograms with a Gaussian
// 3) In case of failure set the bit kFitted to false and take histogram means and RMS
// 4) Check for pickup noise
// 5) Check the fourier spectrum 
// 5) Retrieve the results and store them in this class
//
Bool_t MHCalibrationChargeCam::Finalize()
{

  for (Int_t i=0; i<fHiGainArray->GetSize(); i++)
    {
      
      if ((*this)[i].IsExcluded())
        continue;
      
      MHCalibrationChargeHiGainPix &histhi = (*this)[i];
      MCalibrationChargePix        &pix    = (*fCam)[i];
      MBadPixelsPix                &bad    = (*fBadPixels)[i];
      
      FinalizeHiGainHists(histhi,pix,bad);
      
    }
  
  for (Int_t i=0; i<fLoGainArray->GetSize(); i++)
    {
      
      if ((*this)(i).IsExcluded())
        continue;
      
      MHCalibrationChargeLoGainPix &histlo = (*this)(i);
      MCalibrationChargePix        &pix    = (*fCam)[i];
      MBadPixelsPix                &bad    = (*fBadPixels)[i];
      
      FinalizeLoGainHists(histlo,pix,bad);
      
    }
  
  for (Int_t j=0; j<fAverageHiGainAreas->GetSize(); j++)
    {
      
      MHCalibrationChargeHiGainPix &histhi = GetAverageHiGainArea(j);      
      MCalibrationChargePix        &pix    = fCam->GetAverageArea(j);
      MBadPixelsPix                &bad    = fCam->GetAverageBadArea(j);        
      
      FinalizeHiGainHists(histhi,pix,bad);
    }
  
  for (Int_t j=0; j<fAverageLoGainAreas->GetSize(); j++)
    {
      
      MHCalibrationChargeLoGainPix &histlo = GetAverageLoGainArea(j);      
      MCalibrationChargePix        &pix    = fCam->GetAverageArea(j);
      MBadPixelsPix                &bad    = fCam->GetAverageBadArea(j);        
      
      FinalizeLoGainHists(histlo,pix,bad);
    }

  for (Int_t j=0; j<fAverageHiGainSectors->GetSize(); j++)
    {
      
      MHCalibrationChargeHiGainPix &histhi = GetAverageHiGainSector(j);      
      MCalibrationChargePix        &pix    = fCam->GetAverageSector(j);
      MBadPixelsPix                &bad    = fCam->GetAverageBadSector(j);        
      
      FinalizeHiGainHists(histhi,pix,bad);
    }
  
  for (Int_t j=0; j<fAverageLoGainSectors->GetSize(); j++)
    {
      
      MHCalibrationChargeLoGainPix &histlo = GetAverageLoGainSector(j);      
      MCalibrationChargePix        &pix    = fCam->GetAverageSector(j);
      MBadPixelsPix                &bad    = fCam->GetAverageBadSector(j);        
      
      FinalizeLoGainHists(histlo,pix,bad);
    }
  
  for (Int_t j=0; j<fGeom->GetNumAreas();j++)
    {
      
      MCalibrationChargePix &pix = fCam->GetAverageArea(j);

      if (pix.IsHiGainSaturation())
        fAverageAreaSat[j]++;

      fAverageAreaSigma[j]    = pix.GetSigmaCharge    () * TMath::Sqrt((Float_t)fAverageAreaNum[j]);
      fAverageAreaSigmaErr[j] = pix.GetSigmaChargeErr () * TMath::Sqrt((Float_t)fAverageAreaNum[j]);

      pix.SetSigmaCharge   (fAverageAreaSigma[j]);
      pix.SetSigmaChargeErr(fAverageAreaSigmaErr[j]);

      fAverageAreaRelSigma[j]   = fAverageAreaSigma[j] / pix.GetMeanCharge();
      
      Float_t relsigmaerr       =  fAverageAreaSigmaErr[j]*fAverageAreaSigmaErr[j] 
                                / (fAverageAreaSigma[j]   *fAverageAreaSigma[j]   );
      relsigmaerr               += pix.GetMeanChargeErr()*pix.GetMeanChargeErr() 
                                / (pix.GetMeanCharge()   *pix.GetMeanCharge()   );
      relsigmaerr               *= fAverageAreaRelSigma[j];
      fAverageAreaRelSigmaErr[j] = TMath::Sqrt(relsigmaerr);

    }

  return kTRUE;
}

void MHCalibrationChargeCam::FinalizeHiGainHists(MHCalibrationChargeHiGainPix &hist, 
                                                 MCalibrationChargePix &pix, 
                                                 MBadPixelsPix &bad)
{

    if (hist.IsEmpty())
      {
	*fLog << warn << "Empty Hi Gain histogram in pixel: " << pix.GetPixId() << endl;
        bad.SetUnsuitable(MBadPixelsPix::kUnsuitableRun);        
	return;
      }
    
    if (hist.GetSaturated() > fNumHiGainSaturationLimit*hist.GetHGausHist()->GetEntries())
    {
      *fLog << warn << "Saturated Hi Gain histogram in pixel: " << pix.GetPixId() << endl;
      pix.SetHiGainSaturation();
      //      return;
    }
    
    //
    // 2) Fit the Hi Gain histograms with a Gaussian
    //
    if (!hist.FitGaus())
    //
    // 3) In case of failure set the bit Fitted to false and take histogram means and RMS
    //
      if (!hist.RepeatFit())
        {
          hist.BypassFit();
	  *fLog << warn << "Hi Gain could not be fitted in pixel: " << pix.GetPixId() << endl;
          bad.SetUncalibrated( MBadPixelsPix::kHiGainNotFitted );
          bad.SetUnsuitable(   MBadPixelsPix::kUnreliableRun   );
        }

    //
    // 4) Check for pickup
    //
    hist.CountPickup();
    
    //
    // 5) Check for oscillations
    // 
    hist.CreateFourierSpectrum();
    
    //
    // 6) Retrieve the results and store them in this class
    //
    pix.SetHiGainMeanCharge(     hist.GetMean()     );
    pix.SetHiGainMeanChargeErr(  hist.GetMeanErr()  );
    pix.SetHiGainSigmaCharge(    hist.GetSigma()    );
    pix.SetHiGainSigmaChargeErr( hist.GetSigmaErr() );
    pix.SetHiGainChargeProb    ( hist.GetProb()     );
    
    pix.SetAbsTimeMean         ( hist.GetAbsTimeMean());
    pix.SetAbsTimeRms          ( hist.GetAbsTimeRms() );
    
    pix.SetHiGainNumPickup     (  hist.GetPickup()           );
    
    if (!hist.IsFourierSpectrumOK())
      {
	*fLog << warn << "Oscillating Hi Gain in pixel: " << pix.GetPixId() << endl;
	bad.SetUncalibrated( MBadPixelsPix::kHiGainOscillating );
        bad.SetUnsuitable(   MBadPixelsPix::kUnreliableRun     );
      }
}


void MHCalibrationChargeCam::FinalizeLoGainHists(MHCalibrationChargeLoGainPix &hist, 
                                                 MCalibrationChargePix &pix, 
                                                 MBadPixelsPix &bad)
{

    if (hist.IsEmpty())
      {
        if (pix.IsHiGainSaturation())
          bad.SetUnsuitable(MBadPixelsPix::kUnsuitableRun);
        return;
      }

    if (hist.GetSaturated() > fNumLoGainSaturationLimit*hist.GetHGausHist()->GetEntries())
    {
      *fLog << warn << "Saturated Lo Gain histogram in pixel: " << pix.GetPixId() << endl;
      pix.SetLoGainSaturation();
      bad.SetUncalibrated( MBadPixelsPix::kLoGainSaturation ); 
      bad.SetUnsuitable(   MBadPixelsPix::kUnsuitableRun    );
      //      return;
    }

    //
    // 2) Fit the Lo Gain histograms with a Gaussian
    //
    if (!hist.FitGaus())
    //
    // 3) In case of failure set the bit kFitted to false and take histogram means and RMS
    //
      if (!hist.RepeatFit())
        {
          hist.BypassFit();
	  bad.SetUncalibrated( MBadPixelsPix::kLoGainNotFitted );
          if (pix.IsHiGainSaturation())
            bad.SetUnsuitable(MBadPixelsPix::kUnreliableRun);
        }

    //
    // 4) Check for pickup
    //
    hist.CountPickup();
    
    //
    // 5) Check for oscillations
    // 
    hist.CreateFourierSpectrum();
    
    //
    // 6) Retrieve the results and store them in this class
    //
    pix.SetLoGainMeanCharge(     hist.GetMean()     );
    pix.SetLoGainMeanChargeErr(  hist.GetMeanErr()  );
    pix.SetLoGainSigmaCharge(    hist.GetSigma()    );
    pix.SetLoGainSigmaChargeErr( hist.GetSigmaErr() );
    pix.SetLoGainChargeProb    ( hist.GetProb()     );
    
    if (pix.IsHiGainSaturation())
    {
	pix.SetAbsTimeMean     ( hist.GetAbsTimeMean());
	pix.SetAbsTimeRms      ( hist.GetAbsTimeRms() );
    }	    
    
    pix.SetLoGainNumPickup     (  hist.GetPickup()    );

    if (!hist.IsFourierSpectrumOK())
      {
        bad.SetUncalibrated( MBadPixelsPix::kLoGainOscillating );
        if (pix.IsHiGainSaturation())
          bad.SetUnsuitable(MBadPixelsPix::kUnreliableRun);
      }
}    



Bool_t MHCalibrationChargeCam::GetPixelContent(Double_t &val, Int_t idx, const MGeomCam &cam, Int_t type) const
{
  return kTRUE;
}

// --------------------------------------------------------------------------
//
// What MHCamera needs in order to draw an individual pixel in the camera
//
void MHCalibrationChargeCam::DrawPixelContent(Int_t idx) const
{
  (*this)[idx].DrawClone();
}


void MHCalibrationChargeCam::Draw(const Option_t *opt)
{

  const Int_t nareas = fAverageHiGainAreas->GetEntries();
  if (nareas == 0)
    return;

  TVirtualPad *pad = gPad ? gPad : MH::MakeDefCanvas(this);  
  pad->SetBorderMode(0);

  pad->Divide(2,nareas);

  for (Int_t i=0; i<nareas;i++) 
    {
      pad->cd(2*(i+1)-1);
      GetAverageHiGainArea(i).Draw(opt);

      if (!fAverageAreaSat[i])
        DrawAverageSigma(fAverageAreaSat[i], i,
                         fAverageAreaSigma[i],    fAverageAreaSigmaErr[i],
                         fAverageAreaRelSigma[i], fAverageAreaRelSigmaErr[i]);

      pad->cd(2*(i+1));
      GetAverageLoGainArea(i).Draw(opt);
      
      if (fAverageAreaSat[i])
        DrawAverageSigma(fAverageAreaSat[i], i,
                         fAverageAreaSigma[i], fAverageAreaSigmaErr[i],
                         fAverageAreaRelSigma[i], fAverageAreaRelSigmaErr[i]);
    }
}

void MHCalibrationChargeCam::DrawAverageSigma(Bool_t sat, Bool_t inner,
                                              Float_t sigma, Float_t sigmaerr,
                                              Float_t relsigma, Float_t relsigmaerr) const 
{
  
  if (sigma != 0)
    {
      
      TPad *newpad = new TPad("newpad","transparent",0,0,1,1);
      newpad->SetFillStyle(4000);
      newpad->Draw();
      newpad->cd();
      
      TPaveText *text = new TPaveText(sat? 0.1 : 0.35,0.7,sat ? 0.4 : 0.7,1.0);
      text->SetTextSize(0.07);
      const TString line1 = Form("%s%s%s",inner ? "Outer" : "Inner",
                                 " Pixels ", sat ? "Low Gain" : "High Gain");
      TText *txt1 = text->AddText(line1.Data());
      const TString line2 = Form("Sigma per Pixel: %2.2f #pm %2.2f",sigma,sigmaerr);
      TText *txt2 = text->AddText(line2.Data());
      const TString line3 = Form("Rel. Sigma per Pixel: %2.2f #pm %2.2f",relsigma,relsigmaerr);
      TText *txt3 = text->AddText(line3.Data());
      text->Draw("");
      
      text->SetBit(kCanDelete);
      txt1->SetBit(kCanDelete);
      txt2->SetBit(kCanDelete);
      txt3->SetBit(kCanDelete);
      newpad->SetBit(kCanDelete);
    }
}

