/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug 02/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
//  MHCalibrationRelTimePix
//
//  Histogram class for the relative arrival time calibration.
//  Holds the histogrammed arrival times,
//  derives from MHGausEvents, perform Fourier analysis
//
//////////////////////////////////////////////////////////////////////////////
#include "MHCalibrationRelTimePix.h"

#include <TH1.h>

ClassImp(MHCalibrationRelTimePix);

using namespace std;
//
const Int_t   MHCalibrationRelTimePix::fgChargeNbins    = 900;
const Axis_t  MHCalibrationRelTimePix::fgChargeFirst    = -13.;
const Axis_t  MHCalibrationRelTimePix::fgChargeLast     =  13.;
// --------------------------------------------------------------------------
//
// Default Constructor. 
//
// Sets: 
// - the default number for fChargeNbins  (fgChargeNbins)
// - the default number for fChargeFirst  (fgChargeFirst)
// - the default number for fChargeLast   (fgChargeLast)
//
// - the default name of the  fHGausHist ("HCalibrationRelTime")
// - the default title of the fHGausHist ("Distribution of Relative Arrival Times Pixel ")
// - the default x-axis title for fHGausHist ("FADC Slice")
// - the default y-axis title for fHGausHist ("Nr. of events")
//
// Initializes:
// - fPixId to -1
//
// Executes:
// - MHGausEvents::Clear()
//
MHCalibrationRelTimePix::MHCalibrationRelTimePix(const char *name, const char *title) 
    : fPixId(-1)
{ 

  fName  = name  ? name  : "MHCalibrationRelTimePix";
  fTitle = title ? title : "Histogrammed Calibration Relative Arrival Time events";

  SetChargeNbins();
  SetChargeFirst();
  SetChargeLast();

  // Create a large number of bins, later we will rebin
  fHGausHist.SetName("HCalibrationRelTime");
  fHGausHist.SetTitle("Distribution of Relative Arrival Times Pixel ");
  fHGausHist.SetXTitle("FADC Slice");
  fHGausHist.SetYTitle("Nr. of events");

  Clear();

}

// --------------------------------------------------------------------------
//
// Sets:
// - fPixId to -1
// 
// Executes:
// - MHGausEvents::Clear()
//
void MHCalibrationRelTimePix::Clear(Option_t *o)
{

  fPixId = -1;
  MHGausEvents::Clear();
  return;
}


// --------------------------------------------------------------------------
//
// Empty function to overload MHGausEvents::Reset()
//
void MHCalibrationRelTimePix::Reset()
{
}

// --------------------------------------------------------------------------
//
// Sets:
// - fHGausHist.SetBins(fChargeNbins,fChargeFirst,fChargeLast);
//
void MHCalibrationRelTimePix::Init()
{

  fHGausHist.SetBins(fChargeNbins,fChargeFirst,fChargeLast);

}

// --------------------------------------------------------------------------
//
// - Set fPixId to id
//
// Add id to names and titles of:
// - fHGausHist
//
void MHCalibrationRelTimePix::ChangeHistId(Int_t id)
{

  fPixId = id;

  fHGausHist.SetName(  Form("%s%d", fHGausHist.GetName(),  id));
  fHGausHist.SetTitle( Form("%s%d", fHGausHist.GetTitle(), id));

}


// ----------------------------------------------------------------------
//
// Renorm the results from FADC slices to times. 
// The parameters slicewidth is the inverse of the FADC frequency and has the unit ns.
//
void MHCalibrationRelTimePix::Renorm(const Float_t slicewidth)
{

  SetMean(     GetMean()    * slicewidth  );
  SetMeanErr(  GetMeanErr() * slicewidth  );
  SetSigma(    GetSigma()   * slicewidth  );
  SetSigmaErr( GetSigmaErr()* slicewidth  );
  
}

