/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug   11/2003 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                               
// MCalibrationCam                                               
//                                                               
// Base class for Camera Calibration results. 
//
// Contains TClonesArrays for the following objects:
// - fPixels:         Array of classes derived from MCalibrationPix, one entry 
//                    per pixel. Has to be created
// - fAverageAreas:   Array of classes derived from MCalibrationPix, one entry
//                    per pixel AREA. Has to be created
// - fAverageSectors: Array of classes derived from MCalibrationPix, one entry
//                    per camera SECTOR. Has to be created
//
// - fAverageBadAreas: Array of classes derived from MBadPixelsPix, one entry
//                     per pixel AREA. Is created automatically. 
// - fAverageBadSectors: Array of classes derived from MBadPixelsPix, one entry
//                    per camera SECTOR. Is created automatically.
//
// All TClonesArrays have to enlarged by the corresponding calls to (e.g. in MGeomApply): 
// - InitSize()
// - InitAverageAreas()
// - InitAverageSectors() 
//
/////////////////////////////////////////////////////////////////////////////
#include "MCalibrationCam.h"

#include <TClonesArray.h>

#include "MGeomCam.h"
#include "MGeomPix.h"

#include "MBadPixelsCam.h"
#include "MBadPixelsPix.h"

#include "MCalibrationPix.h"

ClassImp(MCalibrationCam);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor. 
//
// Set the following pointer to NULL:
// - fPixels
// - fAverageAreas
// - fAverageSectors
//
// Creates a TClonesArray of MBadPixelsPix containers for the TClonesArray's: 
// - fAverageBadAreas
// - fAverageBadSectors
// all initialized to 1 entry
//
// Later, a call to InitAverageAreas() and InitAverageSectors() has to be performed in order 
// to get the dimension correctly.
//
MCalibrationCam::MCalibrationCam(const char *name, const char *title)
    : fPixels(NULL), fAverageAreas(NULL), fAverageSectors(NULL)
{
    fName  = name  ? name  : "MCalibrationCam";
    fTitle = title ? title : "Base class Storage container for Camera Calibration";

    fAverageBadAreas    = new TClonesArray("MBadPixelsPix",1);
    fAverageBadSectors  = new TClonesArray("MBadPixelsPix",1);

}

// --------------------------------------------------------------------------
//
// Deletes the following TClonesArray's of MCalibrationPix containers (if exist):
// - fPixels
// - fAverageAreas
// - fAverageSectors
//  
// Deletes the following TClonesArray's of MBadPixelsPix containers (if exist):
// - fAverageBadAreas
// - fAverageBadSectors
//
MCalibrationCam::~MCalibrationCam()
{

  //
  // delete fPixels should delete all Objects stored inside
  // 
  if (fPixels)
    delete fPixels;

  if (fAverageAreas)
    delete fAverageAreas;

  if (fAverageSectors)
    delete fAverageSectors;

  delete fAverageBadAreas;
  delete fAverageBadSectors;
  
}

// --------------------------------------
//
// Calls the ForEach macro for the TClonesArray fPixels with the argument Clear()
// 
// Loops over the fAverageAreas, calling the function Clear() for 
// every entry in:
// - fAverageAreas
// - fAverageBadAreas
// 
// Loops over the fAverageSectors, calling the function Clear() for 
// every entry in:
// - fAverageSectors
// - fAverageBadSectors
// 
void MCalibrationCam::Clear(Option_t *o)
{

  fPixels->ForEach(TObject, Clear)();

  //
  // another ForEach does not compile, thus have to do the loop ourselves:
  //
  for (Int_t i=0;i<GetAverageAreas();i++)
    {
      fAverageAreas[i].Clear();
      fAverageBadAreas[i].Clear();
    }

  //
  // another ForEach does not compile, thus have to do the loop ourselves:
  //
  for (Int_t i=0;i<GetAverageSectors();i++)
    {
      fAverageSectors[i].Clear();
      fAverageBadSectors[i].Clear();
    }
  
  return;
}

// -------------------------------------------------------------------
//
// Calls TClonesArray::ExpandCreate() for fPixels
//
void MCalibrationCam::InitSize(const UInt_t i)
{
  fPixels->ExpandCreate(i);
}

// -------------------------------------------------------------------
//
// Calls TClonesArray::ExpandCreate() for:
// - fAverageAreas
// - fAverageBadAreas
//
void MCalibrationCam::InitAverageAreas(const UInt_t i)
{
  fAverageAreas->ExpandCreate(i);
  fAverageBadAreas->ExpandCreate(i);
}

// -------------------------------------------------------------------
//
// Calls TClonesArray::ExpandCreate() for:
// - fAverageSectors
// - fAverageBadSectors
//
void MCalibrationCam::InitAverageSectors(const UInt_t i)
{
  fAverageSectors->ExpandCreate(i);
  fAverageBadSectors->ExpandCreate(i);
}

// --------------------------------------------------------------------------
//
// Returns the current size of the TClonesArray fPixels 
// independently if the MCalibrationPix is filled with values or not.
//
Int_t MCalibrationCam::GetSize() const
{
  return fPixels->GetEntriesFast();
}

// --------------------------------------------------------------------------
//
// Returns the current size of the TClonesArray fAverageAreas
// independently if the MCalibrationPix is filled with values or not.
//
Int_t MCalibrationCam::GetAverageAreas() const
{
  return fAverageAreas->GetEntriesFast();
}

// --------------------------------------------------------------------------
//
// Returns the current size of the TClonesArray fAverageSectors
// independently if the MCalibrationPix is filled with values or not.
//
Int_t MCalibrationCam::GetAverageSectors() const
{
  return fAverageSectors->GetEntriesFast();
}


// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
MCalibrationPix &MCalibrationCam::operator[](UInt_t i)
{
  return *static_cast<MCalibrationPix*>(fPixels->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
const MCalibrationPix &MCalibrationCam::operator[](UInt_t i) const
{
  return *static_cast<MCalibrationPix*>(fPixels->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th average pixel (area number)
//
MCalibrationPix &MCalibrationCam::GetAverageArea(UInt_t i)
{
  return *static_cast<MCalibrationPix*>(fAverageAreas->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th average pixel (area number)
//
const MCalibrationPix &MCalibrationCam::GetAverageArea(UInt_t i) const 
{
  return *static_cast<MCalibrationPix*>(fAverageAreas->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th average pixel (sector number)
//
MCalibrationPix &MCalibrationCam::GetAverageSector(UInt_t i)
{
  return *static_cast<MCalibrationPix*>(fAverageSectors->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th average pixel (sector number)
//
const MCalibrationPix &MCalibrationCam::GetAverageSector(UInt_t i) const 
{
  return *static_cast<MCalibrationPix*>(fAverageSectors->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th average pixel (area number)
//
MBadPixelsPix &MCalibrationCam::GetAverageBadArea(UInt_t i)
{
  return *static_cast<MBadPixelsPix*>(fAverageBadAreas->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th average pixel (area number)
//
const MBadPixelsPix &MCalibrationCam::GetAverageBadArea(UInt_t i) const 
{
  return *static_cast<MBadPixelsPix*>(fAverageBadAreas->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th average pixel (sector number)
//
MBadPixelsPix &MCalibrationCam::GetAverageBadSector(UInt_t i)
{
  return *static_cast<MBadPixelsPix*>(fAverageBadSectors->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th average pixel (sector number)
//
const MBadPixelsPix &MCalibrationCam::GetAverageBadSector(UInt_t i) const 
{
  return *static_cast<MBadPixelsPix*>(fAverageBadSectors->UncheckedAt(i));
}



// --------------------------------------------------------------------------
//
// Dummy needed for compilation with MCamEvent
//
Bool_t MCalibrationCam::GetPixelContent(Double_t &val, Int_t idx, const MGeomCam &cam, Int_t type) const
{
  return kTRUE;
}



// --------------------------------------------------------------------------
//
// Calls MCalibrationPix::DrawClone()
//
void MCalibrationCam::DrawPixelContent(Int_t idx) const
{
  (*this)[idx].DrawClone();
}

