/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 09/2003 <mailto:tbretz@astro.uni-wuerzburg.de>
!              Markus Gaug,  03/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
//  MGeomApply
//
//  Applies the geometry to geometry dependant containers.
//
//  It changes the size of the arrays in the containers to a size
//  matching the number of pixels, eg:
//
//   MPedestalCam
//   MCalibrationChargeCam
//   MCalibrationQECam
//   MPedPhotCam
//   MExtractedSignalCam
//   MBlindPixels
//   MArrivalTimeCam
//   MArrivalTimePix
//
//  It uses the geometry (MGeomCam) found in the parameter list.
//  If it is not found the task tries to create the geometry
//  specified in the constructor. The default geometry is
//  MGeomCamMagic.
// 
//  Input Containers:
//   [MGeomCam]
//
//  Output Containers:
//   [MPedestalCam]
//   [MCalibrationChargeCam]
//   [MPedPhotCam]
//   [MExtractedSignalCam]
//   [MBlindPixels]
//   [MArrivalTime]
//   [MArrivalTimeCam]
//
//////////////////////////////////////////////////////////////////////////////
#include "MGeomApply.h"

#include <fstream>

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"

#include "MGeomCam.h"
#include "MPedestalCam.h"
#include "MCalibrationChargeCam.h"
#include "MCalibrationRelTimeCam.h"
#include "MCalibrationQECam.h"
#include "MCalibrationPedCam.h"
#include "MPedPhotCam.h"
#include "MExtractedSignalCam.h"
#include "MBlindPixels.h"
#include "MArrivalTimeCam.h"
#include "MBadPixelsCam.h"

ClassImp(MGeomApply);

using namespace std;

// --------------------------------------------------------------------------
//
//  Default constructor. MGeomCamMagic is the default geometry.
//
MGeomApply::MGeomApply(const char *name, const char *title) : fGeomName("MGeomCamMagic")
{
    fName  = name  ? name  : "MGeomApply";
    fTitle = title ? title : "Task to apply geometry settings";
}

// --------------------------------------------------------------------------
//
//  Try to find 'MGeomCam' in the Parameter List. If it is not found,
//  try to create a fGeomName object.
//
Int_t MGeomApply::PreProcess(MParList *pList)
{
    MGeomCam *cam = (MGeomCam*)pList->FindObject(AddSerialNumber("MGeomCam"));
    if (cam)
        return kTRUE;

    cam = (MGeomCam*)pList->FindCreateObj(AddSerialNumber(fGeomName), "MGeomCam");

    return cam ? kTRUE : kFALSE;
}

// --------------------------------------------------------------------------
//
//  Try to find 'MGeomCam' in the Parameter List. If it is not found,
//  processing is stopped.
//
//  The size of MPedestalCam and MBlindPixels is set accordingly.
//
Bool_t MGeomApply::ReInit(MParList *pList)
{
    MGeomCam *cam = (MGeomCam*)pList->FindObject(AddSerialNumber("MGeomCam"));
    if (!cam)
    {
        *fLog << err << GetDescriptor() << ": No MGeomCam found... aborting." << endl;
        return kFALSE;
    }

    // FIXME, workaround: this call to CalcPixRatio is here just to allow
    // the use of some camera files from the 0.7 beta version in which the 
    // array containing pixel ratios is not initialized.
    cam->CalcPixRatio();

    MPedestalCam *ped = (MPedestalCam*)pList->FindObject(AddSerialNumber("MPedestalCam"));
    if (ped)
        ped->InitSize(cam->GetNumPixels());

    MCalibrationChargeCam *cal = 
      (MCalibrationChargeCam*)pList->FindObject(AddSerialNumber("MCalibrationChargeCam"));
    if (cal)
      {
        cal->InitSize          ( cam->GetNumPixels()  );
        cal->InitAverageAreas  ( cam->GetNumAreas()   );
        cal->InitAverageSectors( cam->GetNumSectors() );
      }

    MCalibrationRelTimeCam *cat = 
      (MCalibrationRelTimeCam*)pList->FindObject(AddSerialNumber("MCalibrationRelTimeCam"));
    if (cat)
      {
        cat->InitSize          ( cam->GetNumPixels()  );
        cat->InitAverageAreas  ( cam->GetNumAreas()   );
        cat->InitAverageSectors( cam->GetNumSectors() );
      }
    
    MCalibrationQECam *qe = 
      (MCalibrationQECam*)pList->FindObject(AddSerialNumber("MCalibrationQECam"));
    if (qe)
      {
        qe->InitSize           ( cam->GetNumPixels()  );
        qe->InitAverageAreas   ( cam->GetNumAreas()   );
        qe->InitAverageSectors ( cam->GetNumSectors() );
      }
    
    MCalibrationPedCam *pcam = 
      (MCalibrationPedCam*)pList->FindObject(AddSerialNumber("MCalibrationPedCam"));
    if (pcam)
      {
        pcam->InitSize           ( cam->GetNumPixels()  );
        pcam->InitAverageAreas   ( cam->GetNumAreas()   );
        pcam->InitAverageSectors ( cam->GetNumSectors() );
      }
    
    MPedPhotCam *pedphot = (MPedPhotCam*)pList->FindObject(AddSerialNumber("MPedPhotCam"));
    if (pedphot)
        pedphot->InitSize(cam->GetNumPixels());


    MExtractedSignalCam *ext = (MExtractedSignalCam*)pList->FindObject(AddSerialNumber("MExtractedSignalCam"));
    if (ext)
        ext->InitSize(cam->GetNumPixels());


    MBlindPixels *bnd = (MBlindPixels*)pList->FindObject(AddSerialNumber("MBlindPixels"));
    if (bnd)
        bnd->InitSize(cam->GetNumPixels());

    MArrivalTimeCam *tme = (MArrivalTimeCam*)pList->FindObject(AddSerialNumber("MArrivalTimeCam"));
    if (tme)
      tme->InitSize(cam->GetNumPixels());

    MBadPixelsCam *bad = (MBadPixelsCam*)pList->FindObject(AddSerialNumber("MBadPixelsCam"));
    if (bad)
      bad->InitSize(cam->GetNumPixels());

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Implementation of SavePrimitive. Used to write the call to a constructor
// to a macro. In the original root implementation it is used to write
// gui elements to a macro-file.
//
void MGeomApply::StreamPrimitive(ofstream &out) const
{
    out << "   " << ClassName() << " " << GetUniqueName() << "(\"";
    out << "\"" << fName << "\", \"" << fTitle << "\");" << endl;

    if (fGeomName.IsNull())
        return;

    out << "   " << GetUniqueName() << ".SetGeometry(\"";
    out << fGeomName << "\");" << endl;
}
