/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug   02/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// MCalibrationChargePINDiode                                                            //
//                                                                         //
// This is the storage container to hold informations about the pedestal   //
// (offset) value of one Pixel (PMT).                                      //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
#include "MCalibrationChargePINDiode.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MCalibrationChargePINDiode);

using namespace std;

const Float_t MCalibrationChargePINDiode::fgChargeLimit        = 3.;
const Float_t MCalibrationChargePINDiode::fgChargeErrLimit     = 0.;    
const Float_t MCalibrationChargePINDiode::fgChargeRelErrLimit  = 1.; 

const Float_t MCalibrationChargePINDiode::fgConvPhotons        = -1.; 
const Float_t MCalibrationChargePINDiode::fgConvPhotonsErr     = -1.; 
//
// Area of Inner Pixel w.r.t. PIN Diode (which is 1 cm2)
//
// Distance of PIN Diode to pulser D1:   1.5  +- 0.3 m
// Distance of Inner Pixel to pulser D2: 18.0 +- 0.5 m
//
//
//                 D1*D1
// conversion C = ------ = 0.0069
//                 D2*D2
//
// Delta C / C  = 2 * Sqrt( (Delta D1/D1)2 + (Delta D2/D2)2 )
// Delta C / C  = 0.4
// 
// C = 0.007 +- 0.003
//
const Float_t MCalibrationChargePINDiode::gkFluxCameravsPINDiode      = 0.007;
const Float_t MCalibrationChargePINDiode::gkFluxCameravsPINDiodeErr   = 0.003;
//
// Average QE of the PIN Diode
//
const Float_t MCalibrationChargePINDiode::gkPINDiodeQEGreen    = -1.0;
const Float_t MCalibrationChargePINDiode::gkPINDiodeQEBlue     = -1.0;
const Float_t MCalibrationChargePINDiode::gkPINDiodeQEUV       = -1.0;
const Float_t MCalibrationChargePINDiode::gkPINDiodeQECT1      = -1.0;
//
// Average QE of the PIN Diode
//
const Float_t MCalibrationChargePINDiode::gkPINDiodeQEGreenErr = -1.0;
const Float_t MCalibrationChargePINDiode::gkPINDiodeQEBlueErr  = -1.0;
const Float_t MCalibrationChargePINDiode::gkPINDiodeQEUVErr    = -1.0;
const Float_t MCalibrationChargePINDiode::gkPINDiodeQECT1Err   = -1.0;

const Float_t MCalibrationChargePINDiode::gkPINDiodeArea       = 100;
// --------------------------------------------------------------------------
//
// Default Constructor. 
//
MCalibrationChargePINDiode::MCalibrationChargePINDiode(const char *name, const char *title)
    : fCalibFlags(0)
{

  fName  = name  ? name  : "MCalibrationChargePINDiode";
  fTitle = title ? title : "Container of the fit results of MHCalibrationChargePINDiode";

  Clear();

  SetChargeLimit();
  SetChargeErrLimit();
  SetChargeRelErrLimit();

  SetConvPhotons();
  SetConvPhotonsErr();

  SetOscillating          ( kFALSE );
  SetExcluded             ( kFALSE );
}

// ------------------------------------------------------------------------
//
// Invalidate values
//
void MCalibrationChargePINDiode::Clear(Option_t *o)
{

  SetChargeFitValid     ( kFALSE );
  SetTimeFitValid       ( kFALSE );
    
  fPed                              =  -1.;
  fPedRms                           =  -1.;

  fRmsChargeMean                    =  -1.;
  fRmsChargeMeanErr                 =  -1.;
  fRmsChargeSigma                   =  -1.;  
  fRmsChargeSigmaErr                =  -1.;

  fAbsTimeMean                      =  -1.;
  fAbsTimeRms                       =  -1.;

  fConvertedPhotons                 =  -1.;
  fConvertedPhotonsErr              =  -1.;

  fMeanFluxOutsidePlexiglass        =  -1.;  
  fMeanFluxErrOutsidePlexiglass     =  -1.;

}


// --------------------------------------------------------------------------
//
// Set the pedestals from outside
//
void MCalibrationChargePINDiode::SetPedestal(Float_t ped, Float_t pedrms)
{

  fPed    = ped;    
  fPedRms = pedrms;
  
}

// --------------------------------------------------------------------------
//
// Set the Oscillating Bit from outside 
//
void  MCalibrationChargePINDiode::SetOscillating( const Bool_t b)
{
    b ? SETBIT(fCalibFlags,kOscillating) : CLRBIT(fCalibFlags,kOscillating);
}


// --------------------------------------------------------------------------
//
// Set the Excluded Bit from outside 
//
void MCalibrationChargePINDiode::SetChargeFitValid(Bool_t b )    
{ 
  b ?  SETBIT(fCalibFlags, kChargeFitValid) : CLRBIT(fCalibFlags, kChargeFitValid); 
}

// --------------------------------------------------------------------------
//
// Set the Excluded Bit from outside 
//
void MCalibrationChargePINDiode::SetTimeFitValid(Bool_t b )    
{ 
  b ?  SETBIT(fCalibFlags, kTimeFitValid) : CLRBIT(fCalibFlags, kTimeFitValid); 
}

void MCalibrationChargePINDiode::SetFluxOutsidePlexiglassAvailable (const Bool_t b)
{
  b ?  SETBIT(fCalibFlags, kFluxOutsidePlexiglassAvailable) : CLRBIT(fCalibFlags, kFluxOutsidePlexiglassAvailable); 
}

Bool_t MCalibrationChargePINDiode::IsChargeFitValid() const 
{
  return TESTBIT(fCalibFlags, kChargeFitValid);  
}

Bool_t MCalibrationChargePINDiode::IsTimeFitValid()   const 
{
  return TESTBIT(fCalibFlags, kTimeFitValid);  
}

Bool_t MCalibrationChargePINDiode::CalcFluxOutsidePlexiglass()
{

  if (IsChargeFitValid())
    return kFALSE;
  
  // Start calculation of number of photons per mm^2 on the camera
  fMeanFluxOutsidePlexiglass  = fConvertedPhotons * gkPINDiodeArea;
  // Correct for the distance between camera and PIN Diode and for different areas.
  fMeanFluxOutsidePlexiglass *= gkFluxCameravsPINDiode;

  // Start calculation of number of photons relative Variance (!!)
  fMeanFluxErrOutsidePlexiglass  = fConvertedPhotonsErr * fConvertedPhotonsErr 
                                 / fConvertedPhotons    / fConvertedPhotons    ;
  fMeanFluxErrOutsidePlexiglass += gkFluxCameravsPINDiodeErr*gkFluxCameravsPINDiodeErr
                                 / gkFluxCameravsPINDiode/gkFluxCameravsPINDiode;
  
  switch (fColor)
    {
    case kGREEN:
      fMeanFluxOutsidePlexiglass    /= gkPINDiodeQEGreen;
      fMeanFluxErrOutsidePlexiglass += gkPINDiodeQEGreenErr*gkPINDiodeQEGreenErr
                                     / gkPINDiodeQEGreen/gkPINDiodeQEGreen;
      break;
    case kBLUE:
      fMeanFluxOutsidePlexiglass    /= gkPINDiodeQEBlue;
      fMeanFluxErrOutsidePlexiglass += gkPINDiodeQEBlueErr*gkPINDiodeQEBlueErr
                                     / gkPINDiodeQEBlue/gkPINDiodeQEBlue;
      break; 
    case kUV:
      fMeanFluxOutsidePlexiglass    /= gkPINDiodeQEUV;
      fMeanFluxErrOutsidePlexiglass += gkPINDiodeQEUVErr*gkPINDiodeQEUVErr
                                     / gkPINDiodeQEUV/gkPINDiodeQEUV;
      break;
    case kCT1:
    default:
      fMeanFluxOutsidePlexiglass    /= gkPINDiodeQECT1;
      fMeanFluxErrOutsidePlexiglass += gkPINDiodeQECT1Err*gkPINDiodeQECT1Err
                                     / gkPINDiodeQECT1/gkPINDiodeQECT1;
      break;
    }


  *fLog << inf << endl;
  *fLog << inf << " Mean Photon flux [ph/mm^2] outside Plexiglass: " 
        << fMeanFluxOutsidePlexiglass << endl;

  if (fMeanFluxOutsidePlexiglass < 0.)
      return kFALSE;

  if (fMeanFluxErrOutsidePlexiglass < 0.)
      return kFALSE;

  SetFluxOutsidePlexiglassAvailable();  

  // Finish calculation of errors -> convert from relative variance to absolute error
  fMeanFluxErrOutsidePlexiglass = TMath::Sqrt(fMeanFluxErrOutsidePlexiglass);
  fMeanFluxErrOutsidePlexiglass *= fMeanFluxOutsidePlexiglass;

  *fLog << inf << " Error on Photon flux [ph/mm^2] outside Plexiglass: " 
        << fMeanFluxErrOutsidePlexiglass << endl;
  *fLog << inf << endl;

  return kTRUE;
}



