/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug   02/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                     
// MCalibrationChargePINDiode       
//                                  
// Storage container of the fit results of the PIN Diode signal 
// (from MHCalibrationChargePINDiode). The PIN Diode is not yet working, so 
// is the documentation for the moment.
//
// Currently, the following numbers are implemented:
//
// Area of Inner Pixel Ai: 779.423 +- 0  mm^2 
// Area of PIN Diode   Ap: 100.000 +- 10 mm^2 
//
// Ratio of areas: 
//
// Distance of PIN Diode to pulser D1:   1.5  +- 0.3 m
// Distance of Inner Pixel to pulser D2: 18.0 +- 0.5 m
//
//                         Ai*D1*D1
// gkSolidAngleRatio   =   -------- = 0.054
//                         Ap*D2*D2
//
// gkSolidAngleRatioErr = 0.01
//
// 
/////////////////////////////////////////////////////////////////////////////
#include "MCalibrationChargePINDiode.h"
#include "MCalibrationChargePix.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MCalibrationChargePINDiode);

using namespace std;

const Float_t MCalibrationChargePINDiode::fgChargeToPhotons    = -1.; 
const Float_t MCalibrationChargePINDiode::fgChargeToPhotonsErr = -1.; 
const Float_t MCalibrationChargePINDiode::gkSolidAngleRatio      = 0.055;
const Float_t MCalibrationChargePINDiode::gkSolidAngleRatioErr   = 0.01;
//
// Average QE of the PIN Diode
//
const Float_t MCalibrationChargePINDiode::gkPINDiodeQEGreen    = -1.0;
const Float_t MCalibrationChargePINDiode::gkPINDiodeQEBlue     = -1.0;
const Float_t MCalibrationChargePINDiode::gkPINDiodeQEUV       = -1.0;
const Float_t MCalibrationChargePINDiode::gkPINDiodeQECT1      = -1.0;
//
// Average QE of the PIN Diode
//
const Float_t MCalibrationChargePINDiode::gkPINDiodeQEGreenErr = -1.0;
const Float_t MCalibrationChargePINDiode::gkPINDiodeQEBlueErr  = -1.0;
const Float_t MCalibrationChargePINDiode::gkPINDiodeQEUVErr    = -1.0;
const Float_t MCalibrationChargePINDiode::gkPINDiodeQECT1Err   = -1.0;

const Float_t MCalibrationChargePINDiode::gkPINDiodeArea       = 100;
// --------------------------------------------------------------------------
//
// Default Constructor. 
//
// Sets:
// - fCalibFlags to 0
// - fChargeToPhotons to fgChargeToPhotons
// - fChargeToPhotonsVar to fgChargeToPhotonsErr*fgChargeToPhotonsErr
// 
// Calls:
// - Clear()
//
MCalibrationChargePINDiode::MCalibrationChargePINDiode(const char *name, const char *title)
    : fCalibFlags(0)
{

  fName  = name  ? name  : "MCalibrationChargePINDiode";
  fTitle = title ? title : "Container of the fit results of MHCalibrationChargePINDiode";

  SetChargeToPhotons();
  SetChargeToPhotonsErr();

  Clear();

}

// ------------------------------------------------------------------------
//
// Sets:
// - all flags to kFALSE
// - all variables to -1.
//
// Calls: 
// - MCalibrationPix::Clear()
//
void MCalibrationChargePINDiode::Clear(Option_t *o)
{

  SetOscillating        ( kFALSE );
  SetChargeFitValid     ( kFALSE );
  SetTimeFitValid       ( kFALSE );
    
  fAbsTimeMean              =  -1.;
  fAbsTimeRms               =  -1.;
  fFluxOutsidePlexiglass    =  -1.;  
  fFluxOutsidePlexiglassVar =  -1.;
  fNumPhotons               =  -1.;
  fNumPhotonsVar            =  -1.;
  fPed                      =  -1.;
  fPedRms                   =  -1.;
  fRmsChargeMean            =  -1.;
  fRmsChargeMeanErr         =  -1.;
  fRmsChargeSigma           =  -1.;  
  fRmsChargeSigmaErr        =  -1.;

  MCalibrationPix::Clear();
}


// --------------------------------------------------------------------------
//
// Set the pedestals from outside
//
void MCalibrationChargePINDiode::SetPedestal(Float_t ped, Float_t pedrms)
{

  fPed    = ped;    
  fPedRms = pedrms;
  
}

// --------------------------------------------------------------------------
//
// Set the Oscillating Bit from outside 
//
void  MCalibrationChargePINDiode::SetOscillating( const Bool_t b)
{
    b ? SETBIT(fCalibFlags,kOscillating) : CLRBIT(fCalibFlags,kOscillating);
}


// --------------------------------------------------------------------------
//
// Set the ChargeFitValid Bit from outside 
//
void MCalibrationChargePINDiode::SetChargeFitValid(Bool_t b )    
{ 
  b ?  SETBIT(fCalibFlags, kChargeFitValid) : CLRBIT(fCalibFlags, kChargeFitValid); 
}

// --------------------------------------------------------------------------
//
// Set the TimeFitValid Bit from outside 
//
void MCalibrationChargePINDiode::SetTimeFitValid(Bool_t b )    
{ 
  b ?  SETBIT(fCalibFlags, kTimeFitValid) : CLRBIT(fCalibFlags, kTimeFitValid); 
}

// --------------------------------------------------------------------------
//
// Set the FluxOutsidePlexiglassAvailable Bit from outside 
//
void MCalibrationChargePINDiode::SetFluxOutsidePlexiglassAvailable (const Bool_t b)
{
  b ?  SETBIT(fCalibFlags, kFluxOutsidePlexiglassAvailable) 
    : CLRBIT(fCalibFlags, kFluxOutsidePlexiglassAvailable); 
}

// --------------------------------------------------------------------------
//
// Return -1 if fFluxOutsidePlexiglassVar is smaller than 0.
// Return square root of fFluxOutsidePlexiglassVar
// 
Float_t MCalibrationChargePINDiode::GetFluxOutsidePlexiglassErr() const
{
  if (fFluxOutsidePlexiglassVar < 0.)
    return -1.;
  
  return TMath::Sqrt(fFluxOutsidePlexiglassVar);
}


// --------------------------------------------------------------------------
//
// Test bit kChargeFitValid
//
Bool_t MCalibrationChargePINDiode::IsChargeFitValid() const 
{
  return TESTBIT(fCalibFlags, kChargeFitValid);  
}

// --------------------------------------------------------------------------
//
// Test bit kTimeFitValid
//
Bool_t MCalibrationChargePINDiode::IsTimeFitValid()   const 
{
  return TESTBIT(fCalibFlags, kTimeFitValid);  
}

// --------------------------------------------------------------------------
//
// Return kFALSE if IsChargeFitValid() is kFALSE
//
// Calculate fFluxOutsidePlexiglass with the formula:
// - fFluxOutsidePlexiglass    = fNumPhotons*gkSolidAngleRatio / gkPINDiodeQE (of the corr. colour) 
// - fFluxOutsidePlexiglassVar = sqrt( fNumPhotonsVar / ( fNumPhotons * fNumPhotons ) 
//                             + ( gkSolidAngleRatioErr * gkSolidAngleRatioErr / gkSolidAngleRatio / gkSolidAngleRatio )
//                             + ( gkPINDiodeQEErr * gkPINDiodeQEErr / gkPINDiodeQE / gkPINDiodeQE )
//                               ) * fFluxOutsidePlexiglass * * fFluxOutsidePlexiglass
//
// If the fFluxOutsidePlexiglass is smaller than 0., return kFALSE
// If the Variance is smaller than 0., return kFALSE
//
// SetFluxOutsidePlexiglassAvailable() and return kTRUE
//
Bool_t MCalibrationChargePINDiode::CalcFluxOutsidePlexiglass()
{

  if (IsChargeFitValid())
    return kFALSE;
  
  //
  // Start calculation of number of photons per mm^2 on an inner pixel:
  // Correct for the distance between camera and PIN Diode and for different areas.
  //
  fFluxOutsidePlexiglass  = fNumPhotons * gkSolidAngleRatio;

  //
  // Start calculation of number of photons relative Variance (!!)
  //
  const Float_t numphotRelVar    = fNumPhotonsVar / ( fNumPhotons * fNumPhotons  )  ;
  const Float_t solidangleRelVar = gkSolidAngleRatioErr* gkSolidAngleRatioErr
                              / ( gkSolidAngleRatio    * gkSolidAngleRatio   );
  Float_t fluxRelVar            = numphotRelVar + solidangleRelVar ;

  switch (fColor)
    {
    case kGREEN:
      fFluxOutsidePlexiglass /=  gkPINDiodeQEGreen;
      fluxRelVar             +=  gkPINDiodeQEGreenErr* gkPINDiodeQEGreenErr
                             / ( gkPINDiodeQEGreen   * gkPINDiodeQEGreen    );
      break;
    case kBLUE:
      fFluxOutsidePlexiglass /=  gkPINDiodeQEBlue;
      fluxRelVar             +=  gkPINDiodeQEBlueErr* gkPINDiodeQEBlueErr
                             / ( gkPINDiodeQEBlue   * gkPINDiodeQEBlue   );
      break; 
    case kUV:
      fFluxOutsidePlexiglass /=  gkPINDiodeQEUV;
      fluxRelVar             +=  gkPINDiodeQEUVErr* gkPINDiodeQEUVErr
                             / ( gkPINDiodeQEUV   * gkPINDiodeQEUV    );
      break;
    case kCT1:
    default:
      fFluxOutsidePlexiglass /=  gkPINDiodeQECT1;
      fluxRelVar             +=  gkPINDiodeQECT1Err* gkPINDiodeQECT1Err
                             / ( gkPINDiodeQECT1   * gkPINDiodeQECT1    );
      break;
    }


  *fLog << inf << endl;
  *fLog << inf << " Mean Photon flux [ph/mm^2] outside Plexiglass: " 
        << fFluxOutsidePlexiglass << endl;

  if (fFluxOutsidePlexiglass < 0.)
      return kFALSE;

  if (fluxRelVar < 0.)
      return kFALSE;

  SetFluxOutsidePlexiglassAvailable();  

  //
  // Finish calculation of errors -> convert from relative variance to absolute variance
  //
  fFluxOutsidePlexiglassVar *= fluxRelVar * fFluxOutsidePlexiglass * fFluxOutsidePlexiglass;

  *fLog << inf << " Error on Photon flux [ph/mm^2] outside Plexiglass: " 
        << GetFluxOutsidePlexiglassErr() << endl;
  *fLog << inf << endl;

  return kTRUE;
}
