#ifndef MARS_MCalibrationChargeCalc
#define MARS_MCalibrationChargeCalc

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// MCalibrationChargeCalc                                                   //
//                                                                         //
// Integrates the time slices of the all pixels of a calibration event     //
// and substract the pedestal value                                        //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#ifndef MARS_MTask
#include "MTask.h"
#endif

#ifndef MARS_MBadPixelsPix
#include "MBadPixelsPix.h"
#endif

#ifndef MARS_MCalibrationCam
#include "MCalibrationCam.h"
#endif

class MRawEvtData;
class MRawRunHeader;
class MPedestalCam;
class MPedestalPix;
class MCalibrationChargePINDiode;
class MCalibrationChargeBlindPix;
class MCalibrationChargePix;
class MCalibrationChargeCam;
class MCalibrationQECam;
class MGeomCam;
class MExtractedSignalCam;
class MExtractedSignalBlindPixel;
class MExtractedSignalPINDiode;
class MBadPixelsCam;
class MBadPixelsPix;
class MTime;
class MCalibrationChargeCalc : public MTask
{
private:

  static const Float_t fgChargeLimit;          //! Default for fChargeLimit       (now set to: 3.)
  static const Float_t fgChargeErrLimit;       //! Default for fChargeErrLimit    (now set to: 0.)
  static const Float_t fgChargeRelErrLimit;    //! Default for fChargeRelErrLimit (now set to: 1.)
  static const Float_t fgLambdaCheckLimit;     //! Default for fLambdaCheckLimit  (now set to: 0.2)
  static const Float_t fgLambdaErrLimit;       //! Default for fLabmdaErrLimit    (now set to: 0.2)
  static const Float_t fgPheErrLimit;          //! Default for fPheErrLimit       (now set to: 5.)
  static const Float_t fgTimeLowerLimit;       //! Default for fTimeLowerLimit    (now set to: 1.) 
  static const Float_t fgTimeUpperLimit;       //! Default for fTimeUpperLimit    (now set to: 2.)

  // Variables
  Float_t fChargeLimit;                        // Limit acceptance mean charge  (in units of PedRMS)
  Float_t fChargeErrLimit;                     // Limit acceptance charge error (in abs. numbers)
  Float_t fChargeRelErrLimit;                  // Limit acceptance rel. error mean (in abs. numbers)
  Int_t   fFlags;                              // Bit-field for the flags
  Float_t fLambdaCheckLimit;                   // Limit rel. diff. lambda and lambdacheck in Blind Pixel
  Float_t fLambdaErrLimit;                     // Limit acceptance lambda error in Blind Pixel
  Float_t fNumHiGainSamples;                   // Number High-Gain FADC slices used by extractor
  Float_t fNumLoGainSamples;                   // Number Low -Gain FADC slices used by extractor
  Float_t fPheErrLimit;                        // Limit acceptance nr. phe's w.r.t. area idx mean (in sigmas)
  Float_t fSqrtHiGainSamples;                  // Square root nr. High-Gain FADC slices used by extractor
  Float_t fSqrtLoGainSamples;                  // Square root nr. Low -Gain FADC slices used by extractor 
  Float_t fTimeLowerLimit;                     // Limit dist. to first signal slice (in units of FADC slices) 
  Float_t fTimeUpperLimit;                     // Limit dist. to last signal slice  (in units of FADC slices) 
  MCalibrationCam::PulserColor_t fPulserColor; // Calibration LEDs colour 

  // Pointers
  MBadPixelsCam              *fBadPixels;      //  Bad Pixels 
  MCalibrationChargeCam      *fCam;            //  Calibrated Charges of all pixels 
  MCalibrationChargeBlindPix *fBlindPixel;     //  Calibrated Charges of the Blind Pixel 
  MCalibrationChargePINDiode *fPINDiode;       //  Calibrated Charges of the PIN Diode
  MCalibrationQECam          *fQECam;          //  Calibrated Quantum Efficiencies of all pixels 
  MExtractedSignalCam        *fSignals;        //! Extracted signal of all pixels 
  MExtractedSignalBlindPixel *fSigBlind;       //! Extracted signal of Blind Pixel
  MExtractedSignalPINDiode   *fSigPIN;         //! Extracted signal of PIN Diode 
  MGeomCam                   *fGeom;           //! Camera geometry
  MPedestalCam               *fPedestals;      //! Pedestals all pixels (calculated previously from ped.file)
  MTime                      *fEvtTime;        //! Time of the event

  // enums
  enum  { kUseQualityChecks, kHiLoGainCalibration };

  // functions
  void   FinalizePedestals      ( const MPedestalPix &ped, MCalibrationChargePix &cal,
                                  Float_t &avped, Float_t &avrms                      );
  void   FinalizeAvPedestals    ( MCalibrationChargePix &cal, 
	                          Float_t avped, Float_t avrms, Int_t avnum);
  Bool_t FinalizeCharges        ( MCalibrationChargePix &cal, MBadPixelsPix &bad      );
  Bool_t FinalizePINDiode       ();
  Bool_t FinalizeBlindPixel     ();
  Bool_t FinalizeFFactorMethod  ();
  void   FinalizeBadPixels      ();
  void   FinalizeFFactorQECam   ();  
  void   FinalizeBlindPixelQECam();
  void   FinalizePINDiodeQECam  ();
  
  void   PrintUnsuitable  ( MBadPixelsPix::UnsuitableType_t typ  , const char *text) const;   
  void   PrintUncalibrated( MBadPixelsPix::UncalibratedType_t typ, const char *text) const;

  Int_t  PreProcess (MParList *pList);
  Bool_t ReInit     (MParList *pList); 
  Int_t  Process    ();
  Int_t  PostProcess();

public:

  MCalibrationChargeCalc(const char *name=NULL, const char *title=NULL);

  void Clear(const Option_t *o="");
  
  void SetChargeLimit       ( const Float_t f=fgChargeLimit            ) { fChargeLimit       = f;   }
  void SetChargeErrLimit    ( const Float_t f=fgChargeErrLimit         ) { fChargeErrLimit    = f;   }
  void SetChargeRelErrLimit ( const Float_t f=fgChargeRelErrLimit      ) { fChargeRelErrLimit = f;   }
  void SetLambdaErrLimit    ( const Float_t f=fgLambdaErrLimit         ) { fLambdaErrLimit    = f;   }
  void SetLambdaCheckLimit  ( const Float_t f=fgLambdaCheckLimit       ) { fLambdaCheckLimit  = f;   }
  void SetPheErrLimit       ( const Float_t f=fgPheErrLimit            ) { fPheErrLimit       = f;   }  
  void SetPulserColor       ( const MCalibrationCam::PulserColor_t col ) { fPulserColor       = col; }
  void SetTimeLowerLimit    ( const Float_t f=fgTimeLowerLimit         ) { fTimeLowerLimit    = f;   }
  void SetTimeUpperLimit    ( const Float_t f=fgTimeUpperLimit         ) { fTimeUpperLimit    = f;   }

  void SkipHiLoGainCalibration ( const Bool_t b=kTRUE )
      { b ? CLRBIT(fFlags, kHiLoGainCalibration) : SETBIT(fFlags, kHiLoGainCalibration); }
  void SkipQualityChecks       ( const Bool_t b=kTRUE)
      { b ? CLRBIT(fFlags, kUseQualityChecks)    : SETBIT(fFlags, kUseQualityChecks);    }

  ClassDef(MCalibrationChargeCalc, 1)   // Task calculating Calibration Containers and Quantum Efficiencies
};

#endif
