/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug, 04/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
//   MExtractor
//
//   Base class for the signal extractors, used the functions 
//   FindSignalHiGain() and FindSignalLoGain() to extract the signal and 
//   substract the pedestal value    
//
//   The following variables have to be set by the derived class and 
//   do not have defaults:
//   - fNumHiGainSamples
//   - fNumLoGainSamples
//   - fSqrtHiGainSamples
//   - fSqrtLoGainSamples
//
// Input Containers:
//   MRawEvtData
//   MPedestalCam
//
// Output Containers:
//   MExtractedSignalCam
//
//////////////////////////////////////////////////////////////////////////////
#include "MExtractor.h"

#include <fstream>

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"

#include "MRawEvtData.h"
#include "MRawEvtPixelIter.h"

#include "MPedestalCam.h"
#include "MPedestalPix.h"

#include "MExtractedSignalCam.h"
#include "MExtractedSignalPix.h"

ClassImp(MExtractor);

using namespace std;

const Byte_t MExtractor::fgSaturationLimit = 254;
// --------------------------------------------------------------------------
//
// Default constructor. 
//
MExtractor::MExtractor(const char *name, const char *title)
    : fNumHiGainSamples(0.), fNumLoGainSamples(0.), fSaturationLimit(fgSaturationLimit)
{

    fName  = name  ? name  : "MExtractor";
    fTitle = title ? title : "Base class for signal extractors";

    AddToBranchList("MRawEvtData.*");

    SetRange();
}

void MExtractor::SetRange(Byte_t hifirst, Byte_t hilast, Byte_t lofirst, Byte_t lolast)
{

    fHiGainFirst = hifirst;
    fLoGainFirst = lofirst;

    fHiGainLast  = hilast;
    fLoGainLast  = lolast;

}

// --------------------------------------------------------------------------
//
// The PreProcess searches for the following input containers:
//  - MRawEvtData
//  - MPedestalCam
//
// The following output containers are also searched and created if
// they were not found:
//
//  - MExtractedSignalCam
//
Int_t MExtractor::PreProcess(MParList *pList)
{
    fRawEvt = (MRawEvtData*)pList->FindObject(AddSerialNumber("MRawEvtData"));
    if (!fRawEvt)
    {
        *fLog << err << AddSerialNumber("MRawEvtData") << " not found... aborting." << endl;
        return kFALSE;
    }


    fSignals = (MExtractedSignalCam*)pList->FindCreateObj(AddSerialNumber("MExtractedSignalCam"));
    if (!fSignals)
        return kFALSE;

    fSignals->SetUsedFADCSlices(fHiGainFirst, fHiGainLast, fNumHiGainSamples,
                                fLoGainFirst, fLoGainLast, fNumLoGainSamples);

    fPedestals = (MPedestalCam*)pList->FindObject(AddSerialNumber("MPedestalCam"));

    if (!fPedestals)
    {
        *fLog << err << AddSerialNumber("MPedestalCam") << " not found... aborting" << endl;
        return kFALSE;
    }

    return kTRUE;
}

void MExtractor::FindSignalHiGain(Byte_t *ptr, Int_t last, Int_t &sum, Byte_t &sat) const
{
  return;
}

void MExtractor::FindSignalLoGain(Byte_t *ptr, Int_t last, Int_t &sum, Byte_t &sat) const
{
  return;
}

// --------------------------------------------------------------------------
//
// Calculate the integral of the FADC time slices and store them as a new
// pixel in the MExtractedSignalCam container.
//
Int_t MExtractor::Process()
{

  MRawEvtPixelIter pixel(fRawEvt);
  fSignals->Clear();

  UInt_t  sat=0;

  while (pixel.Next())
    {
      Int_t sumhi;
      Byte_t sathi;

      FindSignalHiGain(pixel.GetHiGainSamples()+fHiGainFirst, fHiGainLast, sumhi, sathi);

      Int_t  sumlo = 0;
      Byte_t satlo = 0;

      if (pixel.HasLoGain())
        {
          FindSignalLoGain(pixel.GetLoGainSamples()+fLoGainFirst, fLoGainLast, sumlo, satlo);
          
          if (satlo)
            sat++;
        }
      
      const Int_t pixid = pixel.GetPixelId();
      
      const MPedestalPix  &ped = (*fPedestals)[pixid]; 
      MExtractedSignalPix &pix = (*fSignals)[pixid];
      
      const Float_t pedes  = ped.GetPedestal();
      const Float_t pedrms = ped.GetPedestalRms();
      
      pix.SetExtractedSignal(sumhi - pedes*fNumHiGainSamples, pedrms*fSqrtHiGainSamples,
                             sumlo - pedes*fNumLoGainSamples, pedrms*fSqrtLoGainSamples);
      
      pix.SetGainSaturation(sathi, sathi, satlo);
      
    } /* while (pixel.Next()) */

    fSignals->SetReadyToSave();

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Implementation of SavePrimitive. Used to write the call to a constructor
// to a macro. In the original root implementation it is used to write
// gui elements to a macro-file.
//
void MExtractor::StreamPrimitive(ofstream &out) const
{
}

