
/********************************************************************\

  Name:         HVStatus.cc

  Created by:   Sebastian Commichau, November 2008
                commichau@phys.ethz.ch

  Contents:     Store HV supply status information

\********************************************************************/

#include "HVStatus.h"

static char* state_str[]    = {"active", "stopped", "n.a."}; 
static char* isok_str[]     = {"error", "ok"};
static char* cc_state_str[] = {"disconnected", "connected"};

void InitStatus(Status* status, HVConfig* config) {

  status->state     = active;
  status->cc_state  = disconnected;
  status->Exit      = FALSE;
  status->Stop      = FALSE;
  status->Log       = TRUE;
  status->Verbose   = FALSE;
  status->CCCommand = FALSE;

  status->Socket    = -1;
  sprintf(status->CCClient,"%s", config->fCCClient);
  status->CCPort    = config->fCCPort;


  status->fUSBDevice = new char*[MAX_NUM_HVBOARDS];
  
  for (int i=0; i<MAX_NUM_HVBOARDS; i++) {
    status->fUSBDevice[i]      = new char[FILENAME_MAX_SIZE];
    status->USBDeviceNumber[i] = USB_MAX_DEVICE_NUMBER+1;
  }

  status->USBMinDeviceNumber = USB_MAX_DEVICE_NUMBER;
  status->USBMaxDeviceNumber = USB_MIN_DEVICE_NUMBER;

  for (int i=0;i<MAX_NUM_HVBOARDS;i++) {
    status->IsUpdated[i] = FALSE;
    status->isok[i]      = TRUE;
    status->MR[i]        = FALSE;
    status->WC[0][i]     = 0;
    status->WC[1][i]     = 1;

    for (int j=0;j<MAX_NUM_CHAINS;j++) {
      status->VRef[i][j] = 0;
      status->OC[i][j]   = FALSE;

      for (int k=0;k<MAX_NUM_CHANNELS;k++){
	status->HV[i][j][k] = 0;
	status->HVV[i][j][k] = 0.0;
      }
    }
  }

  status->NumHVBoards = 0;
  status->FirstBoard  = 0;
  status->LastBoard   = -1;

  status->FirstChain  = 0;
  status->LastChain   = 3;
  

  if (config->fTimeOut >= (float)MIN_TIMEOUT && config->fTimeOut <= (float)MAX_TIMEOUT)
    status->fTimeOut  = config->fTimeOut;
  else
    status->fTimeOut  = 1.;

  if (config->fStatusRefreshRate >= (float)MIN_RATE && config->fStatusRefreshRate <= (float)MAX_RATE) 
    status->fStatusRefreshRate = config->fStatusRefreshRate;
  else
    status->fStatusRefreshRate = 1.;
} 


void ReInitStatus(Status* status) {

  for (int i=0;i<MAX_NUM_HVBOARDS;i++) {
    status->IsUpdated[i] = FALSE;
    for (int j=0;j<MAX_NUM_CHAINS;j++) {
      status->VRef[i][j] = 0;
      for (int k=0;k<MAX_NUM_CHANNELS;k++)
	status->HV[i][j][k] = 0;
    }
  }
  
} 


void ReInitStatusOneBoard(Status* status, int board) {

  status->IsUpdated[board] = FALSE;
  for (int j=0;j<MAX_NUM_CHAINS;j++) {
    status->VRef[board][j] = 0;
    for (int k=0;k<MAX_NUM_CHANNELS;k++)
      status->HV[board][j][k] = 0;
  }
    
} 


void PrintStatus(Status* status, HVConfig* config, FILE* fptr) {

  fprintf(fptr,"\n");
  fprintf(fptr,"********************************************* STATUS ********************************************\n\n");
  fprintf(fptr," Status monitor: %s\n",state_str[status->state]);
  fprintf(fptr," Logging: %s\n", ((status->Log) ? "on" : "off"));
  fprintf(fptr," Verbose: %s\n", ((status->Verbose) ? "on" : "off"));
  fprintf(fptr," Timeout [s]: %.2f\n",status->fTimeOut);
  fprintf(fptr," Status refresh rate [Hz]: %.2f\n",status->fStatusRefreshRate);
  fprintf(fptr," CC state: %s\n",                     cc_state_str[status->cc_state]);
  fprintf(fptr," CC client: %s\n",                    status->CCClient);
  fprintf(fptr," CC port: %d\n",                      status->CCPort);
  fprintf(fptr," Total number of HV boards: %d\n",   status->NumHVBoards);
  if (!status->NumHVBoards)
    fprintf(fptr," Active HV boards: 0\n\n");
  else if (status->NumHVBoards == ((status->LastBoard - status->FirstBoard) + 1))
    fprintf(fptr," Active HV boards: all\n\n");
  else
    fprintf(fptr," Active HV boards: %d\n\n",            (status->LastBoard - status->FirstBoard) + 1);

  for (int i=status->FirstBoard;i<=status->LastBoard;i++) {
    fprintf(fptr," BOARD %d (%s):\n",status->USBDeviceNumber[i],status->fUSBDevice[i]);
    fprintf(fptr,"  Wrap counter: %s (%d)\n",isok_str[status->isok[i]], status->WC[1][i]);
    fprintf(fptr,"  Manual reset: %s\n\n",((status->MR[i]) ? "yes" : "no"));

    for (int j=status->FirstChain;j<=status->LastChain;j++) {
      fprintf(fptr,"  CHAIN %d:\n",j);
      fprintf(fptr,"   Over-current: %s\n",((status->OC[i][j]) ? "yes" : "no"));
      fprintf(fptr,"   Reference voltage: %d\n",status->VRef[i][j]);
      fprintf(fptr,"   High voltage:\n");
      for (int k=0;k<4;k++) {
	fprintf(fptr,"    Channels %.2d-%.2d: ",k*8,k*8+7);
	for (int l=0;l<8;l++) 
	  fprintf(fptr," %5d ",status->HV[i][j][k*8+l]);
	fprintf(fptr,"\n");
      }
	fprintf(fptr,"\n");
    }
  }


  fprintf(fptr,"*************************************************************************************************\n\n");

}


void sPrintStatus(Status* status, char* str, int i) {

  sprintf(str,"status board %d (%s): MR %s OC0 %s OC1 %s OC2 %s OC3 %s WC %s (%d)\n",status->USBDeviceNumber[i],status->fUSBDevice[i],
	  ((status->MR[i]) ? "yes" : "no"),
	  ((status->OC[i][0]) ? "yes" : "no"),
	  ((status->OC[i][1]) ? "yes" : "no"),
	  ((status->OC[i][2]) ? "yes" : "no"),
	  ((status->OC[i][3]) ? "yes" : "no"),
	  isok_str[status->isok[i]],
	  status->WC[1][i]);
 
  if (status->Socket != -1) // Print status string to socket if open
    write(status->Socket,str,strlen(str)+1);

}


char* GetStateStr(Status* status) {

  return state_str[status->state];

}
