#ifndef MARS_MTime
#define MARS_MTime

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// MTime                                                                   //
//                                                                         //
// A generalized MARS time stamp                                           //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#ifndef MARS_MParContainer
#include "MParContainer.h"
#endif

#ifndef _CPP_IOSFWD
#include <iosfwd>
#endif

#ifndef ROOT_TTime
#include <TTime.h>
#endif

struct timeval;

class MTime : public MParContainer
{
public:
    static const UInt_t kHour; // [ms] one hour
    static const UInt_t kDay;  // [ms] one day

private:
    UInt_t fMjd;     // [d]  Day in the century        (Day of sun rise)
    TTime  fTime;    // [ms] Time of Day               (-11h<=x<13h)
    UInt_t fNanoSec; // [ns] NanoSec part of TimeOfDay (<1000000)

    ULong_t GetTime24() const
    {
        return (Long_t)fTime<0 ? (Long_t)fTime+kDay : (ULong_t)fTime;
    }
    void Init(const char *name, const char *title)
    {
        fName  = name  ? name  : "MTime";
        fTitle = title ? title : "Generalized time stamp";
    }

public:
    MTime(const char *name=NULL, const char *title=NULL)
    {
        Init(name, title);
        Clear();
    }
    MTime(const struct timeval &tm)
    {
        Init(NULL, NULL);
        Set(tm);
    }
    MTime(Double_t mjd);
    MTime(const MTime& t) : fMjd(t.fMjd), fTime(t.fTime), fNanoSec(t.fNanoSec)
    {
        Init(NULL, NULL);
    }

    void operator=(const MTime &t)
    {
        fMjd     = t.fMjd;
        fTime    = t.fTime;
        fNanoSec = t.fNanoSec;
    }

    void Clear(const Option_t *o="") { fMjd=0; fTime=0; fNanoSec=0; }

    void Print(Option_t *t=NULL) const;

    void Now();

    Bool_t   Set(UInt_t mjd, ULong_t ms, UInt_t ns=0);
    Bool_t   Set(UShort_t y, Byte_t m, Byte_t d, Byte_t h=13, Byte_t min=0, Byte_t s=0, UShort_t ms=0, UInt_t ns=0);
    void     Set(const struct timeval &tv);
    Bool_t   SetString(const char *str);
    Bool_t   SetSqlDateTime(const char *str);
    Bool_t   SetSqlTimeStamp(const char *str);
    void     SetCT1Time(UInt_t mjd, UInt_t t1, UInt_t t0);
    Bool_t   UpdMagicTime(Byte_t h, Byte_t m, Byte_t s, UInt_t ns);
    void     SetMjd(Double_t m);
    Double_t GetMjd() const;
    Double_t GetGmst() const;
    TString  GetString() const;
    TString  GetStringFmt(const char *fmt=0) const;
    TString  GetSqlDateTime() const;
    TString  GetSqlTimeStamp() const;
    TString  GetFileName() const;
    void     GetDate(UShort_t &y, Byte_t &m, Byte_t &d) const;
    TTime    GetRootTime() const;
    Double_t GetAxisTime() const;
    Long_t   GetTime() const { return (Long_t)fTime; } // [ms] Time of Day returned in the range [-11h, 13h)
    void     GetTime(Byte_t &h, Byte_t &m, Byte_t &s, UShort_t &ms) const;
    void     GetTime(Byte_t &h, Byte_t &m, Byte_t &s) const
    {
        UShort_t ms;
        GetTime(h, m, s, ms);
    }

    UInt_t Year() const  { UShort_t y; Byte_t m, d; GetDate(y,m,d); return y; }
    UInt_t Month() const { UShort_t y; Byte_t m, d; GetDate(y,m,d); return m; }
    UInt_t Day() const   { UShort_t y; Byte_t m, d; GetDate(y,m,d); return d; }
    UInt_t Hour() const  { Byte_t h, m, s; GetTime(h,m,s); return h; }
    UInt_t Min() const   { Byte_t h, m, s; GetTime(h,m,s); return m; }
    UInt_t Sec() const   { Byte_t h, m, s; GetTime(h,m,s); return s; }

    istream &ReadBinary(istream &fin);

    operator double() const   //[s]
    {
        return ((Double_t)fMjd*kDay+(Long_t)fTime+fNanoSec/1e6)/1000;
    }
    double operator()() const //[s]
    {
        return operator double();
    }

    bool operator<(const MTime &t) const
    {
        if (fMjd<t.fMjd)
            return true;
        if (fMjd==t.fMjd && fTime<t.fTime)
            return true;
        if (fMjd==t.fMjd && fTime==t.fTime && fNanoSec<t.fNanoSec)
            return true;
        return false;
    }
    bool operator>(const MTime &t) const
    {
        if (fMjd>t.fMjd)
            return true;
        if (fMjd==t.fMjd && fTime>t.fTime)
            return true;
        if (fMjd==t.fMjd && fTime==t.fTime && fNanoSec>t.fNanoSec)
            return true;
        return false;
    }

    bool operator<=(const MTime &t) const
    {
        return !operator>(t);
    }

    bool operator>=(const MTime &t) const
    {
        return !operator<(t);
    }

    bool operator==(const MTime &t) const
    {
        return fNanoSec==t.fNanoSec && fTime==t.fTime && fMjd==t.fMjd;
    }

    bool operator!=(const MTime &t) const
    {
        return fNanoSec!=t.fNanoSec || fTime!=t.fTime || fMjd!=t.fMjd;
    }

    bool operator!() const
    {
        return fNanoSec==0 && (ULong_t)fTime==0 && fMjd==0;
    }

    ClassDef(MTime, 3)	//A generalized MARS time stamp
};

inline ostream &operator<<(ostream &out, const MTime &t)
{
    out << t.GetString();
    return out;
}

inline istream &operator>>(istream &in, MTime &t)
{
    TString date, time;
    date.ReadToDelim(in, ' ');
    time.ReadToDelim(in, ' ');
    t.SetString(Form("%s %s", date.Data(), time.Data()));
    return in;
}

#endif
