/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug, 04/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
//   MExtractTime
//
//   Base class for the signal extractors, used the functions 
//   FindTimeHiGain() and FindTimeLoGain() to extract the signal and 
//   substract the pedestal value    
//
//   The following variables have to be set by the derived class and 
//   do not have defaults:
//   - fNumHiGainSamples
//   - fNumLoGainSamples
//   - fSqrtHiGainSamples
//   - fSqrtLoGainSamples
//
// Input Containers:
//   MRawEvtData
//   MRawRunHeader
//   MPedestalCam
//
// Output Containers:
//   MArrivalTimeCam
//
//////////////////////////////////////////////////////////////////////////////
#include "MExtractTime.h"

#include <fstream>

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"

#include "MRawEvtData.h"
#include "MRawEvtPixelIter.h"
#include "MRawRunHeader.h"

#include "MPedestalCam.h"
#include "MPedestalPix.h"

#include "MArrivalTimeCam.h"
#include "MArrivalTimePix.h"

ClassImp(MExtractTime);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor. 
//
// Set: 
// - all pointers to NULL
// - all variables to 0
// - fSaturationLimit to fgSaturationLimit
//
// Call:
// -  AddToBranchList("MRawEvtData.*")
//
MExtractTime::MExtractTime(const char *name, const char *title)
    : fArrTime(NULL)
{

    fName  = name  ? name  : "MExtractTime";
    fTitle = title ? title : "Base class for signal extractors";

}



// --------------------------------------------------------------------------
//
// The PreProcess searches for the following input containers:
//  - MRawEvtData
//  - MRawRunHeader
//  - MPedestalCam
//
// The following output containers are also searched and created if
// they were not found:
//
//  - MArrivalTimeCam
//
Int_t MExtractTime::PreProcess(MParList *pList)
{

  fRawEvt = (MRawEvtData*)pList->FindObject(AddSerialNumber("MRawEvtData"));
  if (!fRawEvt)
    {
        *fLog << err << AddSerialNumber("MRawEvtData") << " not found... aborting." << endl;
        return kFALSE;
    }
  
  fRunHeader = (MRawRunHeader*)pList->FindObject(AddSerialNumber("MRawRunHeader"));
  if (!fRunHeader)
    {
      *fLog << err << AddSerialNumber("MRawRunHeader") << " not found... aborting." << endl;
      return kFALSE;
    }
  
  
  fArrTime = (MArrivalTimeCam*)pList->FindCreateObj(AddSerialNumber("MArrivalTimeCam"));
  if (!fArrTime)
    return kFALSE;
  

  fPedestals = (MPedestalCam*)pList->FindObject(AddSerialNumber("MPedestalCam"));
  if (!fPedestals)
    {
      *fLog << err << AddSerialNumber("MPedestalCam") << " not found... aborting" << endl;
      return kFALSE;
    }
  
  return kTRUE;
}

// --------------------------------------------------------------------------
//
// The ReInit searches for:
// -  MRawRunHeader::GetNumSamplesHiGain()
// -  MRawRunHeader::GetNumSamplesLoGain()
//
// In case that the variable fLoGainLast is smaller than 
// the even part of the number of samples obtained from the run header, a
// warning is given an the range is set back accordingly. A call to:  
// - SetRange(fHiGainFirst, fHiGainLast, fLoGainFirst, fLoGainLast-diff) 
// is performed in that case. The variable diff means here the difference 
// between the requested range (fLoGainLast) and the available one. Note that 
// the functions SetRange() are mostly overloaded and perform more checks, 
// modifying the ranges again, if necessary.
//
// In case that the variable fHiGainLast is smaller than the available range 
// obtained from the run header, a warning is given that a part of the low-gain 
// samples are used for the extraction of the high-gain signal. 
//
// Call: 
// - MArrivalTimeCam::SetUsedFADCSlices(fHiGainFirst, fHiGainLast, fNumHiGainSamples,
//                                          fLoGainFirst, fLoGainLast, fNumLoGainSamples);
//
Bool_t MExtractTime::ReInit(MParList *pList)
{
  
  Int_t lastdesired   = (Int_t)(fLoGainLast);
  Int_t lastavailable = (Int_t)fRunHeader->GetNumSamplesLoGain()-1;
  
  if (lastdesired > lastavailable)
    {
      const Int_t diff = lastdesired - lastavailable;
      *fLog << endl; 
      *fLog << warn << GetDescriptor()
            << Form("%s%2i%s%2i%s%2i%s",": Selected Lo Gain FADC Window [",
                    (int)fLoGainFirst,",",lastdesired,
                    "] ranges out of the available limits: [0,",lastavailable,"].") << endl;
      *fLog << GetDescriptor() << ": Will reduce the upper edge to " << (int)(fLoGainLast - diff) << endl;
      SetRange(fHiGainFirst, fHiGainLast, fLoGainFirst, fLoGainLast-diff);
    }

  lastdesired   = (Int_t)fHiGainLast;
  lastavailable = (Int_t)fRunHeader->GetNumSamplesHiGain()-1;
  
  if (lastdesired > lastavailable)
    {
      const Int_t diff = lastdesired - lastavailable;
      *fLog << endl;
      *fLog << warn << GetDescriptor()
            << Form("%s%2i%s%2i%s%2i%s",": Selected Hi Gain FADC Window [",
                    (int)fHiGainFirst,",",lastdesired,
                    "] ranges out of the available limits: [0,",lastavailable,"].") << endl;
      *fLog << warn << GetDescriptor() 
            << Form("%s%2i%s",": Will use ",diff," samples from the Low-Gain for the High-Gain extraction")
            << endl;
      fHiGainLast -= diff;
      fHiLoLast    = diff;
    }


  fArrTime->SetUsedFADCSlices(fHiGainFirst, fHiGainLast+fHiLoLast, fLoGainFirst, fLoGainLast);

  return kTRUE;
}



void MExtractTime::FindTimeHiGain(Byte_t *firstused, Float_t &time, Float_t &dtime, 
                                  Byte_t &sat, const MPedestalPix &ped) const
{
  return;
}

void MExtractTime::FindTimeLoGain(Byte_t *firstused, Float_t &time, Float_t &dtime, 
                                  Byte_t &sat, const MPedestalPix &ped) const
{
  return;
}

// --------------------------------------------------------------------------
//
// Calculate the integral of the FADC time slices and store them as a new
// pixel in the MArrivalTimeCam container.
//
Int_t MExtractTime::Process()
{


  MRawEvtPixelIter pixel(fRawEvt);
  fArrTime->Clear();

  while (pixel.Next())
    {
      //
      // Find signal in hi- and lo-gain
      //
      Float_t timehi=0., deltatimehi=0.;
      Byte_t sathi=0;

      const Int_t pixid = pixel.GetPixelId();
      const MPedestalPix  &ped = (*fPedestals)[pixid];
      MArrivalTimePix &pix = (*fArrTime)[pixid];

      FindTimeHiGain(pixel.GetHiGainSamples()+fHiGainFirst, timehi, deltatimehi, sathi, ped);

      Float_t timelo=0., deltatimelo=0.;
      Byte_t satlo=0;

      if (pixel.HasLoGain())
        FindTimeLoGain(pixel.GetLoGainSamples()+fLoGainFirst, timelo, deltatimelo, satlo, ped);
      
      pix.SetArrivalTime(timehi, deltatimehi, timelo, deltatimelo);
      pix.SetGainSaturation(sathi, sathi, satlo);
 
    } /* while (pixel.Next()) */

    fArrTime->SetReadyToSave();

    return kTRUE;
}

