/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug, 11/2003 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */
/////////////////////////////////////////////////////////////////////////////
//
//  readCalibration.C
//
//  Needs as arguments the run number of a calibration file ("*_C_*.root") and 
//  the run number of the corresponding pedestal file ("*_P_*.root"). 
//
//  The TString inpath has to be set correctly.
//
//
/////////////////////////////////////////////////////////////////////////////
static const TString inpath = ".";
//
// Tell if you want to calibrate times:
//
static const  Bool_t useTimes = kTRUE;
//
// the default pedestal run for the calibration
//
static const Int_t   pedrun = 26210;
//
// the default start calibration run 
//
static const Int_t   calrun = 26209;
//
// A switch to output debugging information about Objects use
//
static const Bool_t debug = kFALSE;
//
void readCalibration(const Int_t prun=pedrun, const Int_t crun=calrun)
{

  const TString pedname  = Form("%s/%i-F0.root", (const char*)inpath,prun);
  const TString calname  = Form("%s/%i-F1.root", (const char*)inpath,crun);
  const TString testname = Form("%s/%i-Test.root", (const char*)inpath,crun);

  if (gSystem->AccessPathName(pedname, kFileExists))
    {
      cout << "Input file " << pedname << " doesn't exist." << endl;
      return;
    }

  if (gSystem->AccessPathName(calname, kFileExists))
    {
      cout << "Input file " << calname << " doesn't exist." << endl;
      return;
    }

  if (gSystem->AccessPathName(testname, kFileExists))
    {
      cout << "Input file " << testname << " doesn't exist." << endl;
      return;
    }

  MStatusDisplay *display = new MStatusDisplay;
  MPedestalCam           pedcam;
  MCalibrationChargeCam  chargecam;  
  MCalibrationQECam      qecam;
  MCalibrationRelTimeCam reltimecam;
  MBadPixelsCam          badcam;
  MHCalibrationTestCam     testcam;
  MHCalibrationTestTimeCam testtime;

  cout << "Reading from file: " << pedname << endl;

  TFile pedfile(pedname, "READ");
  if (pedcam.Read()<=0)
    {
      cout << "Unable to read MPedestalCam from " << pedname << endl;
      return;
    }
  
  if (display->GetCanvas("Pedestals"))
    display->Read();

  cout << "Reading from file: " << calname << endl;

  TFile calfile(calname, "READ");
  if (chargecam.Read()<=0)
    {
      cout << "Unable to read MCalibrationChargeCam from " << calname << endl;
      return;
    }

  if (qecam.Read()<=0)
    {
      cout << "Unable to read MCalibrationQECam from " << calname << endl;
      return;
    }

    if (reltimecam.Read()<=0)
    {
      cout << "Unable to read MCalibrationRelTimeCam from " << calname << endl;
      return;
    }
    
    if (calfile.FindKey("MBadPixelsCam"))
      {
        MBadPixelsCam bad;
        if (bad.Read()<=0)
          {
            cout << "Unable to read MBadPixelsCam from " << calname << endl;
            return;
          }
        badcam.Merge(bad);
      }

    display->Read();
    
  cout << "Reading from file: " << testname << endl;

  TFile testfile(testname, "READ");
  if (testcam.Read()<=0)
    {
      cout << "Unable to read MHCalibrationTestCam from " << testname << endl;
      return;
    }
  if (testtime.Read()<=0)
    {
      cout << "Unable to read MHCalibrationTestTimeCam from " << testname << endl;
      return;
    }
  
    display->Read();


  //
  // List of interesting plots:
  // 
/*
  testcam.GetAverageHiGainArea(0).DrawClone();
  testcam.GetAverageLoGainArea(0).DrawClone();
  testcam.GetAverageHiGainArea(1).DrawClone();
  testcam.GetAverageLoGainArea(1).DrawClone();

  testcam.GetAverageHiGainSector(1).DrawClone();
  testcam.GetAverageLoGainSector(1).DrawClone();
  testcam.GetAverageHiGainSector(2).DrawClone();
  testcam.GetAverageLoGainSector(2).DrawClone();
  testcam.GetAverageHiGainSector(3).DrawClone();
  testcam.GetAverageLoGainSector(3).DrawClone();
  testcam.GetAverageHiGainSector(4).DrawClone();
  testcam.GetAverageLoGainSector(4).DrawClone();
  testcam.GetAverageHiGainSector(5).DrawClone();
  testcam.GetAverageLoGainSector(5).DrawClone();
  testcam.GetAverageHiGainSector(6).DrawClone();
  testcam.GetAverageLoGainSector(6).DrawClone();
*/
}

