/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 1/2004 <mailto:tbretz@astro.uni-wuerzburg.de>
!              Markus Gaug, 02/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */
/////////////////////////////////////////////////////////////////////////////
//
//  MJCalibration
//
//  Do one calibration loop over serious of runs with the same pulser 
//  colour and the same intensity. The following containers (rectangular) and 
//  tasks (ellipses) are called to produce an MCalibrationChargeCam and to 
//  update the MCalibrationQECam: (MCalibrate is not called from this class)
//
//Begin_Html
/*
<img src="images/CalibClasses.gif">
*/
//End_Html
//
// Different signal extractors can be set with the command SetExtractor()
// Only extractors deriving from MExtractor can be set, default is MExtractSlidingWindow
//
// Different arrival time extractors can be set with the command SetTimeExtractor()
// Only extractors deriving from MExtractTime can be set, default is MExtractTimeSpline
//
// At the end of the eventloop, plots and results are displayed, depending on 
// the flags set (see DisplayResult()) 
// 
// If the flag SetFullDisplay() is set, all MHCameras will be displayed. 
// if the flag SetDataCheckDisplay() is set, only the most important ones are displayed 
// Otherwise, (default: SetNormalDisplay()), a good selection of plots is given
//
// If the flag SetDataCheck() is set, the calibration is used as in the data check at 
// La Palma, which mean especially running on raw data files.
// 
// The absolute light calibration devices Blind Pixel and PIN Diode can be switched on
// and off with the commands:
//
// - SetUseBlindPixel(Bool_t )
// - SetUsePINDiode(Bool_t )
//
// See also: MHCalibrationChargePix, MHCalibrationChargeCam, MHGausEvents
//           MHCalibrationChargeBlindPix, MHCalibrationChargePINDiode
//           MCalibrationChargePix, MCalibrationChargeCam, MCalibrationChargeCalc
//           MCalibrationChargeBlindPix, MCalibrationChargePINDiode,
//           MCalibrationQECam, MBadPixelsPix, MBadPixelsCam
//
// If the flag RelTimeCalibration() is set, a calibration of the relative arrival 
// times is also performed. The following containers (rectangular) and 
// tasks (ellipses) are called to produce an MCalibrationRelTimeCam used by  
// MCalibrateTime to correct timing offset between pixels: (MCalibrateTime is not 
// called from this class)
//
//Begin_Html
/*
<img src="images/RelTimeClasses.gif">
*/
//End_Html
//
// Different arrival time extractors can be set directly with the command SetTimeExtractor(MExtractor *)
//
// See also: MHCalibrationRelTimePix, MHCalibrationRelTimeCam, MHGausEvents
//           MCalibrationRelTimePix, MCalibrationRelTimeCam
//           MBadPixelsPix, MBadPixelsCam
//
/////////////////////////////////////////////////////////////////////////////
#include "MJCalibration.h"

#include <TFile.h>
#include <TStyle.h>
#include <TCanvas.h>
#include <TSystem.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MRunIter.h"
#include "MParList.h"
#include "MTaskList.h"
#include "MEvtLoop.h"

#include "MHCamera.h"
#include "MGeomCam.h"

#include "MPedestalCam.h"
#include "MCalibrationCam.h"
#include "MCalibrationQECam.h"
#include "MCalibrationChargeCam.h"
#include "MCalibrationChargePINDiode.h"
#include "MCalibrationChargeBlindPix.h"
#include "MCalibrationChargeCalc.h"

#include "MHGausEvents.h"
#include "MHCalibrationCam.h"
#include "MHCalibrationChargeCam.h"
#include "MHCalibrationChargeBlindPix.h"
#include "MHCalibrationRelTimeCam.h"
#include "MCalibrationRelTimeCam.h"
#include "MCalibrationRelTimeCalc.h"

#include "MReadMarsFile.h"
#include "MRawFileRead.h"
#include "MGeomApply.h"
#include "MBadPixelsMerge.h"
#include "MBadPixelsCam.h"
#include "MExtractTime.h"
#include "MExtractor.h"
#include "MExtractPINDiode.h"
#include "MExtractBlindPixel.h"
#include "MExtractSlidingWindow.h"
#include "MExtractTimeSpline.h"
#include "MFCosmics.h"
#include "MContinue.h"
#include "MFillH.h"

#include "MArrivalTimeCam.h"

#include "MStatusDisplay.h"

ClassImp(MJCalibration);

using namespace std;

const Int_t MJCalibration::gkIFAEBoxInaugurationRun = 20113;
// --------------------------------------------------------------------------
//
// Default constructor. 
//
// - Sets fRuns to 0, fExtractor to NULL, fTimeExtractor to NULL, fColor to kNONE, 
//   fDisplay to kNormalDisplay, fRelTime to kFALSE, fDataCheck to kFALSE, 
// - SetUseBlindPixel()
// - SetUsePINDiode()
//
MJCalibration::MJCalibration(const char *name, const char *title) 
    : fRuns(0), fExtractor(NULL), fTimeExtractor(NULL), 
      fColor(MCalibrationCam::kNONE), fDisplayType(kNormalDisplay),
      fRelTimes(kFALSE), fDataCheck(kFALSE)
{

  fName  = name  ? name  : "MJCalibration";
  fTitle = title ? title : "Tool to create the calibration constants for one calibration run";

  SetUseBlindPixel();
  SetUsePINDiode();

}


// --------------------------------------------------------------------------
//
// Display the results in MStatusDisplay: 
//
// - Add "Calibration" to the MStatusDisplay title
// - Retrieve the MGeomCam from MParList
// - Initialize the following MHCamera's:
//   1)  MCalibrationPix::GetMean()
//   2)  MCalibrationPix::Sigma()
//   3)  MCalibrationChargePix::GetRSigma()
//   4)  MCalibrationChargePix::GetRSigmaPerCharge()
//   5)  MCalibrationChargePix::GetPheFFactorMethod()
//   6)  MCalibrationChargePix::GetMeanConvFADC2Phe()
//   7)  MCalibrationChargePix::GetMeanFFactorFADC2Phot()
//   8)  MCalibrationQEPix::GetQECascadesFFactor()
//   9)  MCalibrationQEPix::GetQECascadesBlindPixel()
//   10) MCalibrationQEPix::GetQECascadesPINDiode()
//   11) MCalibrationQEPix::GetQECascadesCombined()
//   12) MCalibrationQEPix::IsAverageQEFFactorAvailable()
//   13) MCalibrationQEPix::IsAverageQEBlindPixelAvailable()
//   14) MCalibrationQEPix::IsAverageQEPINDiodeAvailable()
//   15) MCalibrationQEPix::IsAverageQECombinedAvailable()
//   16) MCalibrationChargePix::IsHiGainSaturation()
//   17) MCalibrationPix::GetHiLoMeansDivided()
//   18) MCalibrationPix::GetHiLoSigmasDivided()
//   19) MCalibrationChargePix::GetHiGainPickup()
//   20) MCalibrationChargePix::GetLoGainPickup()
//   21) MCalibrationChargePix::GetHiGainBlackout()
//   22) MCalibrationChargePix::GetLoGainBlackout()
//   23) MCalibrationPix::IsExcluded()
//   24) MBadPixelsPix::IsUnsuitable(MBadPixelsPix::kUnsuitableRun)
//   25) MBadPixelsPix::IsUnsuitable(MBadPixelsPix::kUnreliableRun)
//   26) MBadPixelsPix::IsUncalibrated(MBadPixelsPix::kHiGainOscillating)
//   27) MBadPixelsPix::IsUncalibrated(MBadPixelsPix::kLoGainOscillating)
//   28) MCalibrationChargePix::GetAbsTimeMean()
//   29) MCalibrationChargePix::GetAbsTimeRms()
//
// If the flag SetFullDisplay() is set, all MHCameras will be displayed. 
// if the flag SetDataCheckDisplay() is set, only the most important ones are displayed 
// and otherwise, (default: SetNormalDisplay()), a good selection of plots is given
//
void MJCalibration::DisplayResult(MParList &plist)
{
    if (!fDisplay)
        return;

    //
    // Update display
    //
    TString title = fDisplay->GetTitle();
    title += "--  Calibration ";
    title += fRuns->GetRunsAsString();
    title += "  --";
    fDisplay->SetTitle(title);

    //
    // Get container from list
    //
    MGeomCam &geomcam = *(MGeomCam*)plist.FindObject("MGeomCam");

    // Create histograms to display
    MHCamera disp1 (geomcam, Form("%s%s","Charge",(fRuns->GetRunsAsFileName()).Data()),
                    "Fitted Mean Charges");
    MHCamera disp2 (geomcam, Form("%s%s","SigmaCharge",(fRuns->GetRunsAsFileName()).Data()),
                    "Sigma of Fitted Charges");
    MHCamera disp3 (geomcam, Form("%s%s","RSigma",(fRuns->GetRunsAsFileName()).Data()),
                    "Reduced Sigmas");
    MHCamera disp4 (geomcam, Form("%s%s","RSigmaPerCharge",(fRuns->GetRunsAsFileName()).Data()),  
                    "Reduced Sigma per Charge");
    MHCamera disp5 (geomcam, Form("%s%s","NumPhes",(fRuns->GetRunsAsFileName()).Data()),
                    "Nr. of Phe's (F-Factor Method)");
    MHCamera disp6 (geomcam, Form("%s%s","ConvFADC2Phes",(fRuns->GetRunsAsFileName()).Data()),
                    "Conversion Factor (F-Factor Method)");
    MHCamera disp7 (geomcam, Form("%s%s","TotalFFactor",(fRuns->GetRunsAsFileName()).Data()),
                    "Total F-Factor (F-Factor Method)");
    MHCamera disp8 (geomcam, Form("%s%s","CascadesQEFFactor",(fRuns->GetRunsAsFileName()).Data()),
                    "Cascades QE (F-Factor Method)");
    MHCamera disp9 (geomcam, Form("%s%s","CascadesQEBlindPix",(fRuns->GetRunsAsFileName()).Data()),
                    "Cascades QE (Blind Pixel Method)");
    MHCamera disp10(geomcam, Form("%s%s","CascadesQEPINDiode",(fRuns->GetRunsAsFileName()).Data()),
                    "Cascades QE (PIN Diode Method)");
    MHCamera disp11(geomcam, Form("%s%s","CascadesQECombined",(fRuns->GetRunsAsFileName()).Data()),
                    "Cascades QE (Combined Method)");
    MHCamera disp12(geomcam, Form("%s%s","FFactorValid",(fRuns->GetRunsAsFileName()).Data()),
                    "Pixels with valid F-Factor calibration");
    MHCamera disp13(geomcam, Form("%s%s","BlindPixelValid",(fRuns->GetRunsAsFileName()).Data()), 
                    "Pixels with valid BlindPixel calibration");
    MHCamera disp14(geomcam, Form("%s%s","PINdiodeValid",(fRuns->GetRunsAsFileName()).Data()),
                    "Pixels with valid PINDiode calibration");
    MHCamera disp15(geomcam, Form("%s%s","CombinedValid",(fRuns->GetRunsAsFileName()).Data()),
                    "Pixels with valid Combined calibration");
    MHCamera disp16(geomcam, Form("%s%s","Saturation",(fRuns->GetRunsAsFileName()).Data()),
                    "Pixels with saturated Hi Gain");
    MHCamera disp17(geomcam, Form("%s%s","ConversionMeans",(fRuns->GetRunsAsFileName()).Data()), 
                    "Conversion HiGain.vs.LoGain Means");
    MHCamera disp18(geomcam, Form("%s%s","ConversionSigmas",(fRuns->GetRunsAsFileName()).Data()), 
                    "Conversion HiGain.vs.LoGain Sigmas");
    MHCamera disp19(geomcam, Form("%s%s","HiGainPickup",(fRuns->GetRunsAsFileName()).Data()),
                    "Number Pickup events Hi Gain");
    MHCamera disp20(geomcam, Form("%s%s","LoGainPickup",(fRuns->GetRunsAsFileName()).Data()),
                    "Number Pickup events Lo Gain");
    MHCamera disp21(geomcam, Form("%s%s","HiGainBlackout",(fRuns->GetRunsAsFileName()).Data()),
                    "Number Blackout events Hi Gain");
    MHCamera disp22(geomcam, Form("%s%s","LoGainBlackout",(fRuns->GetRunsAsFileName()).Data()), 
                    "Number Blackout events Lo Gain");
    MHCamera disp23(geomcam, Form("%s%s","Excluded",(fRuns->GetRunsAsFileName()).Data()),
                    "Pixels previously excluded");
    MHCamera disp24(geomcam, Form("%s%s","UnSuitable",(fRuns->GetRunsAsFileName()).Data()),
                    "Pixels not suited for further analysis");
    MHCamera disp25(geomcam, Form("%s%s","UnReliable",(fRuns->GetRunsAsFileName()).Data()),
                    "Pixels not reliable for further analysis");
    MHCamera disp26(geomcam, Form("%s%s","HiGainOscillating",(fRuns->GetRunsAsFileName()).Data()),
                    "Oscillating Pixels High Gain");
    MHCamera disp27(geomcam, Form("%s%s","LoGainOscillating",(fRuns->GetRunsAsFileName()).Data()),
                    "Oscillating Pixels Low Gain");
    MHCamera disp28(geomcam, Form("%s%s","AbsTimeMean",(fRuns->GetRunsAsFileName()).Data()),
                    "Abs. Arrival Times");
    MHCamera disp29(geomcam, Form("%s%s","AbsTimeRms",(fRuns->GetRunsAsFileName()).Data()),
                    "RMS of Arrival Times");
    MHCamera disp30(geomcam, Form("%s%s","MeanTime",(fRuns->GetRunsAsFileName()).Data()),
                    "Mean Rel. Arrival Times");
    MHCamera disp31(geomcam, Form("%s%s","SigmaTime",(fRuns->GetRunsAsFileName()).Data()),
                    "Sigma Rel. Arrival Times");
    MHCamera disp32(geomcam, Form("%s%s","TimeProb",(fRuns->GetRunsAsFileName()).Data()),
                    "Probability of Time Fit");
    MHCamera disp33(geomcam, Form("%s%s","TimeNotFitValid",(fRuns->GetRunsAsFileName()).Data()),
                    "Pixels with not valid fit results");
    MHCamera disp34(geomcam, Form("%s%s","TimeOscillating",(fRuns->GetRunsAsFileName()).Data()),
                    "Oscillating Pixels");

    // Fitted charge means and sigmas
    disp1.SetCamContent(fCalibrationCam,  0);
    disp1.SetCamError(  fCalibrationCam,  1);
    disp2.SetCamContent(fCalibrationCam,  2);
    disp2.SetCamError(  fCalibrationCam,  3);

    // Reduced Sigmas and reduced sigmas per charge
    disp3.SetCamContent(fCalibrationCam,  5);
    disp3.SetCamError(  fCalibrationCam,  6);
    disp4.SetCamContent(fCalibrationCam,  7);
    disp4.SetCamError(  fCalibrationCam,  8);

    // F-Factor Method
    disp5.SetCamContent(fCalibrationCam,  9);
    disp5.SetCamError(  fCalibrationCam, 10);
    disp6.SetCamContent(fCalibrationCam, 11);
    disp6.SetCamError(  fCalibrationCam, 12);
    disp7.SetCamContent(fCalibrationCam, 13);
    disp7.SetCamError(  fCalibrationCam, 14);

    // Quantum Efficiencies
    disp8.SetCamContent (fQECam, 0 );
    disp8.SetCamError   (fQECam, 1 );
    disp9.SetCamContent (fQECam, 2 );
    disp9.SetCamError   (fQECam, 3 );
    disp10.SetCamContent(fQECam, 4 );
    disp10.SetCamError  (fQECam, 5 );
    disp11.SetCamContent(fQECam, 6 );
    disp11.SetCamError  (fQECam, 7 );

    // Valid flags
    disp12.SetCamContent(fQECam, 8 );
    disp13.SetCamContent(fQECam, 9 );
    disp14.SetCamContent(fQECam, 10);
    disp15.SetCamContent(fQECam, 11);

    // Conversion Hi-Lo
    disp16.SetCamContent(fCalibrationCam, 25);
    disp17.SetCamContent(fCalibrationCam, 16);
    disp17.SetCamError  (fCalibrationCam, 17);
    disp18.SetCamContent(fCalibrationCam, 18);
    disp18.SetCamError  (fCalibrationCam, 19);

    // Pickup and Blackout
    disp19.SetCamContent(fCalibrationCam, 21);
    disp20.SetCamContent(fCalibrationCam, 22);
    disp21.SetCamContent(fCalibrationCam, 23);
    disp22.SetCamContent(fCalibrationCam, 24);

    // Pixels with defects
    disp23.SetCamContent(fCalibrationCam, 20);
    disp24.SetCamContent(fBadPixels, 1);
    disp25.SetCamContent(fBadPixels, 3);

    // Oscillations
    disp26.SetCamContent(fBadPixels, 10);
    disp27.SetCamContent(fBadPixels, 11);

    // Arrival Times
    disp28.SetCamContent(fCalibrationCam, 26);
    disp28.SetCamError(  fCalibrationCam, 27);
    disp29.SetCamContent(fCalibrationCam, 27);

    disp1.SetYTitle("Q [FADC counts]");
    disp2.SetYTitle("\\sigma_{Q} [FADC counts]");

    disp3.SetYTitle("\\sqrt{\\sigma^{2}_{Q} - RMS^{2}_{Ped}} [FADC Counts]");
    disp4.SetYTitle("Red.Sigma/<Q> [1]");

    disp5.SetYTitle("Nr. Phe's [1]");
    disp6.SetYTitle("Conv.Factor [PhE/FADC counts]");
    disp7.SetYTitle("Total F-Factor [1]");

    disp8.SetYTitle("QE [1]");
    disp9.SetYTitle("QE [1]");
    disp10.SetYTitle("QE [1]");
    disp11.SetYTitle("QE [1]");

    disp12.SetYTitle("[1]");
    disp13.SetYTitle("[1]");
    disp14.SetYTitle("[1]");
    disp15.SetYTitle("[1]");
    disp16.SetYTitle("[1]");

    disp17.SetYTitle("<Q>(High)/<Q>(Low) [1]");
    disp18.SetYTitle("\\sigma_{Q}(High)/\\sigma_{Q}(Low) [1]");

    disp19.SetYTitle("[1]");
    disp20.SetYTitle("[1]");
    disp21.SetYTitle("[1]");
    disp22.SetYTitle("[1]");
    disp23.SetYTitle("[1]");
    disp24.SetYTitle("[1]");
    disp25.SetYTitle("[1]");
    disp26.SetYTitle("[1]");
    disp27.SetYTitle("[1]");

    disp28.SetYTitle("Mean Abs. Time [FADC slice]");
    disp29.SetYTitle("RMS Abs. Time [FADC slices]");

    if (fRelTimes)
      {

        disp30.SetCamContent(fRelTimeCam,0);
        disp30.SetCamError(  fRelTimeCam,1);
        disp31.SetCamContent(fRelTimeCam,2);
        disp31.SetCamError(  fRelTimeCam,3);
        disp32.SetCamContent(fRelTimeCam,4);
        disp33.SetCamContent(fBadPixels,20);
        disp34.SetCamContent(fBadPixels,21);
    
        disp30.SetYTitle("Time Offset [FADC units]");
        disp31.SetYTitle("Timing resolution [FADC units]");
        disp32.SetYTitle("P_{Time} [1]");
        disp33.SetYTitle("[1]");
        disp34.SetYTitle("[1]");
      }
    
    if (fDisplayType == kDataCheckDisplay)
      {
        TCanvas &c1 = fDisplay->AddTab("Fit.Charge");
        c1.Divide(3, 3);
        
        disp1.CamDraw(c1, 1, 3, 2);
        disp4.CamDraw(c1, 2, 3, 2);
        disp28.CamDraw(c1, 3, 3, 2);

        //  F-Factor
        TCanvas &c2 = fDisplay->AddTab("Phe's");
        c2.Divide(3,4);
        
        disp6.CamDraw(c2, 1, 3,  2, 1);
        disp7.CamDraw(c2, 2, 3,  2, 1);
        disp8.CamDraw(c2, 3, 3,  2, 1);

        // QE's
        TCanvas &c3 = fDisplay->AddTab("QE's");
        c3.Divide(3,4);

        disp8.CamDraw(c3, 1, 3,  2, 1);
        disp9.CamDraw(c3, 2, 3,  2, 1);
        disp10.CamDraw(c3, 3, 3,  2, 1);

        // Defects
        TCanvas &c4 = fDisplay->AddTab("Defect");
        //        c4.Divide(3,2);
        c4.Divide(2,2);
        
        //        CamDraw(c4, 1, 3, disp23, 0);
        //        CamDraw(c4, 2, 3, disp24, 0);
        //        CamDraw(c4, 3, 3, disp25, 0);
        disp24.CamDraw(c4, 1, 2, 0);
        disp25.CamDraw(c4, 2, 2, 0);

        if (fRelTimes)
          {
            // Rel. Times
            TCanvas &c5 = fDisplay->AddTab("Rel. Times");
            c5.Divide(2,4);
            
            disp30.CamDraw(c5, 1, 2, 2);
            disp31.CamDraw(c5, 2, 2, 2);
          }


        return;
      }
    
    if (fDisplayType == kNormalDisplay)
      {

        // Charges
        TCanvas &c11 = fDisplay->AddTab("Fit.Charge");
        c11.Divide(2, 4);
        
        disp1.CamDraw(c11, 1, 2, 5, 1);
        disp2.CamDraw(c11, 2, 2, 5, 1);
        
        // Reduced Sigmas
        TCanvas &c12 = fDisplay->AddTab("Red.Sigma");
        c12.Divide(2,4);
        
        disp3.CamDraw(c12, 1, 2, 5, 1);
        disp4.CamDraw(c12, 2, 2, 5, 1);
        
        //  F-Factor
        TCanvas &c13 = fDisplay->AddTab("Phe's");
        c13.Divide(3,4);
        
        disp5.CamDraw(c13, 1, 3, 5, 1);
        disp6.CamDraw(c13, 2, 3, 5, 1);
        disp7.CamDraw(c13, 3, 3, 5, 1);
        
        // QE's
        TCanvas &c14 = fDisplay->AddTab("QE's");
        c14.Divide(4,4);
        
        disp8.CamDraw(c14, 1, 4, 5, 1);
        disp9.CamDraw(c14, 2, 4, 5, 1);
        disp10.CamDraw(c14, 3, 4, 5, 1);
        disp11.CamDraw(c14, 4, 4, 5, 1);
        
        // Defects
        TCanvas &c15 = fDisplay->AddTab("Defect");
        //      c15.Divide(5,2);
        c15.Divide(4,2);
        
        /*
        disp23.CamDraw(c15, 1, 5, 0);
        disp24.CamDraw(c15, 2, 5, 0);
        disp25.CamDraw(c15, 3, 5, 0);
        disp26.CamDraw(c15, 4, 5, 0);
        disp27.CamDraw(c15, 5, 5, 0);
        */
        disp24.CamDraw(c15, 1, 4, 0);
        disp25.CamDraw(c15, 2, 4, 0);
        disp26.CamDraw(c15, 3, 4, 0);
        disp27.CamDraw(c15, 4, 4, 0);
        
        // Abs. Times
        TCanvas &c16 = fDisplay->AddTab("Abs. Times");
        c16.Divide(2,3);
        
        disp28.CamDraw(c16, 1, 2, 5);
        disp29.CamDraw(c16, 2, 2, 5);

        if (fRelTimes)
          {
            // Rel. Times
            TCanvas &c17 = fDisplay->AddTab("Rel. Times");
            c17.Divide(2,4);
            
            disp30.CamDraw(c17, 1, 2, 5, 1);
            disp31.CamDraw(c17, 2, 2, 5, 1);
          }
        
        return;
      }
    
    if (fDisplayType == kFullDisplay)
      {

        MHCalibrationCam *cam = (MHCalibrationCam*)plist.FindObject("MHCalibrationChargeCam");

        for (Int_t sector=1;sector<cam->GetAverageSectors();sector++)
          {
            cam->GetAverageHiGainSector(sector).DrawClone("all");
            cam->GetAverageLoGainSector(sector).DrawClone("all");
          }

        // Charges
        TCanvas &c21 = fDisplay->AddTab("Fit.Charge");
        c21.Divide(2, 4);
        
        disp1.CamDraw(c21, 1, 2, 2, 1);
        disp2.CamDraw(c21, 2, 2, 2, 1);
        
        // Reduced Sigmas
        TCanvas &c23 = fDisplay->AddTab("Red.Sigma");
        c23.Divide(2,4);
        
        disp3.CamDraw(c23, 1, 2, 2, 1);
        disp4.CamDraw(c23, 2, 2, 2, 1);
        
        //  F-Factor
        TCanvas &c24 = fDisplay->AddTab("Phe's");
        c24.Divide(3,5);
        
        disp5.CamDraw(c24, 1, 3, 2, 1, 1);
        disp6.CamDraw(c24, 2, 3, 2, 1, 1);
        disp7.CamDraw(c24, 3, 3, 2, 1, 1);
        
        // QE's
        TCanvas &c25 = fDisplay->AddTab("QE's");
        c25.Divide(4,5);
        
        disp8.CamDraw(c25, 1, 4, 2, 1, 1);
        disp9.CamDraw(c25, 2, 4, 2, 1, 1);
        disp10.CamDraw(c25, 3, 4, 2, 1, 1);
        disp11.CamDraw(c25, 4, 4, 2, 1, 1);
        
        // Validity
        TCanvas &c26 = fDisplay->AddTab("Valid");
        c26.Divide(4,2);
        
        disp12.CamDraw(c26, 1, 4, 0);
        disp13.CamDraw(c26, 2, 4, 0);
        disp14.CamDraw(c26, 3, 4, 0);
        disp15.CamDraw(c26, 4, 4, 0);
        
        // Other info
        TCanvas &c27 = fDisplay->AddTab("HiLoGain");
        c27.Divide(3,3);
        
        disp16.CamDraw(c27, 1, 3, 0);
        disp17.CamDraw(c27, 2, 3, 1);
        disp18.CamDraw(c27, 3, 3, 1);
        
        // Pickup
        TCanvas &c28 = fDisplay->AddTab("Pickup");
        c28.Divide(4,2);
        
        disp19.CamDraw(c28, 1, 4, 0);
        disp20.CamDraw(c28, 2, 4, 0);
        disp21.CamDraw(c28, 3, 4, 0);
        disp22.CamDraw(c28, 4, 4, 0);
        
        // Defects
        TCanvas &c29 = fDisplay->AddTab("Defect");
        //      c29.Divide(5,2);
        c29.Divide(4,2);
        
        disp24.CamDraw(c29, 1, 4, 0);
        disp25.CamDraw(c29, 2, 4, 0);
        disp26.CamDraw(c29, 3, 4, 0);
        disp27.CamDraw(c29, 4, 4, 0);
        
        // Abs. Times
        TCanvas &c30 = fDisplay->AddTab("Abs. Times");
        c30.Divide(2,3);
        
        disp28.CamDraw(c30, 1, 2, 2);
        disp29.CamDraw(c30, 2, 2, 1);

        if (fRelTimes)
          {
            // Rel. Times
            TCanvas &c31 = fDisplay->AddTab("Rel. Times");
            c31.Divide(3,5);
            
            disp30.CamDraw(c31, 1, 3, 2, 1, 1);
            disp31.CamDraw(c31, 2, 3, 2, 1, 1);
            disp32.CamDraw(c31, 3, 3, 4, 1, 1);

            // Time Defects
            TCanvas &c32 = fDisplay->AddTab("Time Def.");
            c32.Divide(2,2);
            
            disp33.CamDraw(c32, 1, 2, 0);
            disp34.CamDraw(c32, 2, 2, 0);

            MHCalibrationCam *cam = (MHCalibrationCam*)plist.FindObject("MHCalibrationRelTimeCam");
            
            for (Int_t sector=1;sector<cam->GetAverageSectors();sector++)
              {
                cam->GetAverageHiGainSector(sector).DrawClone("fourierevents");
                cam->GetAverageLoGainSector(sector).DrawClone("fourierevents");
              }
            
          }

        return;
      }
}



// --------------------------------------------------------------------------
//
// Find the colour of the pulsing LED:
// - If the run number is smaller than gkIFAEBoxInaugurationRun, take MCalibrationCam::kCT1
// - Otherwise find the colour out of the run name
// - If no colour is found, return kFALSE
// 
Bool_t MJCalibration::FindColor() 
{

  const UInt_t nruns = fRuns->GetNumRuns();

  if (nruns == 0)
    return kFALSE;
  
  TArrayI arr = fRuns->GetRuns();

  if (arr[nruns-1] < gkIFAEBoxInaugurationRun)
    {
      *fLog << "Found colour kCT1 in runs: " << fRuns->GetRunsAsString() << endl;
      fColor = MCalibrationCam::kCT1;
      return kTRUE;
    }
  
  TString filenames;
  ((MDirIter*)fRuns)->Reset();

  while (!(filenames=((MDirIter*)fRuns)->Next()).IsNull())
    {

      filenames.ToLower();

      //
      // Here starts the list of runs where the shifters did not put 
      // a colour, but which have been found out by other means
      //
      if (filenames.Contains("_26924_"))
        if (fColor == MCalibrationCam::kNONE)
          {
            *fLog << "Found colour: kGREEN  in " << filenames << endl;
            fColor = MCalibrationCam::kGREEN;
          }
        else if (fColor != MCalibrationCam::kGREEN)
          {
            *fLog << err << "Different colour found in " << filenames << "... abort" << endl;
            return kFALSE;
          }


      if (filenames.Contains("_26568_"))
        if (fColor == MCalibrationCam::kNONE)
          {
            *fLog << "Found colour: kGREEN  in " << filenames << endl;
            fColor = MCalibrationCam::kGREEN;
          }
        else if (fColor != MCalibrationCam::kGREEN)
          {
            *fLog << err << "Different colour found in " << filenames << "... abort" << endl;
            return kFALSE;
          }

      if (filenames.Contains("_26412_"))
        if (fColor == MCalibrationCam::kNONE)
          {
            *fLog << "Found colour: kGREEN  in " << filenames << endl;
            fColor = MCalibrationCam::kGREEN;
          }
        else if (fColor != MCalibrationCam::kGREEN)
          {
            *fLog << err << "Different colour found in " << filenames << "... abort" << endl;
            return kFALSE;
          }


      if (filenames.Contains("_26409_"))
        if (fColor == MCalibrationCam::kNONE)
          {
            *fLog << "Found colour: kGREEN  in " << filenames << endl;
            fColor = MCalibrationCam::kGREEN;
          }
        else if (fColor != MCalibrationCam::kGREEN)
          {
            *fLog << err << "Different colour found in " << filenames << "... abort" << endl;
            return kFALSE;
          }

      if (filenames.Contains("_26408_"))
        if (fColor == MCalibrationCam::kNONE)
          {
            *fLog << "Found colour: kGREEN  in " << filenames << endl;
            fColor = MCalibrationCam::kGREEN;
          }
        else if (fColor != MCalibrationCam::kGREEN)
          {
            *fLog << err << "Different colour found in " << filenames << "... abort" << endl;
            return kFALSE;
          }


      if (filenames.Contains("_26402_"))
        if (fColor == MCalibrationCam::kNONE)
          {
            *fLog << "Found colour: kBLUE  in " << filenames << endl;
            fColor = MCalibrationCam::kBLUE;
          }
        else if (fColor != MCalibrationCam::kBLUE)
          {
            *fLog << err << "Different colour found in " << filenames << "... abort" << endl;
            return kFALSE;
          }

      if (filenames.Contains("_20661_"))
        if (fColor == MCalibrationCam::kNONE)
          {
            *fLog << "Found colour: kGREEN  in " << filenames << endl;
            fColor = MCalibrationCam::kGREEN;
          }
        else if (fColor != MCalibrationCam::kGREEN)
          {
            *fLog << err << "Different colour found in " << filenames << "... abort" << endl;
            return kFALSE;
          }

      if (filenames.Contains("_20660_"))
        if (fColor == MCalibrationCam::kNONE)
          {
            *fLog << "Found colour: kGREEN  in " << filenames << endl;
            fColor = MCalibrationCam::kGREEN;
          }
        else if (fColor != MCalibrationCam::kGREEN)
          {
            *fLog << err << "Different colour found in " << filenames << "... abort" << endl;
            return kFALSE;
          }

      //
      // Here start the runs where the shifter put 
      // the colour
      //
      if (filenames.Contains("green"))
        if (fColor == MCalibrationCam::kNONE)
          {
            *fLog << "Found colour: kGREEN  in " << filenames << endl;
            fColor = MCalibrationCam::kGREEN;
          }
        else if (fColor != MCalibrationCam::kGREEN)
          {
            *fLog << err << "Different colour found in " << filenames << "... abort" << endl;
            return kFALSE;
          }

      if (filenames.Contains("blue"))
        if (fColor == MCalibrationCam::kNONE)
          {
            *fLog << "Found colour: kBLUE  in " << filenames << endl;
            fColor = MCalibrationCam::kBLUE;
          }
        else if (fColor != MCalibrationCam::kBLUE)
          {
            *fLog << err << "Different colour found in " << filenames << "... abort" << endl;
            return kFALSE;
          }

      if (filenames.Contains("uv"))
        if (fColor == MCalibrationCam::kNONE)
          {
            *fLog << "Found colour: kUV in " << filenames << endl;
            fColor = MCalibrationCam::kUV;
          }
        else if (fColor != MCalibrationCam::kUV)
          {
            *fLog << err << "Different colour found in " << filenames << "... abort" << endl;
            return kFALSE;
          }

      if (filenames.Contains("ct1"))
        if (fColor == MCalibrationCam::kNONE)
          {
            *fLog << "Found colour: kCT1  in " << filenames << endl;
            fColor = MCalibrationCam::kCT1;
          }
        else if (fColor != MCalibrationCam::kCT1)
          {
            *fLog << err << "Different colour found in " << filenames << "... abort" << endl;
            return kFALSE;
          }
      
    }
  
  if (fColor == MCalibrationCam::kNONE)
    {
      *fLog <<  "No colour found in filenames of runs: " << fRuns->GetRunsAsString() 
           << "... abort" << endl;
      return kFALSE;      
    }

  return kTRUE;
}




// --------------------------------------------------------------------------
//
// Retrieve the output file written by WriteResult()
// 
const char* MJCalibration::GetOutputFile() const
{

  if (!fRuns)
    return "";
  
  return Form("%s/%s-F1.root", (const char*)fOutputPath, (const char*)fRuns->GetRunsAsFileName());
}


Bool_t MJCalibration::IsUseBlindPixel() const 
{
  return TESTBIT(fDevices,kUseBlindPixel);
}


Bool_t MJCalibration::IsUsePINDiode() const 
{
  return TESTBIT(fDevices,kUsePINDiode);
}

  



// --------------------------------------------------------------------------
//
// Call the ProcessFile(MPedestalCam)
// 
Bool_t MJCalibration::Process(MPedestalCam &pedcam)
{
    if (!ReadCalibrationCam())
        return ProcessFile(pedcam);

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Execute the task list and the eventloop:
//
// - Check if there are fRuns, otherwise return
// - Check the colour of the files in fRuns (FindColor()), otherwise return
// - Check for consistency between run numbers and number of files
// - Add fRuns to MReadMarsFile
// - Put into MParList:
//   1) MPedestalCam          (pedcam)
//   2) MCalibrationQECam     (fQECam)
//   3) MCalibrationChargeCam (fCalibrationCam)
//   4) MCalibrationRelTimeCam (fRelTimeCam)   (only if flag fRelTimes is chosen)
//   5) MBadPixelsCam         (fBadPixels)
//   6) MCalibrationChargePINDiode
//   7) MCalibrationChargeBlindPix
// - Put into the MTaskList:
//   1)  MReadMarsFile
//   2)  MBadPixelsMerge
//   3)  MGeomApply
//   4)  MExtractor
//   5)  MExtractPINDiode
//   6)  MExtractBlindPixel
//   7)  MExtractTime (only if flag fRelTimes is chosen)
//   8)  MContinue(MFCosmics)
//   9)  MFillH("MHCalibrationChargePINDiode", "MExtractedSignalPINDiode")
//   10) MFillH("MHCalibrationChargeBlindPix", "MExtractedSignalBlindPixel")
//   11) MFillH("MHCalibrationChargeCam",      "MExtractedSignalCam")
//   12) MFillH("MHCalibrationChargeCam",      "MExtractedSignalCam")
//   13) MCalibrationChargeCalc
//   14) MFillH("MHCalibrationRelTimeCam",     "MArrivalTimeCam") (only if flag fRelTimes is chosen)
//   15) MCalibrationRelTimeCalc
// - Execute MEvtLoop
// - DisplayResult()
// - WriteResult()
//
Bool_t MJCalibration::ProcessFile(MPedestalCam &pedcam)
{
  if (!fRuns)
    {
      *fLog << err << "No Runs choosen... abort." << endl;
      return kFALSE;
    }
  
  if (fRuns->GetNumRuns() != fRuns->GetNumEntries())
    {
      *fLog << err << "Number of files found doesn't match number of runs... abort." 
            << fRuns->GetNumRuns() << " vs. " << fRuns->GetNumEntries() << endl;
      return kFALSE;
    }

  *fLog << inf;
  fLog->Separator(GetDescriptor());

  if (!FindColor())
    return kFALSE;
  
  *fLog << "Calculate MCalibrationCam from Runs " << fRuns->GetRunsAsString() << endl;
  *fLog << endl;
  
  // Setup Tasklist
  MParList plist;
  MTaskList tlist;
  plist.AddToList(&tlist);
  
  MReadMarsFile read("Events");
  MRawFileRead rawread(NULL);

  if (fDataCheck)
  {
     rawread.AddFiles(*fRuns); 
     tlist.AddToList(&rawread);
  }
  else
  {
      read.DisableAutoScheme();
      static_cast<MRead&>(read).AddFiles(*fRuns);
      tlist.AddToList(&read);
  }

  plist.AddToList(&pedcam);
  plist.AddToList(&fBadPixels);
  plist.AddToList(&fQECam);
  plist.AddToList(&fCalibrationCam);
  plist.AddToList(&fCalibrationBlindPix);
  plist.AddToList(&fCalibrationPINDiode);
  plist.AddToList(&fRelTimeCam);

  MGeomApply               apply;
  //    MBadPixelsMerge          merge(&fBadPixels);
  MExtractPINDiode         pinext;
  MExtractBlindPixel       blindext;
  MExtractSlidingWindow    extract2;
  MExtractTimeSpline       timespline;
  MCalibrationChargeCalc   calcalc;
  calcalc.SetOutputPath(fOutputPath);
  calcalc.SetOutputFile(Form("%s-ChargeCalibStat.txt",(const char*)fRuns->GetRunsAsFileName()));

  MCalibrationRelTimeCalc  timecalc;
  timecalc.SetOutputPath(fOutputPath);
  timecalc.SetOutputFile(Form("%s-TimeCalibStat.txt",(const char*)fRuns->GetRunsAsFileName()));
  
  // 
  // As long as there are no DM's, have to colour by hand 
  //
  calcalc.SetPulserColor(fColor);
  
  MFillH fillpin("MHCalibrationChargePINDiode", "MExtractedSignalPINDiode");
  MFillH fillbnd("MHCalibrationChargeBlindPix", "MExtractedSignalBlindPixel");
  MFillH fillcam("MHCalibrationChargeCam",      "MExtractedSignalCam");
  MFillH filltme("MHCalibrationRelTimeCam",     "MArrivalTimeCam");
  fillpin.SetNameTab("PINDiode");
  fillbnd.SetNameTab("BlindPix");
  fillcam.SetNameTab("Charge");
  filltme.SetNameTab("RelTimes");

  if (fDisplayType == kFullDisplay)
    {
      fillcam.SetDrawOption("all");
      filltme.SetDrawOption("all");
    }
  
  
  // 
  // Apply a filter against cosmics
  // (will have to be needed in the future
  // when the calibration hardware-trigger is working)
  // 
  MFCosmics cosmics;
  MContinue cont(&cosmics);
  
  //    tlist.AddToList(&merge);
  tlist.AddToList(&apply);

  if (fExtractor)
    tlist.AddToList(fExtractor);
  else
    {
      *fLog << warn << GetDescriptor() 
            << ": No extractor has been chosen, take default MExtractSlidingWindow " << endl;
      tlist.AddToList(&extract2);
    }
  

  tlist.AddToList(&pinext);  
  tlist.AddToList(&blindext);
  
  if (fRelTimes)
    {
      if (fTimeExtractor)
        tlist.AddToList(fTimeExtractor);
      else
        {
          *fLog << warn << GetDescriptor() 
                << ": No extractor has been chosen, take default MTimeExtractSpline " << endl;
          tlist.AddToList(&timespline);
        }
    }

  if (fColor == MCalibrationCam::kCT1)
    tlist.AddToList(&cont);

  tlist.AddToList(&fillcam);

  if (IsUsePINDiode())
    tlist.AddToList(&fillpin);
  if (IsUseBlindPixel())
    tlist.AddToList(&fillbnd);

  tlist.AddToList(&calcalc);

  if (fRelTimes)
    {
      tlist.AddToList(&filltme);
      tlist.AddToList(&timecalc);
    }
  

  // Create and setup the eventloop
  MEvtLoop evtloop(fName);
  evtloop.SetParList(&plist);
  evtloop.SetDisplay(fDisplay);
  evtloop.SetLogStream(fLog);
  
  // Execute first analysis
  if (!evtloop.Eventloop())
    {
      *fLog << err << GetDescriptor() << ": Failed." << endl;
      return kFALSE;
    }
  
  tlist.PrintStatistics();

  DisplayResult(plist);

  if (!WriteResult())
    return kFALSE;
  
  *fLog << inf << GetDescriptor() << ": Done." << endl;
  
  return kTRUE;
}

// --------------------------------------------------------------------------
//
// Read the following containers from GetOutputFile()
// - MCalibrationChargeCam
// - MCalibrationQECam
// - MBadPixelsCam
//
Bool_t MJCalibration::ReadCalibrationCam()
{

  const TString fname = GetOutputFile();
  
  if (gSystem->AccessPathName(fname, kFileExists))
    {
      *fLog << err << "Input file " << fname << " doesn't exist." << endl;
      return kFALSE;
    }
  
  *fLog << inf << "Reading from file: " << fname << endl;

  TFile file(fname, "READ");
  if (fCalibrationCam.Read()<=0)
    {
      *fLog << err << "Unable to read MCalibrationChargeCam from " << fname << endl;
      return kFALSE;
    }
  
  if (fQECam.Read()<=0)
    {
      *fLog << err << "Unable to read MCalibrationQECam from " << fname << endl;
      return kFALSE;
    }
  

  if (fRelTimes)
    if (fRelTimeCam.Read()<=0)
      {
        *fLog << err << "Unable to read MCalibrationRelTimeCam from " << fname << endl;
        return kFALSE;
      }
  
  if (file.FindKey("MBadPixelsCam"))
    {
      MBadPixelsCam bad;
      if (bad.Read()<=0)
        {
          *fLog << err << "Unable to read MBadPixelsCam from " << fname << endl;
          return kFALSE;
        }
      fBadPixels.Merge(bad);
    }
  
  if (fDisplay /*&& !fDisplay->GetCanvas("Pedestals")*/) // FIXME!
    fDisplay->Read();
  
  return kTRUE;
}


// --------------------------------------------------------------------------
//
// Set the path for output files, written by WriteResult()
// 
void MJCalibration::SetOutputPath(const char *path)
{
    fOutputPath = path;
    if (fOutputPath.EndsWith("/"))
        fOutputPath = fOutputPath(0, fOutputPath.Length()-1);
}

// --------------------------------------------------------------------------
//
// Set the useage of the Blind Pixel device 
// 
void MJCalibration::SetUseBlindPixel(const Bool_t b)
{
  b ? SETBIT(fDevices,kUseBlindPixel) : CLRBIT(fDevices,kUseBlindPixel);
}

// --------------------------------------------------------------------------
//
// Set the useage of the PIN Diode device 
// 
void MJCalibration::SetUsePINDiode(const Bool_t b)
{
  b ? SETBIT(fDevices,kUsePINDiode) : CLRBIT(fDevices,kUsePINDiode);
}

// --------------------------------------------------------------------------
//
// Write the result into the output file GetOutputFile(), if fOutputPath exists.
// 
// The following containers are written:
// - MStatusDisplay
// - MCalibrationChargeCam
// - MCalibrationChargeBlindPix
// - MCalibrationQECam
// - MBadPixelsCam
//
Bool_t MJCalibration::WriteResult()
{
    if (fOutputPath.IsNull())
        return kTRUE;

    const TString oname(GetOutputFile());

    *fLog << inf << "Writing to file: " << oname << endl;

    TFile file(oname, "UPDATE");

    if (fDisplay && fDisplay->Write()<=0)
    {
        *fLog << err << "Unable to write MStatusDisplay to " << oname << endl;
        return kFALSE;
    }

    if (fCalibrationCam.Write()<=0)
    {
        *fLog << err << "Unable to write MCalibrationChargeCam to " << oname << endl;
        return kFALSE;
    }

    if (fCalibrationBlindPix.Write()<=0)
    {
        *fLog << err << "Unable to write MCalibrationChargeBlindPix to " << oname << endl;
        return kFALSE;
    }

    if (fCalibrationPINDiode.Write()<=0)
    {
        *fLog << err << "Unable to write MCalibrationChargePINDiode to " << oname << endl;
        return kFALSE;
    }

    if (fQECam.Write()<=0)
    {
        *fLog << err << "Unable to write MCalibrationQECam to " << oname << endl;
        return kFALSE;
    }

    if (fRelTimes)
	if (fRelTimeCam.Write()<=0)
	{
	    *fLog << err << "Unable to write MCalibrationQECam to " << oname << endl;
	    return kFALSE;
	}

    if (fBadPixels.Write()<=0)
    {
        *fLog << err << "Unable to write MBadPixelsCam to " << oname << endl;
        return kFALSE;
    }

    return kTRUE;

}

