/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug 02/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
//  MHPedestalPix
//
//  Histogram class for pedestal analysis. 
//  Stores and fits the pedestals taken from MPedestalPix on an event-by-event
//  basis. The results are re-normalized to a value per slice with the formulae:
//
// - Mean Pedestal        / slice = Mean Pedestal        / Number slices
// - Mean Pedestal Error  / slice = Mean Pedestal Error  / Number slices
// - Sigma Pedestal       / slice = Sigma Pedestal       / Sqrt (Number slices)
// - Sigma Pedestal Error / slice = Sigma Pedestal Error / Sqrt (Number slices)
// 
//  Derives from MHGausEvents, fits the pedestals to a Gaussian and performs
//  a Fourier analysis.
//
//////////////////////////////////////////////////////////////////////////////
#include "MHPedestalPix.h"

#include "MLog.h"
#include "MLogManip.h"

#include <TH1.h>

ClassImp(MHPedestalPix);

using namespace std;
//
const Int_t   MHPedestalPix::fgChargeNbins    = 500;
const Axis_t  MHPedestalPix::fgChargeFirst    = -49.5;
const Axis_t  MHPedestalPix::fgChargeLast     = 449.5;
// --------------------------------------------------------------------------
//
// Default Constructor. 
//
// Sets: 
// - the default number for fNbins        (fgChargeNbins)
// - the default number for fFirst        (fgChargeFirst)
// - the default number for fLast         (fgChargeLast)
//
// - the default name of the  fHGausHist ("HPedestalCharge")
// - the default title of the fHGausHist ("Distribution of Summed FADC Pedestal slices Pixel ")
// - the default x-axis title for fHGausHist ("Sum FADC Slices")
// - the default y-axis title for fHGausHist ("Nr. of events")
// - TH1::Sumw2() for fHGausHist
//
// Initializes:
// - fNSlices to 1
// - fProbLimit to 0.01
//
MHPedestalPix::MHPedestalPix(const char *name, const char *title) 
    : fNSlices(1)
{ 

  fName  = name  ? name  : "MHPedestalPix";
  fTitle = title ? title : "Histogrammed Pedestal events";

  SetNbins( fgChargeNbins );
  SetFirst( fgChargeFirst );
  SetLast(  fgChargeLast  );

  SetProbLimit(0.01);

  // Create a large number of bins, later we will rebin
  fHGausHist.SetName("HPedestalCharge");
  fHGausHist.SetTitle("Distribution of Summed FADC Pedestal Slices Pixel ");
  fHGausHist.SetXTitle("Sum FADC Slices");
  fHGausHist.SetYTitle("Nr. of events");
  fHGausHist.Sumw2();

}

// --------------------------------------------------------------------------
// 
// If mean and sigma have not yet been set, returns.
//
// Renormalizes the pedestal fit results by the following formulae:
//
// - Mean Pedestal        / slice = Mean Pedestal        / Number slices
// - Mean Pedestal Error  / slice = Mean Pedestal Error  / Number slices
// - Sigma Pedestal       / slice = Sigma Pedestal       / Sqrt (Number slices)
// - Sigma Pedestal Error / slice = Sigma Pedestal Error / Sqrt (Number slices)
// 
void MHPedestalPix::Renorm()
{

  //
  // One never knows...
  //
  if (fNSlices <= 0)
    return;

  const Float_t sqslices = TMath::Sqrt(fNSlices);

  SetMean     ( GetMean()    / fNSlices  );
  //
  // Mean error goes with PedestalRMS/Sqrt(entries) -> scale with slices
  // 
  SetMeanErr  ( GetMeanErr() / fNSlices  );
  //
  // Sigma goes like PedestalRMS -> scale with sqrt(slices)    
  //
  SetSigma    ( GetSigma()   / sqslices  );
  //
  // Sigma error goes like PedestalRMS/2.(entries) -> scale with sqrt(slices)
  //
  SetSigmaErr ( GetSigmaErr() / sqslices );
  
}

