#ifndef MARS_MCalibrateData
#define MARS_MCalibrateData

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// MCalibrateData
//                                                                         //
// Integrates the desired ADC time slices of one pixel and apply           //
// calibration constants                                                   //
//                                                                         //
// Differences between MCalibrateData and MCalibrate :
// in MCalibrateData
// - in ReInit the MPedPhot container is filled using
//      - the pedstals/slice from MPedestalCam
//      - the number of used FADC slices from MExtractedSignalCam   
//      - the photon/ADC conversion factor from MCalibrationCam
//                                                                         //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
#ifndef MARS_MTask
#include "MTask.h"
#endif

class MGeomCam;
class MBadPixelsCam;
class MPedestalCam;
class MCalibrationChargeCam;
class MCalibrationQECam;
class MExtractedSignalCam;

class MPedPhotCam;
class MCerPhotEvt;

class MCalibrateData : public MTask
{
private:
    MGeomCam              *fCam;
    MPedestalCam          *fPedestal;     // Pedestals/slice [ADC counts]
    MPedestalCam          *fPedestalFromData;  // Pedestals/slice [ADC counts]
    MBadPixelsCam         *fBadPixels;    // Bad Pixels information
    MCalibrationChargeCam *fCalibrations; // Calibration constants
    MCalibrationQECam     *fQEs;          // Quantum efficiencies
    MExtractedSignalCam   *fSignals;      // Integrated charge in FADCs counts

    MPedPhotCam           *fPedPhot;      // Pedestals/(used slices) [photons]
    MPedPhotCam           *fPedPhotFromData;// Pedestals/(used slices) [photons]
    MCerPhotEvt           *fCerPhotEvt;   // Cerenkov Photon Event used for calculation

    Float_t fNumUsedHiGainFADCSlices;
    Float_t fNumUsedLoGainFADCSlices;

    Float_t  fConversionHiLo;
    UShort_t fCalibrationMode;

    Int_t        fFlags;            // Flags defining to calibrate the pedestal each event or each run
 
    TString fNamePedADCRunContainer;           // name of the 'MPedestalCam' container
    TString fNamePedADCEventContainer;         // name of the 'MPedestalCam' container
    TString fNamePedPhotRunContainer;          // name of the 'MPedestalCam' container
    TString fNamePedPhotEventContainer;        // name of the 'MPedestalCam' container


    Bool_t GetConversionFactor(UInt_t, 
                               Float_t &, Float_t &, Float_t &, Float_t &);    

    Int_t  PreProcess(MParList *pList);
    Bool_t ReInit(MParList *pList);
    Int_t  Process();

public:

    enum CalibrationMode_t{kNone=0,kBlindPixel,kFfactor,kPinDiode,kCombined};
    static const CalibrationMode_t kDefault = kBlindPixel;

    enum PedestalType_t{
      kRun = BIT(0),
      kEvent = BIT(1)
    };
   


    MCalibrateData(CalibrationMode_t calmode = kDefault,const char *name=NULL, const char *title=NULL);

    void SetConversionHiLo(Float_t conv) { fConversionHiLo = conv; };
    void SetCalibrationMode(CalibrationMode_t calmode=kDefault){ fCalibrationMode=calmode;};

    void SetPedestalType(PedestalType_t i) { fFlags  =  i; } // Int_t f
    void EnablePedestalType(PedestalType_t i)   { fFlags |=  i; } // Int_t f
    void DisablePedestalType(PedestalType_t i)  { fFlags &= ~i; } // Int_t f
    Bool_t TestFlag(PedestalType_t i) const { return fFlags&i; }


    void SetNamePedADCRunContainer(const char *name)    { fNamePedADCRunContainer = name; }
    void SetNamePedADCEventContainer(const char *name)    { fNamePedADCEventContainer = name; }

    void SetNamePedPhotRunContainer(const char *name)    { fNamePedPhotRunContainer = name; }
    void SetNamePedPhotEventContainer(const char *name)    { fNamePedPhotEventContainer = name; }
    
    
    ClassDef(MCalibrateData, 0)   // Task to calculate cerenkov photons using calibration constants
};
 

#endif









