/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// MEvtLoop                                                                //
//                                                                         //
// This class is the core of each event processing.                        //
// First you must set the parameter list to use. The parameter list        //
// must contain the task list (MTaskList) to use. The name of the task     //
// list can be specified if you call Eventloop. The standard name is       //
// "MTaskList". The name you specify must match the name of the MTaskList  //
// object.                                                                 //
//                                                                         //
// If you call Eventloop first all PreProcess functions - with the         //
// parameter list as an argument - of the tasks in the task list are       //
// executed. If one of them returns kFALSE then the execution is stopped.  //
// If the preprocessing was ok. The Process funtion of the tasks are       //
// as long as one function returns kSTOP. Only the tasks which are marked  //
// marked as "All" or with a string which matches the MInputStreamID of    //
// MTaskList are executed. If one tasks returns kCONTINUE the pending      //
// tasks in the list are skipped and the execution in continued with       //
// the first one in the list.                                              //
// Afterwards the PostProcess functions are executed.                      //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
#include "MEvtLoop.h"

#include <iostream.h>

#include <TStopwatch.h>

#include "MParList.h"
#include "MTaskList.h"

ClassImp(MEvtLoop)

MEvtLoop::MEvtLoop() : fParlist(NULL)
{
}

MEvtLoop::~MEvtLoop()
{
}

void MEvtLoop::SetParList(MParList *p)
{
    fParlist = p;
}

void MEvtLoop::Eventloop(Int_t maxcnt, const char *ftasks)
{
    // See class description above.

    //
    // check if the needed parameter list is set.
    //
    if (!fParlist)
    {
        cout << "MEvtLoop::Eventloop - Error: Parlist not initialized." << endl;
        return;
    }

    //
    //  check for the existance of the specified task list
    //  the default name is "MTaskList"
    //
    MTaskList *tasks = (MTaskList*)fParlist->FindObject(ftasks);
    if (!tasks)
    {
        cout << "MEvtLoop::Eventloop - Error: Cannot find tasklist '" << ftasks << "' in parameter list." << endl;
        return;
    }

    //
    //  execute the preprocess of all tasks
    //  connect the different tasks with the right containers in
    //  the parameter list
    //
    if (!tasks->PreProcess(fParlist))
    {
        cout << "Error detected while PreProcessing" << endl;
        return;
    }

    //
    //   loop over all events and process all tasks for
    //   each event
    //
    cout << endl << "Eventloop running (";

    if (maxcnt<0)
        cout << "all";
    else
        cout << dec << maxcnt;

    cout << " events)..." << flush;

    Int_t dummy = maxcnt<0 ? 0 : maxcnt;

    //
    // start a stopwatch
    //
    TStopwatch clock;
    clock.Start();

    //
    // This is the MAIN EVENTLOOP which processes the data
    // if maxcnt<0 the number of processed events is counted
    // else only maxcnt events are processed
    //
    if (maxcnt<0)
        while (tasks->Process() && ++dummy);
    else
        while (tasks->Process() && dummy--);

    //
    // stop stop-watch, print results
    //
    clock.Stop();

    cout << "Ready!" << endl << endl;

    clock.Print();

    cout << dec << endl
        << "Time: " << clock.CpuTime() << "s"
        << " for " << (maxcnt<0?dummy:maxcnt) << " Events"
        << " --> " << (maxcnt<0?dummy:maxcnt)/clock.CpuTime() << " Events/s"
        << endl << endl;

    //
    //  execute the post process of all tasks
    //
    tasks->PostProcess();
}

